/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.factory.tiles;

import javax.annotation.Nonnull;
import java.io.IOException;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.Container;
import net.minecraft.inventory.IContainerListener;
import net.minecraft.inventory.ISidedInventory;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.EnumFacing;

import net.minecraftforge.common.capabilities.Capability;
import net.minecraftforge.fluids.Fluid;
import net.minecraftforge.fluids.FluidStack;
import net.minecraftforge.fluids.capability.CapabilityFluidHandler;

import forestry.api.core.IErrorLogic;
import forestry.api.fuels.FermenterFuel;
import forestry.api.fuels.FuelManager;
import forestry.api.recipes.IFermenterRecipe;
import forestry.api.recipes.IVariableFermentable;
import forestry.core.config.Constants;
import forestry.core.errors.EnumErrorCode;
import forestry.core.fluids.FluidHelper;
import forestry.core.fluids.TankManager;
import forestry.core.fluids.tanks.FilteredTank;
import forestry.core.network.DataInputStreamForestry;
import forestry.core.network.DataOutputStreamForestry;
import forestry.core.render.TankRenderInfo;
import forestry.core.tiles.ILiquidTankTile;
import forestry.core.tiles.TilePowered;
import forestry.factory.gui.ContainerFermenter;
import forestry.factory.gui.GuiFermenter;
import forestry.factory.inventory.InventoryFermenter;
import forestry.factory.recipes.FermenterRecipeManager;

public class TileFermenter extends TilePowered implements ISidedInventory, ILiquidTankTile {
	private final FilteredTank resourceTank;
	private final FilteredTank productTank;
	private final TankManager tankManager;

	private IFermenterRecipe currentRecipe;
	private float currentResourceModifier;
	private int fermentationTime = 0;
	private int fermentationTotalTime = 0;
	private int fuelBurnTime = 0;
	private int fuelTotalTime = 0;
	private int fuelCurrentFerment = 0;

	public TileFermenter() {
		super("fermenter", 2000, 8000);
		setEnergyPerWorkCycle(4200);
		setInternalInventory(new InventoryFermenter(this));

		resourceTank = new FilteredTank(Constants.PROCESSOR_TANK_CAPACITY, true, false);
		resourceTank.setFilters(FermenterRecipeManager.recipeFluidInputs);

		productTank = new FilteredTank(Constants.PROCESSOR_TANK_CAPACITY, false, true);
		productTank.setFilters(FermenterRecipeManager.recipeFluidOutputs);

		tankManager = new TankManager(this, resourceTank, productTank);
	}

	@Nonnull
	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound nbttagcompound) {
		nbttagcompound = super.func_189515_b(nbttagcompound);

		nbttagcompound.func_74768_a("FermentationTime", fermentationTime);
		nbttagcompound.func_74768_a("FermentationTotalTime", fermentationTotalTime);
		nbttagcompound.func_74768_a("FuelBurnTime", fuelBurnTime);
		nbttagcompound.func_74768_a("FuelTotalTime", fuelTotalTime);
		nbttagcompound.func_74768_a("FuelCurrentFerment", fuelCurrentFerment);

		tankManager.writeToNBT(nbttagcompound);
		return nbttagcompound;
	}

	@Override
	public void func_145839_a(NBTTagCompound nbttagcompound) {
		super.func_145839_a(nbttagcompound);

		fermentationTime = nbttagcompound.func_74762_e("FermentationTime");
		fermentationTotalTime = nbttagcompound.func_74762_e("FermentationTotalTime");
		fuelBurnTime = nbttagcompound.func_74762_e("FuelBurnTime");
		fuelTotalTime = nbttagcompound.func_74762_e("FuelTotalTime");
		fuelCurrentFerment = nbttagcompound.func_74762_e("FuelCurrentFerment");

		tankManager.readFromNBT(nbttagcompound);
	}

	@Override
	public void writeData(DataOutputStreamForestry data) throws IOException {
		super.writeData(data);
		tankManager.writeData(data);
	}

	@Override
	public void readData(DataInputStreamForestry data) throws IOException {
		super.readData(data);
		tankManager.readData(data);
	}

	@Override
	public void updateServerSide() {
		super.updateServerSide();

		if (updateOnInterval(20)) {
			FluidHelper.drainContainers(tankManager, this, InventoryFermenter.SLOT_INPUT);

			FluidStack fluidStack = productTank.getFluid();
			if (fluidStack != null) {
				FluidHelper.fillContainers(tankManager, this, InventoryFermenter.SLOT_CAN_INPUT, InventoryFermenter.SLOT_CAN_OUTPUT, fluidStack.getFluid(), true);
			}
		}
	}

	@Override
	public boolean workCycle() {
		int fermented = Math.min(fermentationTime, fuelCurrentFerment);
		int productAmount = Math.round(fermented * currentRecipe.getModifier() * currentResourceModifier);
		productTank.fillInternal(new FluidStack(currentRecipe.getOutput(), productAmount), true);

		fuelBurnTime--;
		resourceTank.drain(fermented, true);
		fermentationTime -= fermented;

		// Not done yet
		if (fermentationTime > 0) {
			return false;
		}

		currentRecipe = null;
		return true;
	}

	private void checkRecipe() {
		if (currentRecipe != null) {
			return;
		}

		ItemStack resource = func_70301_a(InventoryFermenter.SLOT_RESOURCE);
		FluidStack fluid = resourceTank.getFluid();

		currentRecipe = FermenterRecipeManager.findMatchingRecipe(resource, fluid);

		fermentationTotalTime = fermentationTime = currentRecipe == null ? 0 : currentRecipe.getFermentationValue();

		if (currentRecipe != null) {
			currentResourceModifier = determineResourceMod(resource);
			func_70298_a(InventoryFermenter.SLOT_RESOURCE, 1);
		}
	}

	private void checkFuel() {
		if (fuelBurnTime > 0) {
			return;
		}

		ItemStack fuel = func_70301_a(InventoryFermenter.SLOT_FUEL);
		if (fuel == null) {
			return;
		}

		FermenterFuel fermenterFuel = FuelManager.fermenterFuel.get(fuel);
		if (fermenterFuel == null) {
			return;
		}

		fuelBurnTime = fuelTotalTime = fermenterFuel.burnDuration;
		fuelCurrentFerment = fermenterFuel.fermentPerCycle;

		func_70298_a(InventoryFermenter.SLOT_FUEL, 1);
	}

	private static float determineResourceMod(ItemStack itemstack) {
		if (!(itemstack.func_77973_b() instanceof IVariableFermentable)) {
			return 1.0f;
		}

		return ((IVariableFermentable) itemstack.func_77973_b()).getFermentationModifier(itemstack);
	}


	@Override
	public boolean hasResourcesMin(float percentage) {
		ItemStack fermentationStack = func_70301_a(InventoryFermenter.SLOT_RESOURCE);
		if (fermentationStack == null) {
			return false;
		}

		return (float) fermentationStack.field_77994_a / (float) fermentationStack.func_77976_d() > percentage;
	}

	@Override
	public boolean hasFuelMin(float percentage) {
		ItemStack fuelStack = func_70301_a(InventoryFermenter.SLOT_FUEL);
		if (fuelStack == null) {
			return false;
		}

		return (float) fuelStack.field_77994_a / (float) fuelStack.func_77976_d() > percentage;
	}

	@Override
	public boolean hasWork() {
		checkRecipe();
		checkFuel();

		int fermented = Math.min(fermentationTime, fuelCurrentFerment);

		boolean hasRecipe = currentRecipe != null;
		boolean hasFuel = fuelBurnTime > 0;
		boolean hasResource = fermentationTime > 0 || func_70301_a(InventoryFermenter.SLOT_RESOURCE) != null;
		FluidStack drained = resourceTank.drain(fermented, false);
		boolean hasFluidResource = drained != null && drained.amount == fermented;
		boolean hasFluidSpace = true;

		if (hasRecipe) {
			int productAmount = Math.round(fermented * currentRecipe.getModifier() * currentResourceModifier);
			Fluid output = currentRecipe.getOutput();
			FluidStack fluidStack = new FluidStack(output, productAmount);
			hasFluidSpace = productTank.fillInternal(fluidStack, false) == fluidStack.amount;
		}

		IErrorLogic errorLogic = getErrorLogic();
		errorLogic.setCondition(!hasRecipe, EnumErrorCode.NO_RECIPE);
		errorLogic.setCondition(!hasFuel, EnumErrorCode.NO_FUEL);
		errorLogic.setCondition(!hasResource, EnumErrorCode.NO_RESOURCE);
		errorLogic.setCondition(!hasFluidResource, EnumErrorCode.NO_RESOURCE_LIQUID);
		errorLogic.setCondition(!hasFluidSpace, EnumErrorCode.NO_SPACE_TANK);

		return hasRecipe && hasFuel && hasResource && hasFluidResource && hasFluidSpace;
	}

	public int getBurnTimeRemainingScaled(int i) {
		if (fuelTotalTime == 0) {
			return 0;
		}

		return fuelBurnTime * i / fuelTotalTime;
	}

	public int getFermentationProgressScaled(int i) {
		if (fermentationTotalTime == 0) {
			return 0;
		}

		return fermentationTime * i / fermentationTotalTime;
	}

	@Override
	public TankRenderInfo getResourceTankInfo() {
		return new TankRenderInfo(resourceTank);
	}

	@Override
	public TankRenderInfo getProductTankInfo() {
		return new TankRenderInfo(productTank);
	}

	/* SMP GUI */
	public void getGUINetworkData(int i, int j) {
		switch (i) {
			case 0:
				fuelBurnTime = j;
				break;
			case 1:
				fuelTotalTime = j;
				break;
			case 2:
				fermentationTime = j;
				break;
			case 3:
				fermentationTotalTime = j;
				break;
		}
	}

	public void sendGUINetworkData(Container container, IContainerListener iCrafting) {
		iCrafting.func_71112_a(container, 0, fuelBurnTime);
		iCrafting.func_71112_a(container, 1, fuelTotalTime);
		iCrafting.func_71112_a(container, 2, fermentationTime);
		iCrafting.func_71112_a(container, 3, fermentationTotalTime);
	}

	@Nonnull
	@Override
	public TankManager getTankManager() {
		return tankManager;
	}

	/* ITRIGGERPROVIDER */
	// TODO: BuildCraft for 1.9
//	@Optional.Method(modid = "BuildCraftAPI|statements")
//	@Override
//	public Collection<ITriggerExternal> getExternalTriggers(EnumFacing side, TileEntity tile) {
//		LinkedList<ITriggerExternal> res = new LinkedList<>();
//		res.add(FactoryTriggers.lowResource25);
//		res.add(FactoryTriggers.lowResource10);
//		return res;
//	}

	@Override
	public Object getGui(EntityPlayer player, int data) {
		return new GuiFermenter(player.field_71071_by, this);
	}

	@Override
	public Object getContainer(EntityPlayer player, int data) {
		return new ContainerFermenter(player.field_71071_by, this);
	}

	@Override
	public boolean hasCapability(Capability<?> capability, EnumFacing facing) {
		if (super.hasCapability(capability, facing)) {
			return true;
		}
		return capability == CapabilityFluidHandler.FLUID_HANDLER_CAPABILITY;
	}

	@Override
	public <T> T getCapability(Capability<T> capability, EnumFacing facing) {
		if (super.hasCapability(capability, facing)) {
			return super.getCapability(capability, facing);
		}
		if (capability == CapabilityFluidHandler.FLUID_HANDLER_CAPABILITY) {
			return CapabilityFluidHandler.FLUID_HANDLER_CAPABILITY.cast(tankManager);
		}
		return null;
	}
}
