/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.core.utils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.List;

import net.minecraft.block.Block;
import net.minecraft.block.BlockHorizontal;
import net.minecraft.block.BlockOldLog;
import net.minecraft.block.BlockPlanks;
import net.minecraft.block.BlockStaticLiquid;
import net.minecraft.block.state.IBlockState;
import net.minecraft.init.Blocks;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.RayTraceResult;
import net.minecraft.util.math.Vec3d;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;

import net.minecraftforge.oredict.OreDictionary;

import forestry.core.network.packets.PacketFXSignal;
import forestry.core.proxy.Proxies;
import forestry.core.tiles.TileEngine;

import cofh.api.energy.IEnergyConnection;
import cofh.api.energy.IEnergyReceiver;

public abstract class BlockUtil {

	private static final int slabWoodId = OreDictionary.getOreID("slabWood");

	public static List<ItemStack> getBlockDrops(World world, BlockPos posBlock) {
		IBlockState blockState = world.func_180495_p(posBlock);

		return blockState.func_177230_c().getDrops(world, posBlock, blockState, 0);

	}
	
	public static boolean isEnergyReceiverOrEngine(EnumFacing side, TileEntity tile) {
		if (!(tile instanceof IEnergyReceiver) && !(tile instanceof TileEngine)) {
			return false;
		}

		IEnergyConnection receptor = (IEnergyConnection) tile;
		return receptor.canConnectEnergy(side);
	}

	public static boolean tryPlantCocoaPod(World world, BlockPos pos) {

		EnumFacing facing = getValidPodFacing(world, pos);
		if (facing == null) {
			return false;
		}

		IBlockState state = Blocks.field_150375_by.func_176223_P().func_177226_a(BlockHorizontal.field_185512_D, facing);
		world.func_175656_a(pos, state);
		return true;
	}

	@Nullable
	public static EnumFacing getValidPodFacing(World world, BlockPos pos) {
		for (EnumFacing facing : EnumFacing.field_176754_o) {
			if (isValidPodLocation(world, pos, facing)) {
				return facing;
			}
		}
		return null;
	}
	
	public static boolean isValidPodLocation(World world, BlockPos pos, EnumFacing direction) {
		pos = pos.func_177972_a(direction);
		if (!world.func_175667_e(pos)) {
			return false;
		}
		IBlockState state = world.func_180495_p(pos);
		Block block = state.func_177230_c();
		if (block == Blocks.field_150364_r) {
			return state.func_177229_b(BlockOldLog.field_176301_b) == BlockPlanks.EnumType.JUNGLE;
		} else {
			return block.isWood(world, pos);
		}
	}

	public static boolean isWoodSlabBlock(IBlockState blockState, Block block, IBlockAccess world, BlockPos pos) {
		if (blockState == null || block == null || block.isAir(blockState, world, pos)) {
			return false;
		}
		if(Item.func_150898_a(block) == null){
			return false;
		}
		int[] oreIds = OreDictionary.getOreIDs(new ItemStack(block));
		for (int oreId : oreIds) {
			if (oreId == slabWoodId) {
				return true;
			}
		}

		return false;
	}

	public static boolean isReplaceableBlock(IBlockState blockState, World world, BlockPos pos) {
		Block block = blockState.func_177230_c();
		return block.func_176200_f(world, pos) && !(block instanceof BlockStaticLiquid);
	}

	public static RayTraceResult collisionRayTrace(@Nonnull BlockPos pos, @Nonnull Vec3d startVec, @Nonnull Vec3d endVec, @Nonnull AxisAlignedBB bounds) {
		return collisionRayTrace(pos, startVec, endVec, bounds.field_72340_a, bounds.field_72338_b, bounds.field_72339_c, bounds.field_72336_d, bounds.field_72337_e, bounds.field_72334_f);
	}

	/**
	 * Ray traces through the blocks collision from start vector to end vector returning a ray trace hit.
	 */
	public static RayTraceResult collisionRayTrace(@Nonnull BlockPos pos, @Nonnull Vec3d startVec, @Nonnull Vec3d endVec, double minX, double minY, double minZ, double maxX, double maxY, double maxZ) {
		startVec = startVec.func_72441_c(-pos.func_177958_n(), -pos.func_177956_o(), -pos.func_177952_p());
		endVec = endVec.func_72441_c(-pos.func_177958_n(), -pos.func_177956_o(), -pos.func_177952_p());
		Vec3d vec32 = startVec.func_72429_b(endVec, minX);
		Vec3d vec33 = startVec.func_72429_b(endVec, maxX);
		Vec3d vec34 = startVec.func_72435_c(endVec, minY);
		Vec3d vec35 = startVec.func_72435_c(endVec, maxY);
		Vec3d vec36 = startVec.func_72434_d(endVec, minZ);
		Vec3d vec37 = startVec.func_72434_d(endVec, maxZ);

		if (!isVecInsideYZBounds(vec32, minY, minZ, maxY, maxZ)) {
			vec32 = null;
		}

		if (!isVecInsideYZBounds(vec33, minY, minZ, maxY, maxZ)) {
			vec33 = null;
		}

		if (!isVecInsideXZBounds(vec34, minX, minZ, maxX, maxZ)) {
			vec34 = null;
		}

		if (!isVecInsideXZBounds(vec35, minX, minZ, maxX, maxZ)) {
			vec35 = null;
		}

		if (!isVecInsideXYBounds(vec36, minX, minY, maxX, maxY)) {
			vec36 = null;
		}

		if (!isVecInsideXYBounds(vec37, minX, minY, maxX, maxY)) {
			vec37 = null;
		}

		Vec3d minHit = null;

		if (vec32 != null && (minHit == null || startVec.func_72436_e(vec32) < startVec.func_72436_e(minHit))) {
			minHit = vec32;
		}

		if (vec33 != null && (minHit == null || startVec.func_72436_e(vec33) < startVec.func_72436_e(minHit))) {
			minHit = vec33;
		}

		if (vec34 != null && (minHit == null || startVec.func_72436_e(vec34) < startVec.func_72436_e(minHit))) {
			minHit = vec34;
		}

		if (vec35 != null && (minHit == null || startVec.func_72436_e(vec35) < startVec.func_72436_e(minHit))) {
			minHit = vec35;
		}

		if (vec36 != null && (minHit == null || startVec.func_72436_e(vec36) < startVec.func_72436_e(minHit))) {
			minHit = vec36;
		}

		if (vec37 != null && (minHit == null || startVec.func_72436_e(vec37) < startVec.func_72436_e(minHit))) {
			minHit = vec37;
		}

		if (minHit == null) {
			return null;
		} else {
			byte sideHit = -1;

			if (minHit == vec32) {
				sideHit = 4;
			}

			if (minHit == vec33) {
				sideHit = 5;
			}

			if (minHit == vec34) {
				sideHit = 0;
			}

			if (minHit == vec35) {
				sideHit = 1;
			}

			if (minHit == vec36) {
				sideHit = 2;
			}

			if (minHit == vec37) {
				sideHit = 3;
			}

			return new RayTraceResult(minHit.func_72441_c(pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p()), EnumFacing.values()[sideHit], pos);
		}
	}

	/**
	 * Checks if a vector is within the Y and Z bounds of the block.
	 */
	private static boolean isVecInsideYZBounds(Vec3d vec, double minY, double minZ, double maxY, double maxZ) {
		return vec != null && vec.field_72448_b >= minY && vec.field_72448_b <= maxY && vec.field_72449_c >= minZ && vec.field_72449_c <= maxZ;
	}

	/**
	 * Checks if a vector is within the X and Z bounds of the block.
	 */
	private static boolean isVecInsideXZBounds(Vec3d vec, double minX, double minZ, double maxX, double maxZ) {
		return vec != null && vec.field_72450_a >= minX && vec.field_72450_a <= maxX && vec.field_72449_c >= minZ && vec.field_72449_c <= maxZ;
	}

	/**
	 * Checks if a vector is within the X and Y bounds of the block.
	 */
	private static boolean isVecInsideXYBounds(Vec3d vec, double minX, double minY, double maxX, double maxY) {
		return vec != null && vec.field_72450_a >= minX && vec.field_72450_a <= maxX && vec.field_72448_b >= minY && vec.field_72448_b <= maxY;
	}
	
	/* CHUNKS */

	public static boolean canReplace(IBlockState blockState, World world, BlockPos pos) {
		Block block = blockState.func_177230_c();
		return block.func_176200_f(world, pos) && !block.func_149688_o(blockState).func_76224_d();
	}
	
	public static boolean canPlaceTree(IBlockState blockState, World world, BlockPos pos){
		BlockPos downPos = pos.func_177977_b();
		Block block = world.func_180495_p(downPos).func_177230_c();
		if(block.func_176200_f(world, downPos) && block.func_149688_o(blockState).func_76224_d() || block.isLeaves(blockState, world, downPos) || block.isWood(world, downPos)){
			return false;
		}
		return true;
	}
	
	public static BlockPos getNextReplaceableUpPos(World world, BlockPos pos){
		IBlockState blockState;

		do {
			pos = pos.func_177984_a();
			blockState = world.func_180495_p(pos);
		} while (!BlockUtil.canReplace(blockState, world, pos));

		return pos;
	}
	
	public static BlockPos getNextSolidDownPos(World world, BlockPos pos){
		IBlockState blockState;

		do {
			pos = pos.func_177977_b();
			blockState = world.func_180495_p(pos);
		} while (BlockUtil.canReplace(blockState, world, pos));

		return pos;
	}

	/** Copied from {@link Block#shouldSideBeRendered} */
	public static boolean shouldSideBeRendered(IBlockState blockState, IBlockAccess blockAccess, BlockPos pos, EnumFacing side) {
		AxisAlignedBB axisalignedbb = blockState.func_185900_c(blockAccess, pos);

		switch (side) {
			case DOWN:
				if (axisalignedbb.field_72338_b > 0.0D) {
					return true;
				}
				break;
			case UP:
				if (axisalignedbb.field_72337_e < 1.0D) {
					return true;
				}
				break;
			case NORTH:
				if (axisalignedbb.field_72339_c > 0.0D) {
					return true;
				}
				break;
			case SOUTH:
				if (axisalignedbb.field_72334_f < 1.0D) {
					return true;
				}
				break;
			case WEST:
				if (axisalignedbb.field_72340_a > 0.0D) {
					return true;
				}
				break;
			case EAST:
				if (axisalignedbb.field_72336_d < 1.0D) {
					return true;
				}
		}

		return !blockAccess.func_180495_p(pos.func_177972_a(side)).doesSideBlockRendering(blockAccess, pos.func_177972_a(side), side.func_176734_d());
	}

	public static boolean setBlockWithPlaceSound(@Nonnull World world, @Nonnull BlockPos pos, @Nonnull IBlockState blockState) {
		if (world.func_175656_a(pos, blockState)) {
			PacketFXSignal packet = new PacketFXSignal(PacketFXSignal.SoundFXType.BLOCK_PLACE, pos, blockState);
			Proxies.net.sendNetworkPacket(packet, world);
			return true;
		}
		return false;
	}

	public static boolean setBlockWithBreakSound(@Nonnull World world, @Nonnull BlockPos pos, @Nonnull IBlockState blockState, @Nonnull IBlockState oldState) {
		if (world.func_175656_a(pos, blockState)) {
			PacketFXSignal packet = new PacketFXSignal(PacketFXSignal.VisualFXType.BLOCK_BREAK, PacketFXSignal.SoundFXType.BLOCK_BREAK, pos, oldState);
			Proxies.net.sendNetworkPacket(packet, world);
			return true;
		}
		return false;
	}

	public static boolean setBlockToAirWithSound(@Nonnull World world, @Nonnull BlockPos pos, @Nonnull IBlockState oldState) {
		if (world.func_175698_g(pos)) {
			PacketFXSignal packet = new PacketFXSignal(PacketFXSignal.VisualFXType.BLOCK_BREAK, PacketFXSignal.SoundFXType.BLOCK_BREAK, pos, oldState);
			Proxies.net.sendNetworkPacket(packet, world);
			return true;
		}
		return false;
	}
}
