/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.energy.tiles;

import javax.annotation.Nonnull;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.Container;
import net.minecraft.inventory.IContainerListener;
import net.minecraft.inventory.IInventory;
import net.minecraft.inventory.ISidedInventory;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;

import net.minecraftforge.items.IItemHandler;
import net.minecraftforge.items.wrapper.InvWrapper;

import forestry.api.fuels.FuelManager;
import forestry.core.PluginCore;
import forestry.core.config.Constants;
import forestry.core.errors.EnumErrorCode;
import forestry.core.inventory.AdjacentInventoryCache;
import forestry.core.inventory.IInventoryAdapter;
import forestry.core.inventory.wrappers.InventoryMapper;
import forestry.core.tiles.TemperatureState;
import forestry.core.tiles.TileEngine;
import forestry.core.utils.InventoryUtil;
import forestry.core.utils.ItemStackUtil;
import forestry.energy.gui.ContainerEnginePeat;
import forestry.energy.gui.GuiEnginePeat;
import forestry.energy.inventory.InventoryEnginePeat;

public class TileEnginePeat extends TileEngine implements ISidedInventory {
	private Item fuelItem;
	private int fuelItemMeta;
	private int burnTime;
	private int totalBurnTime;
	private int ashProduction;
	private final int ashForItem;
	private final AdjacentInventoryCache inventoryCache = new AdjacentInventoryCache(this, getTileCache());

	public TileEnginePeat() {
		super("engine.copper", Constants.ENGINE_COPPER_HEAT_MAX, 200000);

		ashForItem = Constants.ENGINE_COPPER_ASH_FOR_ITEM;
		setInternalInventory(new InventoryEnginePeat(this));
	}

	private int getFuelSlot() {
		IInventoryAdapter inventory = getInternalInventory();
		if (inventory.func_70301_a(InventoryEnginePeat.SLOT_FUEL) == null) {
			return -1;
		}

		if (determineFuelValue(inventory.func_70301_a(InventoryEnginePeat.SLOT_FUEL)) > 0) {
			return InventoryEnginePeat.SLOT_FUEL;
		}

		return -1;
	}

	private int getFreeWasteSlot() {
		IInventoryAdapter inventory = getInternalInventory();
		for (int i = InventoryEnginePeat.SLOT_WASTE_1; i <= InventoryEnginePeat.SLOT_WASTE_COUNT; i++) {
			ItemStack waste = inventory.func_70301_a(i);
			if (waste == null) {
				return i;
			}

			if (waste.func_77973_b() != PluginCore.items.ash) {
				continue;
			}

			if (waste.field_77994_a < 64) {
				return i;
			}
		}

		return -1;
	}

	@Override
	public void updateServerSide() {
		super.updateServerSide();

		if (!updateOnInterval(40)) {
			return;
		}

		dumpStash();

		int fuelSlot = getFuelSlot();
		boolean hasFuel = fuelSlot >= 0 && determineBurnDuration(getInternalInventory().func_70301_a(fuelSlot)) > 0;
		getErrorLogic().setCondition(!hasFuel, EnumErrorCode.NO_FUEL);
	}

	@Override
	public void burn() {

		currentOutput = 0;

		if (burnTime > 0) {
			burnTime--;
			addAsh(1);

			if (isRedstoneActivated()) {
				currentOutput = determineFuelValue(new ItemStack(fuelItem, 1, fuelItemMeta));
				energyManager.generateEnergy(currentOutput);
			}
		} else if (isRedstoneActivated()) {
			int fuelSlot = getFuelSlot();
			int wasteSlot = getFreeWasteSlot();

			if (fuelSlot >= 0 && wasteSlot >= 0) {
				IInventoryAdapter inventory = getInternalInventory();
				burnTime = totalBurnTime = determineBurnDuration(inventory.func_70301_a(fuelSlot));
				if (burnTime > 0) {
					fuelItem = inventory.func_70301_a(fuelSlot).func_77973_b();
					func_70298_a(fuelSlot, 1);
				}
			}
		}
	}

	@Override
	public int dissipateHeat() {
		if (heat <= 0) {
			return 0;
		}

		int loss = 0;

		if (!isBurning()) {
			loss += 1;
		}

		TemperatureState tempState = getTemperatureState();
		if (tempState == TemperatureState.OVERHEATING || tempState == TemperatureState.OPERATING_TEMPERATURE) {
			loss += 1;
		}

		heat -= loss;
		return loss;
	}

	@Override
	public int generateHeat() {

		int heatToAdd = 0;

		if (isBurning()) {
			heatToAdd++;
			if ((double) energyManager.getTotalEnergyStored() / (double) energyManager.getMaxEnergyStored() > 0.5) {
				heatToAdd++;
			}
		}

		addHeat(heatToAdd);
		return heatToAdd;
	}

	private void addAsh(int amount) {

		ashProduction += amount;
		if (ashProduction < ashForItem) {
			return;
		}

		// If we have reached the necessary amount, we need to add ash
		int wasteSlot = getFreeWasteSlot();
		if (wasteSlot >= 0) {
			IInventoryAdapter inventory = getInternalInventory();
			if (inventory.func_70301_a(wasteSlot) == null) {
				inventory.func_70299_a(wasteSlot, PluginCore.items.ash.getItemStack());
			} else {
				inventory.func_70301_a(wasteSlot).field_77994_a++;
			}
		}
		// Reset
		ashProduction = 0;
		// try to dump stash
		dumpStash();
	}

	/**
	 * Returns the fuel value (power per cycle) an item of the passed ItemStack provides
	 */
	private static int determineFuelValue(ItemStack fuel) {
		if (FuelManager.copperEngineFuel.containsKey(fuel)) {
			return FuelManager.copperEngineFuel.get(fuel).powerPerCycle;
		} else {
			return 0;
		}
	}

	/**
	 * Returns the fuel value (power per cycle) an item of the passed ItemStack provides
	 */
	private static int determineBurnDuration(ItemStack fuel) {
		if (FuelManager.copperEngineFuel.containsKey(fuel)) {
			return FuelManager.copperEngineFuel.get(fuel).burnDuration;
		} else {
			return 0;
		}
	}

	/* AUTO-EJECTING */
	private IInventory getWasteInventory() {
		IInventoryAdapter inventory = getInternalInventory();
		if (inventory == null) {
			return null;
		}

		return new InventoryMapper(inventory, InventoryEnginePeat.SLOT_WASTE_1, InventoryEnginePeat.SLOT_WASTE_COUNT);
	}

	private void dumpStash() {
		IInventory wasteInventory = getWasteInventory();
		if (wasteInventory == null) {
			return;
		}

		IItemHandler wasteItemHandler = new InvWrapper(wasteInventory);

		if (!InventoryUtil.moveOneItemToPipe(wasteItemHandler, getTileCache())) {
			InventoryUtil.moveItemStack(wasteItemHandler, inventoryCache.getAdjacentInventories());
		}
	}

	// / STATE INFORMATION
	@Override
	public boolean isBurning() {
		return mayBurn() && burnTime > 0;
	}

	@Override
	public int getBurnTimeRemainingScaled(int i) {
		if (totalBurnTime == 0) {
			return 0;
		}

		return burnTime * i / totalBurnTime;
	}

	@Override
	public boolean hasFuelMin(float percentage) {
		int fuelSlot = this.getFuelSlot();
		if (fuelSlot < 0) {
			return false;
		}

		IInventoryAdapter inventory = getInternalInventory();
		return (float) inventory.func_70301_a(fuelSlot).field_77994_a / (float) inventory.func_70301_a(fuelSlot).func_77976_d() > percentage;
	}

	// / LOADING AND SAVING
	@Override
	public void func_145839_a(NBTTagCompound nbttagcompound) {
		super.func_145839_a(nbttagcompound);

		String fuelItemName = nbttagcompound.func_74779_i("EngineFuelItem");

		if (!fuelItemName.isEmpty()) {
			fuelItem = ItemStackUtil.getItemFromRegistry(fuelItemName);
		}

		fuelItemMeta = nbttagcompound.func_74762_e("EngineFuelMeta");
		burnTime = nbttagcompound.func_74762_e("EngineBurnTime");
		totalBurnTime = nbttagcompound.func_74762_e("EngineTotalTime");
		if (nbttagcompound.func_74764_b("AshProduction")) {
			ashProduction = nbttagcompound.func_74762_e("AshProduction");
		}
	}

	@Nonnull
	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound nbttagcompound) {
		nbttagcompound = super.func_189515_b(nbttagcompound);

		if (fuelItem != null) {
			nbttagcompound.func_74778_a("EngineFuelItem", ItemStackUtil.getItemNameFromRegistryAsString(fuelItem));
		}

		nbttagcompound.func_74768_a("EngineFuelMeta", fuelItemMeta);
		nbttagcompound.func_74768_a("EngineBurnTime", burnTime);
		nbttagcompound.func_74768_a("EngineTotalTime", totalBurnTime);
		nbttagcompound.func_74768_a("AshProduction", ashProduction);
		return nbttagcompound;
	}

	// / SMP GUI
	@Override
	public void getGUINetworkData(int i, int j) {

		switch (i) {
			case 0:
				burnTime = j;
				break;
			case 1:
				totalBurnTime = j;
				break;
			case 2:
				currentOutput = j;
				break;
			case 3:
				energyManager.fromGuiInt(j);
				break;
			case 4:
				heat = j;
				break;
		}
	}

	@Override
	public void sendGUINetworkData(Container containerEngine, IContainerListener iCrafting) {
		iCrafting.func_71112_a(containerEngine, 0, burnTime);
		iCrafting.func_71112_a(containerEngine, 1, totalBurnTime);
		iCrafting.func_71112_a(containerEngine, 2, currentOutput);
		iCrafting.func_71112_a(containerEngine, 3, energyManager.toGuiInt());
		iCrafting.func_71112_a(containerEngine, 4, heat);
	}

	/* ITriggerProvider */
	// TODO: buildcraft for 1.9
//	@Optional.Method(modid = "BuildCraftAPI|statements")
//	@Override
//	public Collection<ITriggerExternal> getExternalTriggers(EnumFacing side, TileEntity tile) {
//		LinkedList<ITriggerExternal> res = new LinkedList<>();
//		res.add(FactoryTriggers.lowFuel25);
//		res.add(FactoryTriggers.lowFuel10);
//		return res;
//	}

	@Override
	public Object getGui(EntityPlayer player, int data) {
		return new GuiEnginePeat(player.field_71071_by, this);
	}

	@Override
	public Object getContainer(EntityPlayer player, int data) {
		return new ContainerEnginePeat(player.field_71071_by, this);
	}
}
