/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.core.tiles;

import javax.annotation.Nonnull;
import java.io.IOException;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.IInventory;
import net.minecraft.inventory.ISidedInventory;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.EnumFacing;

import net.minecraftforge.fluids.Fluid;
import net.minecraftforge.fluids.FluidStack;
import net.minecraftforge.fluids.FluidTankInfo;
import net.minecraftforge.fluids.IFluidHandler;

import forestry.api.arboriculture.TreeManager;
import forestry.api.core.ForestryAPI;
import forestry.api.genetics.AlleleManager;
import forestry.api.genetics.IIndividual;
import forestry.core.config.Constants;
import forestry.core.errors.EnumErrorCode;
import forestry.core.fluids.FluidHelper;
import forestry.core.fluids.Fluids;
import forestry.core.fluids.TankManager;
import forestry.core.fluids.tanks.FilteredTank;
import forestry.core.gui.ContainerAnalyzer;
import forestry.core.gui.GuiAnalyzer;
import forestry.core.inventory.InventoryAnalyzer;
import forestry.core.inventory.wrappers.InventoryMapper;
import forestry.core.network.DataInputStreamForestry;
import forestry.core.network.DataOutputStreamForestry;
import forestry.core.network.packets.PacketItemStackDisplay;
import forestry.core.proxy.Proxies;
import forestry.core.utils.GeneticsUtil;
import forestry.core.utils.InventoryUtil;
import forestry.plugins.ForestryPluginUids;

public class TileAnalyzer extends TilePowered implements ISidedInventory, ILiquidTankTile, IFluidHandler, IItemStackDisplay {
	private static final int TIME_TO_ANALYZE = 125;
	private static final int HONEY_REQUIRED = 100;
	private static final int ENERGY_PER_WORK_CYCLE = 20320;

	private final FilteredTank resourceTank;
	private final TankManager tankManager;
	private final IInventory invInput;
	private final IInventory invOutput;

	private IIndividual specimenToAnalyze;
	private ItemStack individualOnDisplayClient;

	/* CONSTRUCTOR */
	public TileAnalyzer() {
		super("analyzer", 800, Constants.MACHINE_MAX_ENERGY);
		setInternalInventory(new InventoryAnalyzer(this));
		resourceTank = new FilteredTank(Constants.PROCESSOR_TANK_CAPACITY, Fluids.FOR_HONEY.getFluid());
		tankManager = new TankManager(this, resourceTank);
		invInput = new InventoryMapper(getInternalInventory(), InventoryAnalyzer.SLOT_INPUT_1, InventoryAnalyzer.SLOT_INPUT_COUNT);
		invOutput = new InventoryMapper(getInternalInventory(), InventoryAnalyzer.SLOT_OUTPUT_1, InventoryAnalyzer.SLOT_OUTPUT_COUNT);
	}

	/* SAVING & LOADING */
	@Nonnull
	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound nbttagcompound) {
		nbttagcompound = super.func_189515_b(nbttagcompound);
		tankManager.writeToNBT(nbttagcompound);
		return nbttagcompound;
	}

	@Override
	public void func_145839_a(NBTTagCompound nbttagcompound) {
		super.func_145839_a(nbttagcompound);
		tankManager.readFromNBT(nbttagcompound);

		ItemStack stackToAnalyze = func_70301_a(InventoryAnalyzer.SLOT_ANALYZE);
		if (stackToAnalyze != null) {
			specimenToAnalyze = AlleleManager.alleleRegistry.getIndividual(stackToAnalyze);
		}
	}

	@Override
	protected void updateServerSide() {
		super.updateServerSide();

		if (updateOnInterval(20)) {
			// Check if we have suitable items waiting in the can slot
			FluidHelper.drainContainers(tankManager, this, InventoryAnalyzer.SLOT_CAN);
		}
	}

	/* WORKING */
	@Override
	public boolean workCycle() {
		ItemStack stackToAnalyze = func_70301_a(InventoryAnalyzer.SLOT_ANALYZE);
		if (stackToAnalyze == null) {
			return false;
		}

		if (!specimenToAnalyze.isAnalyzed()) {
			if (!resourceTank.canDrain(HONEY_REQUIRED)) {
				return false;
			}
			resourceTank.drain(HONEY_REQUIRED, true);

			specimenToAnalyze.analyze();

			NBTTagCompound nbttagcompound = new NBTTagCompound();
			specimenToAnalyze.writeToNBT(nbttagcompound);
			stackToAnalyze.func_77982_d(nbttagcompound);
		}

		boolean added = InventoryUtil.tryAddStack(invOutput, stackToAnalyze, true);
		if (!added) {
			return false;
		}

		func_70299_a(InventoryAnalyzer.SLOT_ANALYZE, null);
		PacketItemStackDisplay packet = new PacketItemStackDisplay(this, getIndividualOnDisplay());
		Proxies.net.sendNetworkPacket(packet, field_145850_b);

		return true;
	}

	private Integer getInputSlotIndex() {
		for (int slotIndex = 0; slotIndex < invInput.func_70302_i_(); slotIndex++) {
			ItemStack inputStack = invInput.func_70301_a(slotIndex);
			if (AlleleManager.alleleRegistry.isIndividual(inputStack)) {
				return slotIndex;
			}
		}
		return null;
	}

	/* Network */
	@Override
	public void writeData(DataOutputStreamForestry data) throws IOException {
		super.writeData(data);
		ItemStack displayStack = getIndividualOnDisplay();
		data.writeItemStack(displayStack);
		tankManager.writeData(data);
	}

	@Override
	public void readData(DataInputStreamForestry data) throws IOException {
		super.readData(data);
		individualOnDisplayClient = data.readItemStack();
		tankManager.readData(data);
	}

	@Override
	public void handleItemStackForDisplay(ItemStack itemStack) {
		if (!ItemStack.func_77989_b(itemStack, individualOnDisplayClient)) {
			individualOnDisplayClient = itemStack;
			field_145850_b.func_175704_b(func_174877_v(), func_174877_v());
		}
	}

	/* STATE INFORMATION */
	@Override
	public boolean hasWork() {
		moveSpecimenToAnalyzeSlot();

		ItemStack specimen = func_70301_a(InventoryAnalyzer.SLOT_ANALYZE);

		boolean hasSpecimen = specimen != null;
		boolean hasResource = true;
		boolean hasSpace = true;

		if (hasSpecimen) {
			hasSpace = InventoryUtil.tryAddStack(invOutput, specimen, true, false);

			if (!specimenToAnalyze.isAnalyzed()) {
				hasResource = resourceTank.canDrain(HONEY_REQUIRED);
			}
		}

		getErrorLogic().setCondition(!hasSpecimen, EnumErrorCode.NO_SPECIMEN);
		getErrorLogic().setCondition(!hasResource, EnumErrorCode.NO_RESOURCE_LIQUID);
		getErrorLogic().setCondition(!hasSpace, EnumErrorCode.NO_SPACE_INVENTORY);

		return hasSpecimen && hasResource && hasSpace;
	}

	private void moveSpecimenToAnalyzeSlot() {
		if (func_70301_a(InventoryAnalyzer.SLOT_ANALYZE) != null) {
			return;
		}

		Integer slotIndex = getInputSlotIndex();
		if (slotIndex == null) {
			return;
		}

		ItemStack inputStack = invInput.func_70301_a(slotIndex);
		if (inputStack == null) {
			return;
		}

		if (ForestryAPI.enabledPlugins.contains(ForestryPluginUids.ARBORICULTURE) && !TreeManager.treeRoot.isMember(inputStack)) {
			ItemStack ersatz = GeneticsUtil.convertToGeneticEquivalent(inputStack);
			if (ersatz != null) {
				inputStack = ersatz;
			}
		}

		specimenToAnalyze = AlleleManager.alleleRegistry.getIndividual(inputStack);
		if (specimenToAnalyze == null) {
			return;
		}

		func_70299_a(InventoryAnalyzer.SLOT_ANALYZE, inputStack);
		invInput.func_70299_a(slotIndex, null);

		if (specimenToAnalyze.isAnalyzed()) {
			setTicksPerWorkCycle(1);
			setEnergyPerWorkCycle(0);
		} else {
			setTicksPerWorkCycle(TIME_TO_ANALYZE);
			setEnergyPerWorkCycle(ENERGY_PER_WORK_CYCLE);
		}

		PacketItemStackDisplay packet = new PacketItemStackDisplay(this, getIndividualOnDisplay());
		Proxies.net.sendNetworkPacket(packet, field_145850_b);
	}

	public ItemStack getIndividualOnDisplay() {
		if (field_145850_b.field_72995_K) {
			return individualOnDisplayClient;
		}
		return func_70301_a(InventoryAnalyzer.SLOT_ANALYZE);
	}

	/* ILiquidTankTile */
	@Nonnull
	@Override
	public TankManager getTankManager() {
		return tankManager;
	}

	/* IFluidHandler */
	@Override
	public int fill(EnumFacing from, FluidStack resource, boolean doFill) {
		return tankManager.fill(from, resource, doFill);
	}

	@Override
	public FluidStack drain(EnumFacing from, FluidStack resource, boolean doDrain) {
		return tankManager.drain(from, resource, doDrain);
	}

	@Override
	public FluidStack drain(EnumFacing from, int maxDrain, boolean doDrain) {
		return tankManager.drain(from, maxDrain, doDrain);
	}

	@Override
	public boolean canFill(EnumFacing from, Fluid fluid) {
		return tankManager.canFill(from, fluid);
	}

	@Override
	public boolean canDrain(EnumFacing from, Fluid fluid) {
		return tankManager.canDrain(from, fluid);
	}

	@Override
	public FluidTankInfo[] getTankInfo(EnumFacing from) {
		return tankManager.getTankInfo(from);
	}

	@Override
	public Object getGui(EntityPlayer player, int data) {
		return new GuiAnalyzer(player.field_71071_by, this);
	}

	@Override
	public Object getContainer(EntityPlayer player, int data) {
		return new ContainerAnalyzer(player.field_71071_by, this);
	}
}
