/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.core.utils;

import com.google.common.collect.AbstractIterator;

import java.util.Iterator;
import java.util.Random;

import net.minecraft.util.EnumFacing;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.Vec3i;
import net.minecraft.world.World;

public final class VectUtil {
	public static BlockPos getRandomPositionInArea(Random random, Vec3i area) {
		int x = random.nextInt(area.func_177958_n());
		int y = random.nextInt(area.func_177956_o());
		int z = random.nextInt(area.func_177952_p());
		return new BlockPos(x, y, z);
	}

	public static BlockPos add(Vec3i... vects) {
		int x = 0;
		int y = 0;
		int z = 0;
		for (Vec3i vect : vects) {
			x += vect.func_177958_n();
			y += vect.func_177956_o();
			z += vect.func_177952_p();
		}
		return new BlockPos(x, y, z);
	}

	public static BlockPos scale(Vec3i vect, float factor) {
		return new BlockPos(vect.func_177958_n() * factor, vect.func_177956_o() * factor, vect.func_177952_p() * factor);
	}

	public static EnumFacing direction(Vec3i a, Vec3i b) {
		int x = Math.abs(a.func_177958_n() - b.func_177958_n());
		int y = Math.abs(a.func_177956_o() - b.func_177956_o());
		int z = Math.abs(a.func_177952_p() - b.func_177952_p());
		int max = Math.max(x, Math.max(y, z));
		if (max == x) {
			return EnumFacing.EAST;
		} else if (max == z) {
			return EnumFacing.SOUTH;
		} else {
			return EnumFacing.UP;
		}
	}

	public static Iterable<BlockPos.MutableBlockPos> getAllInBoxFromCenterMutable(World world, final BlockPos from, final BlockPos center, final BlockPos to) {
		final BlockPos minPos = new BlockPos(Math.min(from.func_177958_n(), to.func_177958_n()), Math.min(from.func_177956_o(), to.func_177956_o()), Math.min(from.func_177952_p(), to.func_177952_p()));
		final BlockPos maxPos = new BlockPos(Math.max(from.func_177958_n(), to.func_177958_n()), Math.max(from.func_177956_o(), to.func_177956_o()), Math.max(from.func_177952_p(), to.func_177952_p()));

		return new Iterable<BlockPos.MutableBlockPos>() {
			@Override
			public Iterator<BlockPos.MutableBlockPos> iterator() {
				return new MutableBlockPosSpiralIterator(world, center, maxPos, minPos);
			}
		};
	}

	private static class MutableBlockPosSpiralIterator extends AbstractIterator<BlockPos.MutableBlockPos> {
		private final World world;
		private final BlockPos center;
		private final BlockPos maxPos;
		private final BlockPos minPos;
		private int spiralLayer;
		private final int maxSpiralLayers;
		private int direction;

		private BlockPos.MutableBlockPos theBlockPos;

		public MutableBlockPosSpiralIterator(World world, BlockPos center, BlockPos maxPos, BlockPos minPos) {
			this.world = world;
			this.center = center;
			this.maxPos = maxPos;
			this.minPos = minPos;

			int xDiameter = maxPos.func_177958_n() - minPos.func_177958_n();
			int zDiameter = maxPos.func_177952_p() - minPos.func_177952_p();
			this.maxSpiralLayers = Math.max(xDiameter, zDiameter) / 2;
			this.spiralLayer = 1;
		}

		@Override
		protected BlockPos.MutableBlockPos computeNext() {
			BlockPos.MutableBlockPos pos;

			do {
				pos = nextPos();
			}
			while (pos != null && (pos.func_177958_n() > maxPos.func_177958_n() || pos.func_177952_p() > maxPos.func_177952_p() || pos.func_177958_n() < minPos.func_177958_n() || pos.func_177952_p() < minPos.func_177952_p()));

			return pos;
		}

		protected BlockPos.MutableBlockPos nextPos() {
			if (this.theBlockPos == null) {
				this.theBlockPos = new BlockPos.MutableBlockPos(center.func_177958_n(), maxPos.func_177956_o(), center.func_177952_p());
				int y = Math.min(this.maxPos.func_177956_o(), this.world.func_175645_m(this.theBlockPos).func_177956_o());
				this.theBlockPos.func_185336_p(y);
				return this.theBlockPos;
			} else if (spiralLayer > maxSpiralLayers) {
				return this.endOfData();
			} else {
				int x = this.theBlockPos.func_177958_n();
				int y = this.theBlockPos.func_177956_o();
				int z = this.theBlockPos.func_177952_p();

				if (y > minPos.func_177956_o() && y > 0) {
					y--;
				} else {
					switch (direction) {
						case 0:
							++x;
							if (x == center.func_177958_n() + spiralLayer) {
								++direction;
							}
							break;
						case 1:
							++z;
							if (z == center.func_177952_p() + spiralLayer) {
								++direction;
							}
							break;
						case 2:
							--x;
							if (x == center.func_177958_n() - spiralLayer) {
								++direction;
							}
							break;
						case 3:
							--z;
							if (z == center.func_177952_p() - spiralLayer) {
								direction = 0;
								++spiralLayer;
							}
							break;
					}

					this.theBlockPos.func_181079_c(x, y, z);
					y = Math.min(this.maxPos.func_177956_o(), this.world.func_175645_m(this.theBlockPos).func_177956_o());
				}

				return this.theBlockPos.func_181079_c(x, y, z);
			}
		}
	}
}
