/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.core.utils;

import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Random;

import net.minecraft.block.Block;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.item.Item;
import net.minecraft.item.ItemBlock;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.ResourceLocation;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;

import net.minecraftforge.fml.common.registry.ForgeRegistries;
import net.minecraftforge.fml.common.registry.IForgeRegistry;
import net.minecraftforge.oredict.OreDictionary;

public abstract class ItemStackUtil {

	public static final ItemStack[] EMPTY_STACK_ARRAY = new ItemStack[0];

	/**
	 * Compares item id, damage and NBT. Accepts wildcard damage.
	 */
	public static boolean isIdenticalItem(ItemStack lhs, ItemStack rhs) {
		if (lhs == rhs) {
			return true;
		}

		if (lhs == null || rhs == null) {
			return false;
		}

		if (lhs.func_77973_b() != rhs.func_77973_b()) {
			return false;
		}

		if (lhs.func_77952_i() != OreDictionary.WILDCARD_VALUE) {
			if (lhs.func_77952_i() != rhs.func_77952_i()) {
				return false;
			}
		}

		return ItemStack.func_77970_a(lhs, rhs);
	}
	
	/**
	 *
	 * @return The registry name of the item as {@link ResourceLocation}
	 */
	public static ResourceLocation getItemNameFromRegistry(Item item) {
		if (item == null) {
			return null;
		}
		IForgeRegistry<Item> itemRegistry = ForgeRegistries.ITEMS;
		return itemRegistry.getKey(item);
	}
	
	/**
	 *
	 * @return The registry name of the item as {@link String}
	 */
	@Nullable
	public static String getItemNameFromRegistryAsString(Item item) {
		ResourceLocation itemNameFromRegistry = getItemNameFromRegistry(item);
		if (itemNameFromRegistry == null) {
			return null;
		}
		return itemNameFromRegistry.toString();
	}

	@Nullable
	public static String getStringForItemStack(ItemStack itemStack) {
		if (itemStack == null) {
			return null;
		}

		Item item = itemStack.func_77973_b();
		if (item == null) {
			return null;
		}

		String itemStackString = getItemNameFromRegistryAsString(item);
		if (itemStackString == null) {
			return null;
		}

		int meta = itemStack.func_77952_i();
		if (meta != OreDictionary.WILDCARD_VALUE) {
			return itemStackString + ':' + meta;
		} else {
			return itemStackString;
		}
	}
	
	/**
	 * @return The item from the item registry
	 * @param itemName The registry name from the item as {@link String}
	 */
	public static Item getItemFromRegistry(String itemName) {
		return getItemFromRegistry(new ResourceLocation(itemName));
	}
	
	/**
	 * @return The item from the item registry
	 * @param itemName The registry name from the item as {@link ResourceLocation}
	 */
	public static Item getItemFromRegistry(ResourceLocation itemName) {
		if (itemName == null) {
			return null;
		}
		IForgeRegistry<Item> itemRegistry = ForgeRegistries.ITEMS;
		return itemRegistry.getValue(itemName);
	}
	
	/**
	 *
	 * @return The registry name of the block as {@link ResourceLocation}
	 */
	public static ResourceLocation getBlockNameFromRegistry(Block block) {
		if (block == null) {
			return null;
		}
		IForgeRegistry<Block> blockRegistry = ForgeRegistries.BLOCKS;
		return blockRegistry.getKey(block);
	}
	
	/**
	 *
	 * @return The registry name of the block as {@link String}
	 */
	public static String getBlockNameFromRegistryAsSting(Block block) {
		return getBlockNameFromRegistry(block).toString();
	}
	
	/**
	 * @return The block from the block registry
	 * @param itemName The registry name from the block as {@link String}
	 */
	public static Block getBlockFromRegistry(String itemName) {
		return getBlockFromRegistry(new ResourceLocation(itemName));
	}
	
	/**
	 * @return The block from the block registry
	 * @param itemName The registry name from the block as {@link ResourceLocation}
	 */
	public static Block getBlockFromRegistry(ResourceLocation itemName) {
		if (itemName == null) {
			return null;
		}
		IForgeRegistry<Block> blockRegistry = ForgeRegistries.BLOCKS;
		return blockRegistry.getValue(itemName);
	}

	/**
	 * Merges the giving stack into the receiving stack as far as possible
	 */
	public static void mergeStacks(ItemStack giver, ItemStack receptor) {
		if (receptor.field_77994_a >= 64) {
			return;
		}

		if (!receptor.func_77969_a(giver)) {
			return;
		}

		if (giver.field_77994_a <= receptor.func_77976_d() - receptor.field_77994_a) {
			receptor.field_77994_a += giver.field_77994_a;
			giver.field_77994_a = 0;
			return;
		}

		ItemStack temp = giver.func_77979_a(receptor.func_77976_d() - receptor.field_77994_a);
		receptor.field_77994_a += temp.field_77994_a;
		temp.field_77994_a = 0;
	}

	/**
	 * Creates a split stack of the specified amount, preserving NBT data,
	 * without decreasing the source stack.
	 */
	public static ItemStack createSplitStack(ItemStack stack, int amount) {
		ItemStack split = new ItemStack(stack.func_77973_b(), amount, stack.func_77952_i());
		if (stack.func_77978_p() != null) {
			NBTTagCompound nbttagcompound = (NBTTagCompound) stack.func_77978_p().func_74737_b();
			split.func_77982_d(nbttagcompound);
		}
		return split;
	}

	/**
	 */
	public static ItemStack[] condenseStacks(ItemStack[] stacks) {
		List<ItemStack> condensed = new ArrayList<>();

		for (ItemStack stack : stacks) {
			if (stack == null) {
				continue;
			}
			if (stack.field_77994_a <= 0) {
				continue;
			}

			boolean matched = false;
			for (ItemStack cached : condensed) {
				if (cached.func_77969_a(stack) && ItemStack.func_77970_a(cached, stack)) {
					cached.field_77994_a += stack.field_77994_a;
					matched = true;
				}
			}

			if (!matched) {
				ItemStack cached = stack.func_77946_l();
				condensed.add(cached);
			}

		}

		return condensed.toArray(new ItemStack[condensed.size()]);
	}

	public static boolean containsItemStack(Iterable<ItemStack> list, ItemStack itemStack) {
		for (ItemStack listStack : list) {
			if (isIdenticalItem(listStack, itemStack)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Counts how many full sets are contained in the passed stock
	 */
	public static int containsSets(ItemStack[] set, ItemStack[] stock) {
		return containsSets(set, stock, false, false);
	}

	/**
	 * Counts how many full sets are contained in the passed stock
	 */
	public static int containsSets(ItemStack[] set, ItemStack[] stock, boolean oreDictionary, boolean craftingTools) {
		int totalSets = 0;

		ItemStack[] condensedRequired = ItemStackUtil.condenseStacks(set);
		ItemStack[] condensedOffered = ItemStackUtil.condenseStacks(stock);

		for (ItemStack req : condensedRequired) {

			int reqCount = 0;
			for (ItemStack offer : condensedOffered) {
				if (isCraftingEquivalent(req, offer, oreDictionary, craftingTools)) {
					int stackCount = (int) Math.floor(offer.field_77994_a / req.field_77994_a);
					reqCount = Math.max(reqCount, stackCount);
				}
			}

			if (reqCount == 0) {
				return 0;
			} else if (totalSets == 0) {
				totalSets = reqCount;
			} else if (totalSets > reqCount) {
				totalSets = reqCount;
			}
		}

		return totalSets;
	}

	public static boolean equalSets(ItemStack[] set1, ItemStack[] set2) {
		if (set1 == set2) {
			return true;
		}

		if (set1 == null || set2 == null) {
			return false;
		}

		if (set1.length != set2.length) {
			return false;
		}

		for (int i = 0; i < set1.length; i++) {
			if (!isIdenticalItem(set1[i], set2[i])) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Compare two item stacks for crafting equivalency without oreDictionary or craftingTools
	 */
	public static boolean isCraftingEquivalent(ItemStack base, ItemStack comparison) {
		if (base == null || comparison == null) {
			return false;
		}

		if (base.func_77973_b() != comparison.func_77973_b()) {
			return false;
		}

		if (base.func_77952_i() != OreDictionary.WILDCARD_VALUE) {
			if (base.func_77952_i() != comparison.func_77952_i()) {
				return false;
			}
		}

		// When the base stackTagCompound is null or empty, treat it as a wildcard for crafting
		if (!base.func_77942_o() || base.func_77978_p().func_82582_d()) {
			return true;
		} else {
			return ItemStack.func_77970_a(base, comparison);
		}
	}

	/**
	 * Compare two item stacks for crafting equivalency.
	 */
	public static boolean isCraftingEquivalent(ItemStack base, ItemStack comparison, boolean oreDictionary, boolean craftingTools) {
		if (isCraftingEquivalent(base, comparison)) {
			return true;
		}

		if (base == null || comparison == null) {
			return false;
		}

		if (craftingTools && isCraftingToolEquivalent(base, comparison)) {
			return true;
		}

		if (base.func_77942_o() && !base.func_77978_p().func_82582_d()) {
			if (!ItemStack.func_77989_b(base, comparison)) {
				return false;
			}
		}

		if (oreDictionary) {
			int[] idsBase = OreDictionary.getOreIDs(base);
			Arrays.sort(idsBase);
			int[] idsComp = OreDictionary.getOreIDs(comparison);
			Arrays.sort(idsComp);

			// check if the sorted arrays "idsBase" and "idsComp" have any ID in common.
			int iBase = 0;
			int iComp = 0;
			while (iBase < idsBase.length && iComp < idsComp.length) {
				if (idsBase[iBase] < idsComp[iComp]) {
					iBase++;
				} else if (idsBase[iBase] > idsComp[iComp]) {
					iComp++;
				} else {
					return true;
				}
			}
		}

		return false;
	}

	public static boolean isCraftingToolEquivalent(ItemStack base, ItemStack comparison) {
		if (base == null || comparison == null) {
			return false;
		}

		Item baseItem = base.func_77973_b();

		if (baseItem != comparison.func_77973_b()) {
			return false;
		}

		if (!base.func_77942_o() || base.func_77978_p().func_82582_d()) {
			// tool uses meta for damage
			return true;
		} else {
			// tool uses NBT for damage
			if (base.func_77952_i() == OreDictionary.WILDCARD_VALUE) {
				return true;
			}
			return base.func_77952_i() == comparison.func_77952_i();
		}
	}

	public static void dropItemStackAsEntity(ItemStack items, World world, double x, double y, double z) {
		dropItemStackAsEntity(items, world, x, y, z, 10);
	}
	
	public static void dropItemStackAsEntity(ItemStack items, World world, BlockPos pos) {
		dropItemStackAsEntity(items, world, pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p(), 10);
	}

	public static void dropItemStackAsEntity(ItemStack items, World world, double x, double y, double z, int delayForPickup) {
		if (items == null || items.field_77994_a <= 0 || world.field_72995_K) {
			return;
		}

		float f1 = 0.7F;
		double d = world.field_73012_v.nextFloat() * f1 + (1.0F - f1) * 0.5D;
		double d1 = world.field_73012_v.nextFloat() * f1 + (1.0F - f1) * 0.5D;
		double d2 = world.field_73012_v.nextFloat() * f1 + (1.0F - f1) * 0.5D;
		EntityItem entityitem = new EntityItem(world, x + d, y + d1, z + d2, items);
		entityitem.func_174867_a(delayForPickup);

		world.func_72838_d(entityitem);
	}

	public static ItemStack copyWithRandomSize(ItemStack template, int max, Random rand) {
		int size = rand.nextInt(max);
		ItemStack created = template.func_77946_l();
		created.field_77994_a = size <= 0 ? 1 : size > created.func_77976_d() ? created.func_77976_d() : size;
		return created;
	}

	@Nullable
	public static Block getBlock(ItemStack stack) {
		Item item = stack.func_77973_b();

		if (item instanceof ItemBlock) {
			return ((ItemBlock) item).func_179223_d();
		} else {
			return null;
		}
	}

	public static boolean equals(Block block, ItemStack stack) {
		return block == getBlock(stack);
	}
	
	public static boolean equals(IBlockState state, ItemStack stack) {
		return state.func_177230_c() == getBlock(stack) && state.func_177230_c().func_176201_c(state) == stack.func_77952_i();
	}

	public static boolean equals(Block block, int meta, ItemStack stack) {
		return block == getBlock(stack) && meta == stack.func_77952_i();
	}

	public static List<ItemStack> parseItemStackStrings(String[] itemStackStrings, int missingMetaValue) {
		List<Stack> stacks = Stack.parseStackStrings(itemStackStrings, missingMetaValue);
		return getItemStacks(stacks);
	}

	public static List<ItemStack> parseItemStackStrings(String itemStackStrings, int missingMetaValue) {
		List<Stack> stacks = Stack.parseStackStrings(itemStackStrings, missingMetaValue);
		return getItemStacks(stacks);
	}
	
	public static ItemStack parseItemStackString(String itemStackString, int missingMetaValue) {
		Stack stack = Stack.parseStackString(itemStackString, missingMetaValue);
		return getItemStack(stack);
	}

	private static List<ItemStack> getItemStacks(List<Stack> stacks) {
		List<ItemStack> itemStacks = new ArrayList<>(stacks.size());
		for (Stack stack : stacks) {
			Item item = stack.getItem();
			if (item != null) {
				int meta = stack.getMeta();
				ItemStack itemStack = new ItemStack(item, 1, meta);
				itemStacks.add(itemStack);
			}
		}
		return itemStacks;
	}
	
	private static ItemStack getItemStack(Stack stack) {
		Item item = stack.getItem();
		if (item != null) {
			int meta = stack.getMeta();
			return new ItemStack(item, 1, meta);
		}
		return null;
	}

}
