/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.storage;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;

import javax.annotation.Nonnull;
import java.awt.Color;
import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import net.minecraft.init.Blocks;
import net.minecraft.init.Items;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;

import net.minecraftforge.client.event.ModelBakeEvent;
import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.common.config.Property;
import net.minecraftforge.fluids.Fluid;
import net.minecraftforge.fluids.FluidRegistry;
import net.minecraftforge.fluids.FluidStack;
import net.minecraftforge.fml.common.SidedProxy;
import net.minecraftforge.fml.common.event.FMLInterModComms.IMCMessage;
import net.minecraftforge.fml.common.eventhandler.SubscribeEvent;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;
import net.minecraftforge.oredict.OreDictionary;

import forestry.Forestry;
import forestry.api.core.ForestryAPI;
import forestry.api.recipes.RecipeManagers;
import forestry.api.storage.BackpackManager;
import forestry.api.storage.IBackpackDefinition;
import forestry.api.storage.StorageManager;
import forestry.apiculture.PluginApiculture;
import forestry.apiculture.blocks.BlockRegistryApiculture;
import forestry.core.IPickupHandler;
import forestry.core.IResupplyHandler;
import forestry.core.PluginCore;
import forestry.core.config.Constants;
import forestry.core.config.LocalizedConfiguration;
import forestry.core.items.ItemCrated;
import forestry.core.models.ModelCrate;
import forestry.core.recipes.RecipeUtil;
import forestry.core.utils.IMCUtil;
import forestry.core.utils.ItemStackUtil;
import forestry.core.utils.Log;
import forestry.core.utils.OreDictUtil;
import forestry.core.utils.Translator;
import forestry.lepidopterology.PluginLepidopterology;
import forestry.lepidopterology.blocks.BlockRegistryLepidopterology;
import forestry.plugins.BlankForestryPlugin;
import forestry.plugins.ForestryPlugin;
import forestry.plugins.ForestryPluginUids;
import forestry.storage.items.ItemRegistryStorage;
import forestry.storage.proxy.ProxyStorage;

@ForestryPlugin(pluginID = ForestryPluginUids.STORAGE, name = "Storage", author = "SirSengir", url = Constants.URL, unlocalizedDescription = "for.plugin.storage.description")
public class PluginStorage extends BlankForestryPlugin {

	@SidedProxy(clientSide = "forestry.storage.proxy.ProxyStorageClient", serverSide = "forestry.storage.proxy.ProxyStorage")
	public static ProxyStorage proxy;
	
	private static final List<ItemCrated> crates = new ArrayList<>();
	private static final String CONFIG_CATEGORY = "backpacks";

	public static ItemRegistryStorage items;

	private final Multimap<String, String> backpackOreDictRegexpDefaults = HashMultimap.create();
	private final Multimap<String, String> backpackItemDefaults = HashMultimap.create();

	private final List<String> forestryBackpackUids = Arrays.asList(
			BackpackManager.MINER_UID,
			BackpackManager.DIGGER_UID,
			BackpackManager.FORESTER_UID,
			BackpackManager.HUNTER_UID,
			BackpackManager.ADVENTURER_UID,
			BackpackManager.BUILDER_UID
	);

	@Override
	public void setupAPI() {
		super.setupAPI();

		StorageManager.crateRegistry = new CrateRegistry();

		BackpackManager.backpackInterface = new BackpackInterface();

		BackpackDefinition definition;

		if (ForestryAPI.enabledPlugins.contains(ForestryPluginUids.APICULTURE)) {
			definition = new BackpackDefinition.BackpackDefinitionNaturalist(new Color(0xc4923d), "rootBees");
			BackpackManager.backpackInterface.registerBackpack("apiarist", definition);
		}

		if (ForestryAPI.enabledPlugins.contains(ForestryPluginUids.LEPIDOPTEROLOGY)) {
			definition = new BackpackDefinition.BackpackDefinitionNaturalist(new Color(0x995b31), "rootButterflies");
			BackpackManager.backpackInterface.registerBackpack("lepidopterist", definition);
		}

		definition = new BackpackDefinition(new Color(0x36187d));
		BackpackManager.backpackInterface.registerBackpack(BackpackManager.MINER_UID, definition);

		definition = new BackpackDefinition(new Color(0x363cc5));
		BackpackManager.backpackInterface.registerBackpack(BackpackManager.DIGGER_UID, definition);

		definition = new BackpackDefinition(new Color(0x347427));
		BackpackManager.backpackInterface.registerBackpack(BackpackManager.FORESTER_UID, definition);

		definition = new BackpackDefinition(new Color(0x412215));
		BackpackManager.backpackInterface.registerBackpack(BackpackManager.HUNTER_UID, definition);

		definition = new BackpackDefinition(new Color(0x7fb8c2));
		BackpackManager.backpackInterface.registerBackpack(BackpackManager.ADVENTURER_UID, definition);

		definition = new BackpackDefinition(new Color(0xdd3a3a));
		BackpackManager.backpackInterface.registerBackpack(BackpackManager.BUILDER_UID, definition);
		
		proxy.registerCrateModel();
	}

	@Override
	public void registerItemsAndBlocks() {
		items = new ItemRegistryStorage();
	}
	
	@Override
	public void preInit() {
		registerFenceAndFenceGatesToOreDict();
		MinecraftForge.EVENT_BUS.register(this);
	}

	@Override
	public void postInit() {
		final String newConfig = CONFIG_CATEGORY + ".cfg";

		File configFile = new File(Forestry.instance.getConfigFolder(), newConfig);
		LocalizedConfiguration config = new LocalizedConfiguration(configFile, "1.0.0");

		setDefaultsForConfig();

		for (String backpackUid : forestryBackpackUids)  {
			handleBackpackConfig(config, backpackUid);
		}

		config.save();
	}

	private void setDefaultsForConfig() {
		backpackOreDictRegexpDefaults.get(BackpackManager.MINER_UID).addAll(Arrays.asList(
				"obsidian",
				"ore[A-Z].*",
				"dust[A-Z].*",
				"gem[A-Z].*",
				"ingot[A-Z].*",
				"nugget[A-Z].*",
				"crushed[A-Z].*",
				"cluster[A-Z].*",
				"denseore[A-Z].*"
		));

		backpackOreDictRegexpDefaults.get(BackpackManager.DIGGER_UID).addAll(Arrays.asList(
				"cobblestone",
				"dirt",
				"gravel",
				"netherrack",
				"stone",
				"stone[A-Z].*",
				"sand"
		));

		backpackOreDictRegexpDefaults.get(BackpackManager.HUNTER_UID).addAll(Arrays.asList(
				"bone",
				"egg",
				"enderpearl",
				"feather",
				"fish[A-Z].*",
				"gunpowder",
				"leather",
				"slimeball",
				"string"
		));

		backpackOreDictRegexpDefaults.get(BackpackManager.FORESTER_UID).addAll(Arrays.asList(
				"logWood",
				"stickWood",
				"woodStick",
				"saplingTree",
				"treeSapling",
				"vine",
				"sugarcane",
				"blockCactus",
				"crop[A-Z].*",
				"seed[A-Z].*",
				"tree[A-Z].*"
		));

		backpackOreDictRegexpDefaults.get(BackpackManager.BUILDER_UID).addAll(Arrays.asList(
				"block[A-Z].*",
				"paneGlass[A-Z].*",
				"slabWood[A-Z].*",
				"stainedClay[A-Z].*",
				"stainedGlass[A-Z].*",
				"stone",
				"sandstone",
				"plankWood",
				"stairWood",
				"slabWood",
				OreDictUtil.FENCE_WOOD,
				OreDictUtil.FENCE_GATE_WOOD,
				"glass",
				"paneGlass",
				"torch",
				"chest",
				"chest[A-Z].*",
				"workbench",
				"doorWood"
		));

		backpackItemDefaults.get(BackpackManager.MINER_UID).addAll(getItemStrings(Arrays.asList(
				new ItemStack(Blocks.field_150365_q),
				new ItemStack(Items.field_151044_h),
				PluginCore.items.bronzePickaxe.getItemStack(),
				PluginCore.items.kitPickaxe.getItemStack(),
				PluginCore.items.brokenBronzePickaxe.getItemStack()
		)));

		backpackItemDefaults.get(BackpackManager.DIGGER_UID).addAll(getItemStrings(Arrays.asList(
				new ItemStack(Blocks.field_150346_d, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Items.field_151145_ak),
				new ItemStack(Blocks.field_150322_A, 1, 0),
				new ItemStack(Items.field_151119_aD),
				new ItemStack(Items.field_151126_ay),
				new ItemStack(Blocks.field_150425_aM),
				PluginCore.items.bronzeShovel.getItemStack(),
				PluginCore.items.kitShovel.getItemStack(),
				PluginCore.items.brokenBronzeShovel.getItemStack()
		)));

		backpackItemDefaults.get(BackpackManager.FORESTER_UID).addAll(getItemStrings(Arrays.asList(
				new ItemStack(Blocks.field_150337_Q),
				new ItemStack(Blocks.field_150338_P),
				new ItemStack(Blocks.field_150328_O, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_150327_N),
				new ItemStack(Blocks.field_150329_H, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_150398_cm, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_150423_aK),
				new ItemStack(Blocks.field_150440_ba),
				new ItemStack(Items.field_151153_ao),
				new ItemStack(Items.field_151075_bm),
				new ItemStack(Items.field_151014_N),
				new ItemStack(Items.field_151080_bb),
				new ItemStack(Items.field_151081_bc),
				new ItemStack(Items.field_185163_cU),
				new ItemStack(Items.field_185164_cV),
				new ItemStack(Items.field_185161_cS),
				new ItemStack(Items.field_151034_e)
		)));

		backpackItemDefaults.get(BackpackManager.HUNTER_UID).addAll(getItemStrings(Arrays.asList(
				new ItemStack(Items.field_151065_br),
				new ItemStack(Items.field_151072_bj),
				new ItemStack(Items.field_151078_bh),
				new ItemStack(Items.field_151144_bL, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Items.field_151073_bk),
				new ItemStack(Items.field_151074_bl),
				new ItemStack(Items.field_151032_g),
				new ItemStack(Items.field_185166_h),
				new ItemStack(Items.field_185167_i),
				new ItemStack(Items.field_151147_al),
				new ItemStack(Items.field_151157_am),
				new ItemStack(Items.field_151082_bd),
				new ItemStack(Items.field_151083_be),
				new ItemStack(Items.field_151076_bf),
				new ItemStack(Items.field_151077_bg),
				new ItemStack(Items.field_179561_bm),
				new ItemStack(Items.field_179557_bn),
				new ItemStack(Items.field_179558_bo),
				new ItemStack(Items.field_179559_bp),
				new ItemStack(Items.field_179556_br),
				new ItemStack(Items.field_179555_bs),
				new ItemStack(Items.field_151070_bp),
				new ItemStack(Items.field_151071_bq),
				new ItemStack(Items.field_151100_aR, 1, 0),
				new ItemStack(Blocks.field_150407_cf),
				new ItemStack(Blocks.field_150325_L, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Items.field_151061_bv),
				new ItemStack(Items.field_151064_bs),
				new ItemStack(Items.field_151060_bw),
				new ItemStack(Items.field_151115_aP, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Items.field_179566_aV, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Items.field_151058_ca),
				new ItemStack(Items.field_151112_aM),
				new ItemStack(Items.field_151057_cb),
				new ItemStack(Items.field_151141_av),
				new ItemStack(Items.field_151125_bZ),
				new ItemStack(Items.field_151136_bY),
				new ItemStack(Items.field_151138_bX)
		)));

		backpackItemDefaults.get(BackpackManager.BUILDER_UID).addAll(getItemStrings(Arrays.asList(
				new ItemStack(Blocks.field_150429_aA),
				new ItemStack(Blocks.field_150379_bu),
				new ItemStack(Blocks.field_180398_cJ),
				new ItemStack(Blocks.field_185764_cQ),
				new ItemStack(Blocks.field_150417_aV, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_150336_V),
				new ItemStack(Blocks.field_150435_aG),
				new ItemStack(Blocks.field_150405_ch, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_150406_ce, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_150403_cj),
				new ItemStack(Blocks.field_150385_bj),
				new ItemStack(Blocks.field_150386_bk),
				new ItemStack(Blocks.field_150462_ai),
				new ItemStack(Blocks.field_150460_al),
				new ItemStack(Blocks.field_150442_at),
				new ItemStack(Blocks.field_150367_z),
				new ItemStack(Blocks.field_150409_cd),
				new ItemStack(Blocks.field_150468_ap),
				new ItemStack(Blocks.field_150411_aY),
				new ItemStack(Blocks.field_150371_ca, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_150370_cb),
				new ItemStack(Blocks.field_150372_bz),
				new ItemStack(Blocks.field_180396_cN),
				new ItemStack(Blocks.field_150463_bK, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_150430_aB),
				new ItemStack(Blocks.field_150471_bO),
				new ItemStack(Blocks.field_150333_U, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_180389_cP, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_150376_bx, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_185767_cT),
				new ItemStack(Blocks.field_185768_cU),
				new ItemStack(Blocks.field_185769_cV),
				new ItemStack(Blocks.field_185771_cX),
				new ItemStack(Blocks.field_185772_cY),
				new ItemStack(Blocks.field_150404_cg, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_150415_aT),
				new ItemStack(Blocks.field_180400_cw),
				new ItemStack(Blocks.field_150456_au),
				new ItemStack(Blocks.field_150452_aw),
				new ItemStack(Blocks.field_150445_bS),
				new ItemStack(Blocks.field_150443_bT),
				new ItemStack(Items.field_151155_ap),
				new ItemStack(Items.field_151160_bD),
				new ItemStack(Items.field_179572_au),
				new ItemStack(Items.field_179568_as),
				new ItemStack(Items.field_179571_av),
				new ItemStack(Items.field_151139_aw),
				new ItemStack(Items.field_179567_at),
				new ItemStack(Items.field_179570_aq),
				new ItemStack(Items.field_179569_ar)
		)));

		BlockRegistryApiculture beeBlocks = PluginApiculture.blocks;
		if (beeBlocks != null) {
			backpackItemDefaults.get(BackpackManager.BUILDER_UID).addAll(getItemStrings(Arrays.asList(
					new ItemStack(beeBlocks.candle, 1, OreDictionary.WILDCARD_VALUE),
					new ItemStack(beeBlocks.stump, 1, OreDictionary.WILDCARD_VALUE)
			)));
		}

		// include everything added via the API
		for (String backpackUid : forestryBackpackUids)  {
			BackpackDefinition backpackDefinition = (BackpackDefinition) BackpackManager.backpackInterface.getBackpack(backpackUid);

			Collection<String> oreDictDefaults = backpackOreDictRegexpDefaults.get(backpackUid);
			for (int oreId : backpackDefinition.getValidOreIds()) {
				String oreName = OreDictionary.getOreName(oreId);
				oreDictDefaults.add(oreName);
			}

			backpackItemDefaults.get(backpackUid).addAll(backpackDefinition.getValidItemStacks());
		}
	}

	// Should be ore dicted in Forge at some point.
	private static void registerFenceAndFenceGatesToOreDict() {
		OreDictionary.registerOre(OreDictUtil.FENCE_WOOD, Blocks.field_180407_aO);
		OreDictionary.registerOre(OreDictUtil.FENCE_WOOD, Blocks.field_180408_aP);
		OreDictionary.registerOre(OreDictUtil.FENCE_WOOD, Blocks.field_180404_aQ);
		OreDictionary.registerOre(OreDictUtil.FENCE_WOOD, Blocks.field_180403_aR);
		OreDictionary.registerOre(OreDictUtil.FENCE_WOOD, Blocks.field_180406_aS);
		OreDictionary.registerOre(OreDictUtil.FENCE_WOOD, Blocks.field_180405_aT);
		OreDictionary.registerOre(OreDictUtil.FENCE_GATE_WOOD, Blocks.field_180390_bo);
		OreDictionary.registerOre(OreDictUtil.FENCE_GATE_WOOD, Blocks.field_180391_bp);
		OreDictionary.registerOre(OreDictUtil.FENCE_GATE_WOOD, Blocks.field_180392_bq);
		OreDictionary.registerOre(OreDictUtil.FENCE_GATE_WOOD, Blocks.field_180386_br);
		OreDictionary.registerOre(OreDictUtil.FENCE_GATE_WOOD, Blocks.field_180385_bs);
		OreDictionary.registerOre(OreDictUtil.FENCE_GATE_WOOD, Blocks.field_180387_bt);
	}

	@Nonnull
	private static Set<String> getItemStrings(List<ItemStack> itemStacks) {
		Set<String> itemStrings = new HashSet<>(itemStacks.size());
		for (ItemStack itemStack : itemStacks) {
			String itemString = ItemStackUtil.getStringForItemStack(itemStack);
			itemStrings.add(itemString);
		}
		return itemStrings;
	}

	private void handleBackpackConfig(LocalizedConfiguration config, String backpackUid) {
		BackpackDefinition backpackDefinition = (BackpackDefinition) BackpackManager.backpackInterface.getBackpack(backpackUid);
		backpackDefinition.clearAllValid();

		{
			List<String> defaultItemNames = new ArrayList<>(backpackItemDefaults.get(backpackUid));
			Collections.sort(defaultItemNames);
			String[] defaultValidItems = defaultItemNames.toArray(new String[defaultItemNames.size()]);

			Property backpackConf = config.get("backpacks." + backpackUid, "item.stacks", defaultValidItems);
			backpackConf.setComment(Translator.translateToLocalFormatted("for.config.backpacks.item.stacks.format", backpackUid));

			String[] backpackItemList = backpackConf.getStringList();
			List<ItemStack> backpackItems = ItemStackUtil.parseItemStackStrings(backpackItemList, OreDictionary.WILDCARD_VALUE);
			backpackDefinition.addValidItems(backpackItems);
		}

		{

			List<String> defaultOreRegexpList = new ArrayList<>(backpackOreDictRegexpDefaults.get(backpackUid));
			Collections.sort(defaultOreRegexpList);
			String[] defaultOreRegexpNames = defaultOreRegexpList.toArray(new String[defaultOreRegexpList.size()]);

			Property backpackConf = config.get("backpacks." + backpackUid, "ore.dict", defaultOreRegexpNames);
			backpackConf.setComment(Translator.translateToLocalFormatted("for.config.backpacks.ore.dict.format", backpackUid));

			List<String> oreDictNameList = new ArrayList<>();
			for (String name : OreDictionary.getOreNames()) {
				if (name == null) {
					Log.error("Found a null oreName in the ore dictionary");
				} else {
					for (String regex : backpackConf.getStringList()) {
						if (name.matches(regex)) {
							oreDictNameList.add(name);
						}
					}
				}
			}

			backpackDefinition.addValidOreDictNames(oreDictNameList);
		}
	}

	public static void registerCrate(ItemCrated crate) {
		crates.add(crate);
	}

	public static void createCrateRecipes() {
		for (ItemCrated crate : crates) {
			ItemStack crateStack = new ItemStack(crate);
			ItemStack uncrated = crate.getContained();
			if (uncrated != null) {
				if (crate.getOreDictName() != null) {
					addCrating(crateStack, crate.getOreDictName());
				} else {
					addCrating(crateStack, uncrated);
				}
				addUncrating(crateStack, uncrated);
			}
		}
	}

	private static void addCrating(@Nonnull ItemStack crateStack, @Nonnull Object uncrated) {
		FluidStack water = new FluidStack(FluidRegistry.WATER, Constants.CARPENTER_CRATING_LIQUID_QUANTITY);
		ItemStack box = items.crate.getItemStack();
		RecipeManagers.carpenterManager.addRecipe(Constants.CARPENTER_CRATING_CYCLES, water, box, crateStack, "###", "###", "###", '#', uncrated);
	}

	private static void addUncrating(@Nonnull ItemStack crateStack, @Nonnull ItemStack uncrated) {
		ItemStack product = new ItemStack(uncrated.func_77973_b(), 9, uncrated.func_77952_i());
		RecipeManagers.carpenterManager.addRecipe(Constants.CARPENTER_UNCRATING_CYCLES, null, product, "#", '#', crateStack);
	}

	@Override
	public boolean processIMCMessage(IMCMessage message) {
		if (message.key.equals("add-backpack-items")) {
			String[] tokens = message.getStringValue().split("@");
			if (tokens.length != 2) {
				IMCUtil.logInvalidIMCMessage(message);
				return true;
			}

			IBackpackDefinition backpackDefinition = BackpackManager.backpackInterface.getBackpack(tokens[0]);
			if (backpackDefinition == null) {
				String errorMessage = IMCUtil.getInvalidIMCMessageText(message);
				Log.warning("%s For non-existent backpack %s.", errorMessage, tokens[0]);
				return true;
			}
			List<ItemStack> itemStacks = ItemStackUtil.parseItemStackStrings(tokens[1], 0);
			backpackDefinition.addValidItems(itemStacks);

			return true;
		}
		return false;
	}

	@Override
	public IPickupHandler getPickupHandler() {
		return new PickupHandlerStorage();
	}

	@Override
	public IResupplyHandler getResupplyHandler() {
		return new ResupplyHandler();
	}

	@Override
	public void registerRecipes() {
		BlockRegistryApiculture beeBlocks = PluginApiculture.blocks;
		if (items.apiaristBackpack != null && beeBlocks != null) {
			addBackpackRecipe(items.apiaristBackpack, "stickWood", beeBlocks.beeChest);
		}

		BlockRegistryLepidopterology butterflyBlocks = PluginLepidopterology.blocks;
		if (items.lepidopteristBackpack != null && butterflyBlocks != null) {
			ItemStack chest = new ItemStack(butterflyBlocks.butterflyChest);
			addBackpackRecipe(items.lepidopteristBackpack, "stickWood", chest);
		}

		addBackpackRecipe(items.minerBackpack, "ingotIron");
		addBackpackRecipe(items.diggerBackpack, "stone");
		addBackpackRecipe(items.foresterBackpack, "logWood");
		addBackpackRecipe(items.hunterBackpack, Items.field_151008_G);
		addBackpackRecipe(items.adventurerBackpack, Items.field_151103_aS);
		addBackpackRecipe(items.builderBackpack, Items.field_151119_aD);

		// / CARPENTER
		if (ForestryAPI.enabledPlugins.contains(ForestryPluginUids.FACTORY)) {
			// / CRATES
			RecipeManagers.carpenterManager.addRecipe(20, new FluidStack(FluidRegistry.WATER, Fluid.BUCKET_VOLUME), null, items.crate.getItemStack(24),
					" # ", "# #", " # ", '#', "logWood");

			// / BACKPACKS WOVEN
			addT2BackpackRecipe(items.minerBackpack, items.minerBackpackT2);
			addT2BackpackRecipe(items.diggerBackpack, items.diggerBackpackT2);
			addT2BackpackRecipe(items.foresterBackpack, items.foresterBackpackT2);
			addT2BackpackRecipe(items.hunterBackpack, items.hunterBackpackT2);
			addT2BackpackRecipe(items.adventurerBackpack, items.adventurerBackpackT2);
			addT2BackpackRecipe(items.builderBackpack, items.builderBackpackT2);
		}
	}

	private static void addBackpackRecipe(Item backpack, Object material) {
		addBackpackRecipe(backpack, material, "chestWood");
	}

	private static void addBackpackRecipe(Item backpack, Object material, Object chest) {
		RecipeUtil.addRecipe(backpack,
				"X#X",
				"VYV",
				"X#X",
				'#', Blocks.field_150325_L,
				'X', Items.field_151007_F,
				'V', material,
				'Y', chest);
	}

	private static void addT2BackpackRecipe(Item backpackT1, Item backpackT2) {
		ItemStack wovenSilk = PluginCore.items.craftingMaterial.getWovenSilk();
		RecipeManagers.carpenterManager.addRecipe(200, new FluidStack(FluidRegistry.WATER, Fluid.BUCKET_VOLUME), null, new ItemStack(backpackT2),
				"WXW",
				"WTW",
				"WWW",
				'X', "gemDiamond",
				'W', wovenSilk,
				'T', backpackT1);
	}
	
	@SubscribeEvent
	@SideOnly(Side.CLIENT)
	public void onBakeModel(ModelBakeEvent event) {
		ModelCrate.initModel(event);
	}
}
