/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.apiculture.worldgen;

import javax.annotation.Nonnull;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import net.minecraft.block.Block;
import net.minecraft.block.BlockDoor;
import net.minecraft.block.BlockHorizontal;
import net.minecraft.block.BlockLog;
import net.minecraft.block.BlockSlab;
import net.minecraft.block.BlockStairs;
import net.minecraft.block.BlockTorch;
import net.minecraft.block.state.IBlockState;
import net.minecraft.init.Blocks;
import net.minecraft.item.ItemStack;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;
import net.minecraft.world.biome.Biome;
import net.minecraft.world.gen.structure.StructureBoundingBox;
import net.minecraft.world.gen.structure.StructureComponent;
import net.minecraft.world.gen.structure.StructureVillagePieces;

import net.minecraftforge.fml.common.registry.FMLControlledNamespacedRegistry;
import net.minecraftforge.fml.common.registry.VillagerRegistry;

import forestry.api.apiculture.BeeManager;
import forestry.api.apiculture.EnumBeeType;
import forestry.api.apiculture.FlowerManager;
import forestry.api.apiculture.IAlleleBeeSpecies;
import forestry.api.apiculture.IBee;
import forestry.api.apiculture.IBeeGenome;
import forestry.api.arboriculture.EnumForestryWoodType;
import forestry.api.arboriculture.EnumVanillaWoodType;
import forestry.api.arboriculture.IWoodAccess;
import forestry.api.arboriculture.IWoodType;
import forestry.api.arboriculture.TreeManager;
import forestry.api.arboriculture.WoodBlockKind;
import forestry.api.core.EnumHumidity;
import forestry.api.core.EnumTemperature;
import forestry.api.core.ForestryAPI;
import forestry.api.genetics.AlleleManager;
import forestry.apiculture.PluginApiculture;
import forestry.apiculture.flowers.Flower;
import forestry.apiculture.flowers.FlowerRegistry;
import forestry.apiculture.tiles.TileBeeHouse;
import forestry.arboriculture.PluginArboriculture;
import forestry.core.PluginCore;
import forestry.core.blocks.BlockBase;
import forestry.core.config.Constants;
import forestry.core.tiles.TileUtil;
import forestry.plugins.ForestryPluginUids;

public class VillageApiaristHouse extends StructureVillagePieces.House1 {

	private static final Random random = new Random();

	private int averageGroundLevel = -1;
	private boolean isInDesert = false;
	private IBlockState planks;
	private IBlockState slabs;
	private IBlockState logs;
	private IBlockState stairs;
	private IBlockState fence;
	private IBlockState door;
	private IBlockState fenceGate;

	@SuppressWarnings("unused")
	public VillageApiaristHouse() {
		createBuildingBlocks(random);
	}

	public VillageApiaristHouse(StructureVillagePieces.Start startPiece, int componentType, Random random, StructureBoundingBox boundingBox, EnumFacing facing) {
		super(startPiece, componentType, random, boundingBox, facing);

		isInDesert = startPiece.field_74927_b;

		createBuildingBlocks(random);
	}

	private void createBuildingBlocks(Random random) {
		boolean fireproof = random.nextInt(4) == 0;
		IWoodType woodType;

		if (ForestryAPI.enabledPlugins.contains(ForestryPluginUids.ARBORICULTURE)) {
			woodType = EnumForestryWoodType.getRandom(random);
		} else {
			woodType = EnumVanillaWoodType.getRandom(random);
		}

		IWoodAccess woodAccess = TreeManager.woodAccess;
		this.logs = woodAccess.getBlock(woodType, WoodBlockKind.LOG, fireproof).func_177226_a(BlockLog.field_176299_a, BlockLog.EnumAxis.X);
		this.planks = woodAccess.getBlock(woodType, WoodBlockKind.PLANKS, fireproof);
		this.slabs = woodAccess.getBlock(woodType, WoodBlockKind.SLAB, fireproof);
		this.stairs = woodAccess.getBlock(woodType, WoodBlockKind.STAIRS, fireproof);
		this.fence = woodAccess.getBlock(woodType, WoodBlockKind.FENCE, fireproof);
		this.door = woodAccess.getBlock(woodType, WoodBlockKind.DOOR, false);
		this.fenceGate = woodAccess.getBlock(woodType, WoodBlockKind.FENCE_GATE, fireproof);
	}

	public static VillageApiaristHouse buildComponent(StructureVillagePieces.Start startPiece, List<StructureComponent> par1List, Random random, int structureMinX, int structureMinY, int structureMinZ, EnumFacing facing, int componentType) {
		StructureBoundingBox bbox = StructureBoundingBox.func_175897_a(structureMinX, structureMinY, structureMinZ, -4, 0, 0, 12, 9, 12, facing);
		if (!func_74895_a(bbox) || StructureComponent.func_74883_a(par1List, bbox) != null) {
			return null;
		}

		return new VillageApiaristHouse(startPiece, componentType, random, bbox, facing);
	}

	@Override
	public boolean func_74875_a(@Nonnull World world, @Nonnull Random random, @Nonnull StructureBoundingBox structBoundingBox) {

		if (averageGroundLevel < 0) {
			averageGroundLevel = func_74889_b(world, structBoundingBox);
			if (averageGroundLevel < 0) {
				return true;
			}

			field_74887_e.func_78886_a(0, averageGroundLevel - field_74887_e.field_78894_e + 9 - 1, 0);
		}

		func_175804_a(world, structBoundingBox, 1, 1, 1, 7, 4, 4, Blocks.field_150350_a.func_176223_P(), Blocks.field_150350_a.func_176223_P(), false);
		func_175804_a(world, structBoundingBox, 2, 1, 6, 8, 4, 10, Blocks.field_150350_a.func_176223_P(), Blocks.field_150350_a.func_176223_P(), false);

		// Garden
		buildGarden(world, structBoundingBox);

		// Garden fence
		func_175804_a(world, structBoundingBox, 1, 1, 6, 1, 1, 10, fence, fence, false);
		func_175804_a(world, structBoundingBox, 8, 1, 6, 8, 1, 10, fence, fence, false);
		func_175804_a(world, structBoundingBox, 2, 1, 10, 7, 1, 10, fence, fence, false);

		func_175811_a(world, fenceGate.func_177226_a(BlockHorizontal.field_185512_D, EnumFacing.EAST), 8, 1, 8, structBoundingBox);
		func_175811_a(world, fenceGate.func_177226_a(BlockHorizontal.field_185512_D, EnumFacing.EAST), 1, 1, 8, structBoundingBox);
		func_175811_a(world, fenceGate.func_177226_a(BlockHorizontal.field_185512_D, EnumFacing.NORTH), 4, 1, 10, structBoundingBox);

		// Flowers
		plantFlowerGarden(world, structBoundingBox, 2, 1, 5, 7, 1, 9);

		// Apiaries
		buildApiaries(world, structBoundingBox);

		// Floor
		IBlockState slabFloor = slabs.func_177226_a(BlockSlab.field_176554_a, BlockSlab.EnumBlockHalf.BOTTOM);
		func_175804_a(world, structBoundingBox, 2, 0, 1, 6, 0, 4, slabFloor, slabFloor, false);
		func_175804_a(world, structBoundingBox, 1, 0, 1, 1, 0, 4, planks, planks, false);
		func_175804_a(world, structBoundingBox, 7, 0, 1, 7, 0, 4, planks, planks, false);

		IBlockState cobblestoneState = Blocks.field_150347_e.func_176223_P();
		func_175804_a(world, structBoundingBox, 0, 0, 0, 0, 2, 5, cobblestoneState, cobblestoneState, false);
		func_175804_a(world, structBoundingBox, 8, 0, 0, 8, 2, 5, cobblestoneState, cobblestoneState, false);
		func_175804_a(world, structBoundingBox, 1, 0, 0, 7, 1, 0, cobblestoneState, cobblestoneState, false);
		func_175804_a(world, structBoundingBox, 1, 0, 5, 7, 1, 5, cobblestoneState, cobblestoneState, false);

		func_175804_a(world, structBoundingBox, 0, 3, 0, 0, 3, 5, planks, planks, false);
		func_175804_a(world, structBoundingBox, 8, 3, 0, 8, 3, 5, planks, planks, false);
		func_175804_a(world, structBoundingBox, 1, 2, 0, 7, 3, 0, planks, planks, false);
		func_175804_a(world, structBoundingBox, 1, 2, 5, 7, 3, 5, planks, planks, false);
		
		// Ceiling
		IBlockState slabCeiling = slabs.func_177226_a(BlockSlab.field_176554_a, BlockSlab.EnumBlockHalf.TOP);
		func_175804_a(world, structBoundingBox, 1, 4, 1, 7, 4, 4, slabCeiling, slabCeiling, false);

		IBlockState logBracing = logs.func_177226_a(BlockLog.field_176299_a, BlockLog.EnumAxis.X);
		func_175804_a(world, structBoundingBox, 0, 4, 1, 8, 4, 1, logBracing, logBracing, false);
		func_175804_a(world, structBoundingBox, 0, 4, 4, 8, 4, 4, logBracing, logBracing, false);

		func_175804_a(world, structBoundingBox, 0, 5, 2, 8, 5, 3, planks, planks, false);

		func_175811_a(world, planks, 0, 4, 2, structBoundingBox);
		func_175811_a(world, planks, 0, 4, 3, structBoundingBox);
		func_175811_a(world, planks, 8, 4, 2, structBoundingBox);
		func_175811_a(world, planks, 8, 4, 3, structBoundingBox);

		buildRoof(world, structBoundingBox);

		// sides of windows
		func_175811_a(world, planks, 0, 2, 1, structBoundingBox);
		func_175811_a(world, planks, 0, 2, 4, structBoundingBox);
		func_175811_a(world, planks, 8, 2, 1, structBoundingBox);
		func_175811_a(world, planks, 8, 2, 4, structBoundingBox);
		
		IBlockState glassPaneState = Blocks.field_150410_aZ.func_176223_P();

		// Windows on east side
		func_175811_a(world, glassPaneState, 0, 2, 2, structBoundingBox);
		func_175811_a(world, glassPaneState, 0, 2, 3, structBoundingBox);
		// stairs over window
		IBlockState eastStairs = stairs.func_177226_a(BlockStairs.field_176309_a, EnumFacing.EAST);
		func_175811_a(world, eastStairs, -1, 3, 2, structBoundingBox);
		func_175811_a(world, eastStairs, -1, 3, 3, structBoundingBox);

		// Windows on west side
		func_175811_a(world, glassPaneState, 8, 2, 2, structBoundingBox);
		func_175811_a(world, glassPaneState, 8, 2, 3, structBoundingBox);
		// stairs over window
		IBlockState westStairs = stairs.func_177226_a(BlockStairs.field_176309_a, EnumFacing.WEST);
		func_175811_a(world, westStairs, 9, 3, 2, structBoundingBox);
		func_175811_a(world, westStairs, 9, 3, 3, structBoundingBox);

		// Windows garden side
		func_175811_a(world, glassPaneState, 2, 2, 5, structBoundingBox);
		func_175811_a(world, glassPaneState, 3, 2, 5, structBoundingBox);
		func_175811_a(world, glassPaneState, 4, 2, 5, structBoundingBox);

		// Windows front side
		func_175811_a(world, glassPaneState, 5, 2, 0, structBoundingBox);
		func_175811_a(world, glassPaneState, 6, 2, 5, structBoundingBox);

		// Escritoire
		if (random.nextInt(2) == 0) {
			IBlockState escritoireBlock = PluginCore.blocks.escritoire.func_176223_P().func_177226_a(BlockBase.FACING, EnumFacing.EAST);
			func_175811_a(world, escritoireBlock, 1, 1, 3, structBoundingBox);
		}

		IBlockState airState = Blocks.field_150350_a.func_176223_P();

		this.func_175811_a(world, this.door.func_177226_a(BlockDoor.field_176520_a, EnumFacing.NORTH), 2, 1, 0, structBoundingBox);
		this.func_175811_a(world, this.door.func_177226_a(BlockDoor.field_176520_a, EnumFacing.NORTH).func_177226_a(BlockDoor.field_176523_O, BlockDoor.EnumDoorHalf.UPPER), 2, 2, 0, structBoundingBox);

		this.func_175811_a(world, Blocks.field_150478_aa.func_176223_P().func_177226_a(BlockTorch.field_176596_a, EnumFacing.NORTH), 2, 3, 1, structBoundingBox);

		if (isAirBlockAtCurrentPosition(world, new BlockPos(2, 0, -1), structBoundingBox) && !isAirBlockAtCurrentPosition(world, new BlockPos(2, -1, -1), structBoundingBox)) {
			func_175811_a(world, stairs.func_177226_a(BlockStairs.field_176309_a, EnumFacing.NORTH), 2, 0, -1, structBoundingBox);
		}

		func_175811_a(world, airState, 6, 1, 5, structBoundingBox);
		func_175811_a(world, airState, 6, 2, 5, structBoundingBox);

		this.func_175811_a(world, Blocks.field_150478_aa.func_176223_P().func_177226_a(BlockTorch.field_176596_a, EnumFacing.SOUTH), 6, 3, 4, structBoundingBox);

		this.func_175811_a(world, this.door.func_177226_a(BlockDoor.field_176520_a, EnumFacing.SOUTH), 6, 1, 5, structBoundingBox);
		this.func_175811_a(world, this.door.func_177226_a(BlockDoor.field_176520_a, EnumFacing.SOUTH).func_177226_a(BlockDoor.field_176523_O, BlockDoor.EnumDoorHalf.UPPER), 6, 2, 5, structBoundingBox);

		for (int i = 0; i < 5; ++i) {
			for (int j = 0; j < 9; ++j) {
				func_74871_b(world, j, 7, i, structBoundingBox);
				func_175808_b(world, Blocks.field_150347_e.func_176223_P(), j, -1, i, structBoundingBox);
			}
		}

		func_186167_a(world, structBoundingBox, random, 7, 1, 3, Constants.VILLAGE_NATURALIST_LOOT_KEY);

		// Inside Corners
		func_175804_a(world, structBoundingBox, 1, 1, 1, 1, 3, 1, fence, fence, false);
		func_175804_a(world, structBoundingBox, 1, 1, 4, 1, 3, 4, fence, fence, false);
		func_175804_a(world, structBoundingBox, 7, 1, 1, 7, 3, 1, fence, fence, false);
		func_175804_a(world, structBoundingBox, 7, 1, 4, 7, 3, 4, fence, fence, false);

		func_74893_a(world, field_74887_e, 2, 1, 2, 2);

		return true;
	}

	private void buildRoof(World world, StructureBoundingBox structBoundingBox) {
		for (int z = -1; z <= 2; ++z) {
			for (int x = 0; x <= 8; ++x) {
				IBlockState northStairs = stairs.func_177226_a(BlockStairs.field_176309_a, EnumFacing.NORTH);
				func_175811_a(world, northStairs, x, 4 + z, z, structBoundingBox);

				IBlockState southStairs = stairs.func_177226_a(BlockStairs.field_176309_a, EnumFacing.SOUTH);
				func_175811_a(world, southStairs, x, 4 + z, 5 - z, structBoundingBox);
			}
		}
	}

	private void buildGarden(World world, StructureBoundingBox box) {

		Block ground = Blocks.field_150346_d;
		if (isInDesert) {
			ground = Blocks.field_150354_m;
		}

		for (int i = 1; i <= 8; i++) {
			for (int j = 6; j <= 10; j++) {
				func_175808_b(world, ground.func_176223_P(), i, 0, j, box);
			}
		}
	}

	private void plantFlowerGarden(World world, StructureBoundingBox box, int minX, int minY, int minZ, int maxX, int maxY, int maxZ) {

		if (isInDesert) {
			func_175811_a(world, Blocks.field_150434_aF.func_176223_P(), 4, 1, 7, box);
			return;
		}

		for (int i = minY; i <= maxY; ++i) {
			for (int j = minX; j <= maxX; ++j) {
				for (int k = minZ; k <= maxZ; ++k) {
					if (world.field_73012_v.nextBoolean()) {
						int xCoord = func_74865_a(j, k);
						int yCoord = func_74862_a(i);
						int zCoord = func_74873_b(j, k);

						BlockPos pos = new BlockPos(xCoord, yCoord, zCoord);
						IBlockState blockState = world.func_180495_p(pos);
						if (!Blocks.field_150328_O.func_180671_f(world, pos, blockState)) {
							continue;
						}

						FlowerRegistry flowerRegistry = (FlowerRegistry) FlowerManager.flowerRegistry;
						Flower flower = flowerRegistry.getRandomPlantableFlower(FlowerManager.FlowerTypeVanilla, world.field_73012_v);
						func_175811_a(world, flower.getBlockState(), j, i, k, box);
					}
				}
			}
		}
	}

	private void buildApiaries(World world, StructureBoundingBox box) {
		populateApiary(world, box, new BlockPos(3, 1, 8));
		populateApiary(world, box, new BlockPos(6, 1, 8));
	}

	private void populateApiary(World world, StructureBoundingBox box, BlockPos pos) {
		int xCoord = func_74865_a(pos.func_177958_n(), pos.func_177952_p());
		int yCoord = func_74862_a(pos.func_177956_o());
		int zCoord = func_74873_b(pos.func_177958_n(), pos.func_177952_p());
		BlockPos posNew = new BlockPos(xCoord, yCoord, zCoord);

		if (!box.func_175898_b(posNew)) {
			return;
		}

		IBlockState blockState = world.func_180495_p(posNew);
		if (PluginApiculture.blocks.beeHouse == blockState.func_177230_c() || !world.func_175667_e(posNew.func_177977_b())) {
			return;
		}

		IBlockState beeHouseDefaultState = PluginApiculture.blocks.beeHouse.func_176223_P();
		world.func_180501_a(posNew, beeHouseDefaultState, Constants.FLAG_BLOCK_SYNC);

		TileBeeHouse beeHouse = TileUtil.getTile(world, posNew, TileBeeHouse.class);
		if (beeHouse == null) {
			return;
		}

		ItemStack randomVillagePrincess = getRandomVillageBeeStack(world, posNew, EnumBeeType.PRINCESS);
		beeHouse.getBeeInventory().setQueen(randomVillagePrincess);

		ItemStack randomVillageDrone = getRandomVillageBeeStack(world, posNew, EnumBeeType.DRONE);
		beeHouse.getBeeInventory().setDrone(randomVillageDrone);
	}

	private static ItemStack getRandomVillageBeeStack(World world, BlockPos pos, EnumBeeType beeType) {
		IBee randomVillageBee = getRandomVillageBee(world, pos);
		return BeeManager.beeRoot.getMemberStack(randomVillageBee, beeType);
	}

	private static IBee getRandomVillageBee(World world, BlockPos pos) {

		// Get current biome
		Biome biome = world.func_180494_b(pos);

		List<IBeeGenome> candidates;
		if (BeeManager.uncommonVillageBees != null && !BeeManager.uncommonVillageBees.isEmpty() && world.field_73012_v.nextDouble() < 0.2) {
			candidates = BeeManager.uncommonVillageBees;
		} else {
			candidates = BeeManager.commonVillageBees;
		}

		EnumTemperature biomeTemperature = EnumTemperature.getFromBiome(biome, world, pos);
		EnumHumidity biomeHumidity = EnumHumidity.getFromValue(biome.func_76727_i());

		// Add bees that can live in this environment
		List<IBeeGenome> valid = new ArrayList<>();
		for (IBeeGenome genome : candidates) {
			if (checkBiomeHazard(genome, biomeTemperature, biomeHumidity)) {
				valid.add(genome);
			}
		}

		// No valid ones found, return any of the common ones.
		if (valid.isEmpty()) {
			int index = world.field_73012_v.nextInt(BeeManager.commonVillageBees.size());
			IBeeGenome genome = BeeManager.commonVillageBees.get(index);
			return BeeManager.beeRoot.getBee(genome);
		}

		return BeeManager.beeRoot.getBee(valid.get(world.field_73012_v.nextInt(valid.size())));
	}

	private static boolean checkBiomeHazard(IBeeGenome genome, EnumTemperature biomeTemperature, EnumHumidity biomeHumidity) {
		IAlleleBeeSpecies species = genome.getPrimary();
		return AlleleManager.climateHelper.isWithinLimits(biomeTemperature, biomeHumidity,
				species.getTemperature(), genome.getToleranceTemp(),
				species.getHumidity(), genome.getToleranceHumid());
	}

	@Override
	protected int func_180779_c(int villagerCount, int currentVillagerProfession) {
		FMLControlledNamespacedRegistry<VillagerRegistry.VillagerProfession> registry = (FMLControlledNamespacedRegistry<VillagerRegistry.VillagerProfession>) VillagerRegistry.instance().getRegistry();

		if (villagerCount <= 0) {
			return registry.getId(PluginApiculture.villagerApiarist);
		} else if (ForestryAPI.enabledPlugins.contains(ForestryPluginUids.ARBORICULTURE)) {
			return registry.getId(PluginArboriculture.villagerArborist);
		} else {
			return currentVillagerProfession;
		}
	}

	private boolean isAirBlockAtCurrentPosition(World world, BlockPos pos, StructureBoundingBox box) {
		IBlockState blockStateFromPos = func_175807_a(world, pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p(), box);
		return blockStateFromPos.func_177230_c().isAir(blockStateFromPos, world, pos);
	}
}
