/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.core.gui;

import javax.annotation.Nonnull;
import java.awt.Rectangle;
import java.io.IOException;
import java.util.List;

import net.minecraft.client.gui.FontRenderer;
import net.minecraft.client.gui.inventory.GuiContainer;
import net.minecraft.client.renderer.GlStateManager;
import net.minecraft.client.renderer.RenderHelper;
import net.minecraft.entity.player.InventoryPlayer;
import net.minecraft.inventory.Container;
import net.minecraft.inventory.IInventory;
import net.minecraft.inventory.Slot;
import net.minecraft.util.ResourceLocation;

import forestry.api.core.IErrorLogicSource;
import forestry.api.core.IErrorSource;
import forestry.core.access.IRestrictedAccess;
import forestry.core.config.Config;
import forestry.core.gui.ledgers.ClimateLedger;
import forestry.core.gui.ledgers.HintLedger;
import forestry.core.gui.ledgers.LedgerManager;
import forestry.core.gui.ledgers.OwnerLedger;
import forestry.core.gui.ledgers.PowerLedger;
import forestry.core.gui.widgets.WidgetManager;
import forestry.core.proxy.Proxies;
import forestry.core.render.FontColour;
import forestry.core.render.ForestryResource;
import forestry.core.tiles.IClimatised;
import forestry.core.tiles.IPowerHandler;

public abstract class GuiForestry<C extends Container, I extends IInventory> extends GuiContainer {
	protected final I inventory;
	protected final C container;

	public final ResourceLocation textureFile;
	protected final WidgetManager widgetManager;
	protected LedgerManager ledgerManager;
	protected TextLayoutHelper textLayout;
	protected FontColour fontColor;

	protected GuiForestry(String texture, C container, I inventory) {
		this(new ForestryResource(texture), container, inventory);
	}

	protected GuiForestry(ResourceLocation texture, C container, I inventory) {
		super(container);

		this.widgetManager = new WidgetManager(this);

		this.textureFile = texture;

		this.inventory = inventory;
		this.container = container;

		this.fontColor = new FontColour(Proxies.render.getSelectedTexturePack());
		this.textLayout = new TextLayoutHelper(this, this.fontColor);
	}

	/* LEDGERS */
	@Override
	public void func_73866_w_() {
		super.func_73866_w_();

		int maxLedgerWidth = (this.field_146294_l - this.field_146999_f) / 2;
		this.ledgerManager = new LedgerManager(this, maxLedgerWidth);

		addLedgers();
	}

	protected void addLedgers() {
		if (inventory instanceof IErrorSource) {
			ledgerManager.add((IErrorSource) inventory);
		}

		if (inventory instanceof IErrorLogicSource) {
			IErrorLogicSource errorLogicSource = (IErrorLogicSource) inventory;
			ledgerManager.add(errorLogicSource.getErrorLogic());
		}

		if (inventory instanceof IClimatised) {
			ledgerManager.add(new ClimateLedger(ledgerManager, (IClimatised) inventory));
		}

		if (Config.enableEnergyStat && inventory instanceof IPowerHandler && ((IPowerHandler) inventory).getEnergyManager().getMaxEnergyStored() > 0) {
			ledgerManager.add(new PowerLedger(ledgerManager, (IPowerHandler) inventory));
		}

		if (Config.enableHints && inventory instanceof IHintSource) {
			IHintSource hintSource = (IHintSource) inventory;
			List<String> hints = hintSource.getHints();
			if (hints != null && !hints.isEmpty()) {
				ledgerManager.add(new HintLedger(ledgerManager, hintSource));
			}
		}

		if (inventory instanceof IRestrictedAccess) {
			ledgerManager.add(new OwnerLedger(ledgerManager, (IRestrictedAccess) inventory));
		}
	}

	@Override
	public void func_146281_b() {
		super.func_146281_b();
		ledgerManager.onGuiClosed();
	}

	public FontColour getFontColor() {
		return fontColor;
	}

	public FontRenderer getFontRenderer() {
		return field_146289_q;
	}

	@Override
	protected void func_73864_a(int xPos, int yPos, int mouseButton) throws IOException {
		super.func_73864_a(xPos, yPos, mouseButton);

		// / Handle ledger clicks
		ledgerManager.handleMouseClicked(xPos, yPos, mouseButton);
		widgetManager.handleMouseClicked(xPos, yPos, mouseButton);
	}
	
	@Override
	protected void func_146286_b(int mouseX, int mouseY, int state) {
		super.func_146286_b(mouseX, mouseY, state);
		
		widgetManager.handleMouseRelease(mouseX, mouseY, state);
	}

	@Override
	protected void func_146273_a(int mouseX, int mouseY, int mouseButton, long time) {

		widgetManager.handleMouseMove(mouseX, mouseY, mouseButton, time);

		super.func_146273_a(mouseX, mouseY, mouseButton, time);
	}

	protected Slot getSlotAtPosition(int par1, int par2) {
		for (int k = 0; k < this.field_147002_h.field_75151_b.size(); ++k) {
			Slot slot = this.field_147002_h.field_75151_b.get(k);

			if (isMouseOverSlot(slot, par1, par2)) {
				return slot;
			}
		}

		return null;
	}

	private boolean isMouseOverSlot(Slot par1Slot, int par2, int par3) {
		return func_146978_c(par1Slot.field_75223_e, par1Slot.field_75221_f, 16, 16, par2, par3);
	}

	@Override
	protected void func_146979_b(int mouseX, int mouseY) {
		ledgerManager.drawTooltips(mouseX, mouseY);

		InventoryPlayer playerInv = field_146297_k.field_71439_g.field_71071_by;

		if (playerInv.func_70445_o() == null) {
			GuiUtil.drawToolTips(this, widgetManager.getWidgets(), mouseX, mouseY);
			GuiUtil.drawToolTips(this, field_146292_n, mouseX, mouseY);
			GuiUtil.drawToolTips(this, field_147002_h.field_75151_b, mouseX, mouseY);
		}
	}

	@Override
	protected void func_146976_a(float f, int mouseX, int mouseY) {
		bindTexture(textureFile);

		int x = (field_146294_l - field_146999_f) / 2;
		int y = (field_146295_m - field_147000_g) / 2;
		func_73729_b(x, y, 0, 0, field_146999_f, field_147000_g);

		RenderHelper.func_74520_c();
		GlStateManager.func_179140_f();
		GlStateManager.func_179091_B();
		GlStateManager.func_179131_c(1.0F, 1.0F, 1.0F, 1.0F);
		GlStateManager.func_179094_E();
		{
			GlStateManager.func_179109_b(field_147003_i, field_147009_r, 0.0F);
			drawWidgets();
		}
		GlStateManager.func_179121_F();

		bindTexture(textureFile);
	}

	protected void drawWidgets() {
		ledgerManager.drawLedgers();
		widgetManager.drawWidgets();
	}

	protected void bindTexture(ResourceLocation texturePath) {
		GlStateManager.func_179131_c(1.0F, 1.0F, 1.0F, 1.0F);
		Proxies.render.bindTexture(texturePath);
	}

	public void setZLevel(float level) {
		this.field_73735_i = level;
	}

	public int getSizeX() {
		return field_146999_f;
	}

	public int getSizeY() {
		return field_147000_g;
	}

	public int getGuiLeft() {
		return field_147003_i;
	}

	public int getGuiTop() {
		return field_147009_r;
	}

	@Override
	public void func_73733_a(int par1, int par2, int par3, int par4, int par5, int par6) {
		super.func_73733_a(par1, par2, par3, par4, par5, par6);
	}

	@Nonnull
	public List<Rectangle> getExtraGuiAreas() {
		return ledgerManager.getLedgerAreas();
	}
}
