/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.arboriculture;

import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.common.base.Preconditions;
import forestry.api.arboriculture.EnumVanillaWoodType;
import forestry.api.arboriculture.IWoodAccess;
import forestry.api.arboriculture.IWoodType;
import forestry.api.arboriculture.WoodBlockKind;
import forestry.arboriculture.blocks.BlockArbDoor;
import forestry.arboriculture.blocks.BlockForestryFence;
import forestry.arboriculture.blocks.BlockForestryFenceGate;
import forestry.arboriculture.blocks.BlockForestryLog;
import forestry.arboriculture.blocks.BlockForestryPlanks;
import forestry.arboriculture.blocks.BlockForestrySlab;
import forestry.arboriculture.blocks.BlockForestryStairs;
import forestry.arboriculture.blocks.PropertyWoodType;
import net.minecraft.block.Block;
import net.minecraft.block.BlockNewLog;
import net.minecraft.block.BlockOldLog;
import net.minecraft.block.BlockPlanks;
import net.minecraft.block.BlockWoodSlab;
import net.minecraft.block.state.IBlockState;
import net.minecraft.init.Blocks;
import net.minecraft.init.Items;
import net.minecraft.item.ItemStack;

public class WoodAccess implements IWoodAccess {
	@Nullable
	private static WoodAccess INSTANCE;

	public static WoodAccess getInstance() {
		if (INSTANCE == null) {
			INSTANCE = new WoodAccess();
		}
		return INSTANCE;
	}

	private final Map<WoodBlockKind, WoodMap> woodMaps = new EnumMap<>(WoodBlockKind.class);
	private final List<IWoodType> registeredWoodTypes = new ArrayList<>();

	private WoodAccess() {
		for (WoodBlockKind woodBlockKind : WoodBlockKind.values()) {
			woodMaps.put(woodBlockKind, new WoodMap(woodBlockKind));
		}
		registerVanilla();
	}

	public void registerLogs(List<? extends BlockForestryLog> blocks) {
		for (BlockForestryLog block : blocks) {
			//noinspection unchecked
			registerWithVariants(block, WoodBlockKind.LOG, block.getVariant());
		}
	}

	public void registerPlanks(List<? extends BlockForestryPlanks> blocks) {
		for (BlockForestryPlanks block : blocks) {
			//noinspection unchecked
			registerWithVariants(block, WoodBlockKind.PLANKS, block.getVariant());
		}
	}

	public void registerSlabs(List<? extends BlockForestrySlab> blocks) {
		for (BlockForestrySlab block : blocks) {
			//noinspection unchecked
			registerWithVariants(block, WoodBlockKind.SLAB, block.getVariant());
		}
	}

	public void registerFences(List<? extends BlockForestryFence> blocks) {
		for (BlockForestryFence block : blocks) {
			//noinspection unchecked
			registerWithVariants(block, WoodBlockKind.FENCE, block.getVariant());
		}
	}

	public void registerFenceGates(List<BlockForestryFenceGate> blocks) {
		for (BlockForestryFenceGate block : blocks) {
			registerWithoutVariants(block, WoodBlockKind.FENCE_GATE);
		}
	}

	public void registerStairs(List<? extends BlockForestryStairs> blocks) {
		for (BlockForestryStairs block : blocks) {
			registerWithoutVariants(block, WoodBlockKind.STAIRS);
		}
	}

	public void registerDoors(List<BlockArbDoor> blocks) {
		for (BlockArbDoor block : blocks) {
			registerWithoutVariants(block, WoodBlockKind.DOOR);
		}
	}

	private void registerVanilla() {
		IBlockState defaultLogState = Blocks.field_150364_r.func_176223_P();
		List<EnumVanillaWoodType> oldLogTypes = Arrays.asList(EnumVanillaWoodType.OAK, EnumVanillaWoodType.SPRUCE, EnumVanillaWoodType.BIRCH, EnumVanillaWoodType.JUNGLE);
		for (EnumVanillaWoodType woodType : oldLogTypes) {
			BlockPlanks.EnumType vanillaType = woodType.getVanillaType();
			ItemStack itemStack = new ItemStack(Blocks.field_150364_r, 1, vanillaType.func_176839_a());
			IBlockState blockState = defaultLogState.func_177226_a(BlockOldLog.field_176301_b, vanillaType);
			register(woodType, WoodBlockKind.LOG, false, blockState, itemStack);
		}

		IBlockState defaultLog2State = Blocks.field_150363_s.func_176223_P();
		List<EnumVanillaWoodType> newLogTypes = Arrays.asList(EnumVanillaWoodType.ACACIA, EnumVanillaWoodType.DARK_OAK);
		for (EnumVanillaWoodType woodType : newLogTypes) {
			BlockPlanks.EnumType vanillaType = woodType.getVanillaType();
			ItemStack itemStack = new ItemStack(Blocks.field_150363_s, 1, vanillaType.func_176839_a() - 4);
			IBlockState blockState = defaultLog2State.func_177226_a(BlockNewLog.field_176300_b, vanillaType);
			register(woodType, WoodBlockKind.LOG, false, blockState, itemStack);
		}

		IBlockState defaultPlanksState = Blocks.field_150344_f.func_176223_P();
		IBlockState defaultSlabState = Blocks.field_150376_bx.func_176223_P();
		for (EnumVanillaWoodType woodType : EnumVanillaWoodType.VALUES) {
			BlockPlanks.EnumType vanillaType = woodType.getVanillaType();

			ItemStack plankStack = new ItemStack(Blocks.field_150344_f, 1, vanillaType.func_176839_a());
			IBlockState plankState = defaultPlanksState.func_177226_a(BlockPlanks.field_176383_a, vanillaType);
			register(woodType, WoodBlockKind.PLANKS, false, plankState, plankStack);

			ItemStack slabStack = new ItemStack(Blocks.field_150376_bx, 1, vanillaType.func_176839_a());
			IBlockState slabState = defaultSlabState.func_177226_a(BlockWoodSlab.field_176557_b, vanillaType);
			register(woodType, WoodBlockKind.SLAB, false, slabState, slabStack);
		}

		register(EnumVanillaWoodType.OAK, WoodBlockKind.FENCE, false, Blocks.field_180407_aO.func_176223_P(), new ItemStack(Blocks.field_180407_aO));
		register(EnumVanillaWoodType.SPRUCE, WoodBlockKind.FENCE, false, Blocks.field_180408_aP.func_176223_P(), new ItemStack(Blocks.field_180408_aP));
		register(EnumVanillaWoodType.BIRCH, WoodBlockKind.FENCE, false, Blocks.field_180404_aQ.func_176223_P(), new ItemStack(Blocks.field_180404_aQ));
		register(EnumVanillaWoodType.JUNGLE, WoodBlockKind.FENCE, false, Blocks.field_180403_aR.func_176223_P(), new ItemStack(Blocks.field_180403_aR));
		register(EnumVanillaWoodType.ACACIA, WoodBlockKind.FENCE, false, Blocks.field_180405_aT.func_176223_P(), new ItemStack(Blocks.field_180405_aT));
		register(EnumVanillaWoodType.DARK_OAK, WoodBlockKind.FENCE, false, Blocks.field_180406_aS.func_176223_P(), new ItemStack(Blocks.field_180406_aS));

		register(EnumVanillaWoodType.OAK, WoodBlockKind.FENCE_GATE, false, Blocks.field_180390_bo.func_176223_P(), new ItemStack(Blocks.field_180390_bo));
		register(EnumVanillaWoodType.SPRUCE, WoodBlockKind.FENCE_GATE, false, Blocks.field_180391_bp.func_176223_P(), new ItemStack(Blocks.field_180391_bp));
		register(EnumVanillaWoodType.BIRCH, WoodBlockKind.FENCE_GATE, false, Blocks.field_180392_bq.func_176223_P(), new ItemStack(Blocks.field_180392_bq));
		register(EnumVanillaWoodType.JUNGLE, WoodBlockKind.FENCE_GATE, false, Blocks.field_180386_br.func_176223_P(), new ItemStack(Blocks.field_180386_br));
		register(EnumVanillaWoodType.ACACIA, WoodBlockKind.FENCE_GATE, false, Blocks.field_180387_bt.func_176223_P(), new ItemStack(Blocks.field_180387_bt));
		register(EnumVanillaWoodType.DARK_OAK, WoodBlockKind.FENCE_GATE, false, Blocks.field_180385_bs.func_176223_P(), new ItemStack(Blocks.field_180385_bs));

		register(EnumVanillaWoodType.OAK, WoodBlockKind.STAIRS, false, Blocks.field_150476_ad.func_176223_P(), new ItemStack(Blocks.field_150476_ad));
		register(EnumVanillaWoodType.SPRUCE, WoodBlockKind.STAIRS, false, Blocks.field_150485_bF.func_176223_P(), new ItemStack(Blocks.field_150485_bF));
		register(EnumVanillaWoodType.BIRCH, WoodBlockKind.STAIRS, false, Blocks.field_150487_bG.func_176223_P(), new ItemStack(Blocks.field_150487_bG));
		register(EnumVanillaWoodType.JUNGLE, WoodBlockKind.STAIRS, false, Blocks.field_150481_bH.func_176223_P(), new ItemStack(Blocks.field_150481_bH));
		register(EnumVanillaWoodType.ACACIA, WoodBlockKind.STAIRS, false, Blocks.field_150400_ck.func_176223_P(), new ItemStack(Blocks.field_150400_ck));
		register(EnumVanillaWoodType.DARK_OAK, WoodBlockKind.STAIRS, false, Blocks.field_150401_cl.func_176223_P(), new ItemStack(Blocks.field_150401_cl));

		register(EnumVanillaWoodType.OAK, WoodBlockKind.DOOR, false, Blocks.field_180413_ao.func_176223_P(), new ItemStack(Items.field_179570_aq));
		register(EnumVanillaWoodType.SPRUCE, WoodBlockKind.DOOR, false, Blocks.field_180414_ap.func_176223_P(), new ItemStack(Items.field_179569_ar));
		register(EnumVanillaWoodType.BIRCH, WoodBlockKind.DOOR, false, Blocks.field_180412_aq.func_176223_P(), new ItemStack(Items.field_179568_as));
		register(EnumVanillaWoodType.JUNGLE, WoodBlockKind.DOOR, false, Blocks.field_180411_ar.func_176223_P(), new ItemStack(Items.field_179567_at));
		register(EnumVanillaWoodType.ACACIA, WoodBlockKind.DOOR, false, Blocks.field_180410_as.func_176223_P(), new ItemStack(Items.field_179572_au));
		register(EnumVanillaWoodType.DARK_OAK, WoodBlockKind.DOOR, false, Blocks.field_180409_at.func_176223_P(), new ItemStack(Items.field_179571_av));
	}

	private <T extends Block & IWoodTyped, V extends Enum<V> & IWoodType> void registerWithVariants(T woodTyped, WoodBlockKind woodBlockKind, PropertyWoodType<V> property) {
		boolean fireproof = woodTyped.isFireproof();

		for (V value : property.func_177700_c()) {
			IBlockState blockState = woodTyped.func_176223_P().func_177226_a(property, value);
			int meta = woodTyped.func_176201_c(blockState);
			IWoodType woodType = woodTyped.getWoodType(meta);
			ItemStack itemStack = new ItemStack(woodTyped, 1, meta);
			if (!(woodType instanceof EnumVanillaWoodType)) {
				PluginArboriculture.proxy.registerWoodModel(woodTyped, true);
			}
			register(woodType, woodBlockKind, fireproof, blockState, itemStack);
		}
	}

	/**
	 * Register wood blocks that have no variant property
	 */
	private <T extends Block & IWoodTyped> void registerWithoutVariants(T woodTyped, WoodBlockKind woodBlockKind) {
		boolean fireproof = woodTyped.isFireproof();
		IBlockState blockState = woodTyped.func_176223_P();
		IWoodType woodType = woodTyped.getWoodType(0);
		ItemStack itemStack = new ItemStack(woodTyped);
		if (!(woodType instanceof EnumVanillaWoodType)) {
			PluginArboriculture.proxy.registerWoodModel(woodTyped, false);
		}
		register(woodType, woodBlockKind, fireproof, blockState, itemStack);
	}

	@Override
	public void register(IWoodType woodType, WoodBlockKind woodBlockKind, boolean fireproof, IBlockState blockState, ItemStack itemStack) {
		if (woodBlockKind == WoodBlockKind.DOOR) {
			fireproof = true;
		}
		Preconditions.checkArgument(!itemStack.func_190926_b(), "Empty Itemstack");
		WoodMap woodMap = woodMaps.get(woodBlockKind);
		if (!registeredWoodTypes.contains(woodType)) {
			registeredWoodTypes.add(woodType);
		}
		woodMap.getItem(fireproof).put(woodType, itemStack);
		woodMap.getBlock(fireproof).put(woodType, blockState);
	}

	@Override
	public ItemStack getStack(IWoodType woodType, WoodBlockKind woodBlockKind, boolean fireproof) {
		if (woodBlockKind == WoodBlockKind.DOOR) {
			fireproof = true;
		}
		WoodMap woodMap = woodMaps.get(woodBlockKind);
		ItemStack itemStack = woodMap.getItem(fireproof).get(woodType);
		if (itemStack == null) {
			String errMessage = String.format("No stack found for %s %s %s", woodType, woodMap.getName(), fireproof ? "fireproof" : "non-fireproof");
			throw new IllegalStateException(errMessage);
		}
		return itemStack.func_77946_l();
	}

	@Override
	public IBlockState getBlock(IWoodType woodType, WoodBlockKind woodBlockKind, boolean fireproof) {
		if (woodBlockKind == WoodBlockKind.DOOR) {
			fireproof = true;
		}
		WoodMap woodMap = woodMaps.get(woodBlockKind);
		IBlockState blockState = woodMap.getBlock(fireproof).get(woodType);
		if (blockState == null) {
			String errMessage = String.format("No block found for %s %s %s", woodType, woodMap.getName(), fireproof ? "fireproof" : "non-fireproof");
			throw new IllegalStateException(errMessage);
		}
		return blockState;
	}

	@Override
	public List<IWoodType> getRegisteredWoodTypes() {
		return registeredWoodTypes;
	}

	private static class WoodMap {
		private final Map<IWoodType, ItemStack> normalItems = new HashMap<>();
		private final Map<IWoodType, ItemStack> fireproofItems = new HashMap<>();
		private final Map<IWoodType, IBlockState> normalBlocks = new HashMap<>();
		private final Map<IWoodType, IBlockState> fireproofBlocks = new HashMap<>();
		private final WoodBlockKind woodBlockKind;

		public WoodMap(WoodBlockKind woodBlockKind) {
			this.woodBlockKind = woodBlockKind;
		}

		public String getName() {
			return woodBlockKind.name();
		}

		public Map<IWoodType, ItemStack> getItem(boolean fireproof) {
			return fireproof ? this.fireproofItems : this.normalItems;
		}

		public Map<IWoodType, IBlockState> getBlock(boolean fireproof) {
			return fireproof ? this.fireproofBlocks : this.normalBlocks;
		}
	}
}
