/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.core.models;

import javax.annotation.Nullable;
import java.util.Collections;
import java.util.List;

import com.google.common.base.Preconditions;
import forestry.api.core.IModelBaker;
import forestry.api.core.IModelBakerModel;
import forestry.core.blocks.properties.UnlistedBlockAccess;
import forestry.core.blocks.properties.UnlistedBlockPos;
import forestry.core.models.baker.ModelBaker;
import net.minecraft.block.Block;
import net.minecraft.block.state.IBlockState;
import net.minecraft.client.Minecraft;
import net.minecraft.client.renderer.block.model.BakedQuad;
import net.minecraft.client.renderer.block.model.IBakedModel;
import net.minecraft.client.renderer.block.model.ItemCameraTransforms;
import net.minecraft.client.renderer.block.model.ItemOverrideList;
import net.minecraft.client.renderer.texture.TextureAtlasSprite;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.item.ItemStack;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;
import net.minecraftforge.common.property.IExtendedBlockState;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

@SideOnly(Side.CLIENT)
public abstract class ModelBlockDefault<B extends Block, K> implements IBakedModel {
	@Nullable
	private ItemOverrideList overrideList;

	protected final Class<B> blockClass;

	@Nullable
	protected IModelBakerModel blockModel;
	@Nullable
	protected IModelBakerModel itemModel;

	protected ModelBlockDefault(Class<B> blockClass) {
		this.blockClass = blockClass;
	}

	protected IBakedModel bakeModel(IBlockState state, K key, B block) {
		IModelBaker baker = new ModelBaker();

		if (state instanceof IExtendedBlockState) {
			IExtendedBlockState stateExtended = (IExtendedBlockState) state;
			IBlockAccess world = stateExtended.getValue(UnlistedBlockAccess.BLOCKACCESS);
			BlockPos pos = stateExtended.getValue(UnlistedBlockPos.POS);
		}

		bakeBlock(block, key, baker, false);

		blockModel = baker.bakeModel(false);
		onCreateModel(blockModel);
		return blockModel;
	}

	protected IBakedModel getModel(IBlockState state) {
		Preconditions.checkArgument(blockClass.isInstance(state.func_177230_c()));

		K worldKey = getWorldKey(state);
		B block = blockClass.cast(state.func_177230_c());
		return bakeModel(state, worldKey, block);
	}

	protected IBakedModel bakeModel(ItemStack stack, World world, K key) {
		IModelBaker baker = new ModelBaker();
		Block block = Block.func_149634_a(stack.func_77973_b());
		Preconditions.checkArgument(blockClass.isInstance(block));
		B bBlock = blockClass.cast(block);
		bakeBlock(bBlock, key, baker, true);

		return itemModel = baker.bakeModel(true);
	}

	protected IBakedModel getModel(ItemStack stack, World world) {
		return bakeModel(stack, world, getInventoryKey(stack));
	}

	@Override
	public List<BakedQuad> func_188616_a(@Nullable IBlockState state, @Nullable EnumFacing side, long rand) {
		Preconditions.checkNotNull(state);
		IBakedModel model = getModel(state);
		return model.func_188616_a(state, side, rand);
	}

	protected void onCreateModel(IModelBakerModel model) {
		model.setAmbientOcclusion(true);
	}

	@Override
	public boolean func_177555_b() {
		return (itemModel != null || blockModel != null) &&
				(blockModel != null ? blockModel.func_177555_b() : itemModel.func_177555_b());
	}

	@Override
	public boolean func_177556_c() {
		return itemModel != null && itemModel.func_177556_c();
	}

	@Override
	public boolean func_188618_c() {
		return (itemModel != null || blockModel != null) &&
				(blockModel != null ? blockModel.func_188618_c() : itemModel.func_188618_c());
	}

	@Override
	public TextureAtlasSprite func_177554_e() {
		if (blockModel != null) {
			return blockModel.func_177554_e();
		}
		return Minecraft.func_71410_x().func_147117_R().func_174944_f();
	}


	@Override
	public ItemCameraTransforms func_177552_f() {
		if (itemModel == null) {
			return ItemCameraTransforms.field_178357_a;
		}
		return itemModel.func_177552_f();
	}

	protected ItemOverrideList createOverrides() {
		return new DefaultItemOverrideList();
	}

	@Override
	public ItemOverrideList func_188617_f() {
		if (overrideList == null) {
			overrideList = createOverrides();
		}
		return overrideList;
	}

	protected abstract K getInventoryKey(ItemStack stack);

	protected abstract K getWorldKey(IBlockState state);

	protected abstract void bakeBlock(B block, K key, IModelBaker baker, boolean inventory);

	private class DefaultItemOverrideList extends ItemOverrideList {
		public DefaultItemOverrideList() {
			super(Collections.emptyList());
		}

		@Override
		public IBakedModel handleItemState(IBakedModel originalModel, ItemStack stack, @Nullable World world, @Nullable EntityLivingBase entity) {
			if (world == null) {
				world = Minecraft.func_71410_x().field_71441_e;
			}
			return getModel(stack, world);
		}
	}
}
