/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.apiculture.entities;

import java.io.IOException;
import java.util.Optional;
import java.util.Random;

import com.mojang.authlib.GameProfile;
import forestry.api.apiculture.BeeManager;
import forestry.api.apiculture.IBeeHousing;
import forestry.api.apiculture.IBeekeepingLogic;
import forestry.api.core.EnumHumidity;
import forestry.api.core.EnumTemperature;
import forestry.api.core.ForestryAPI;
import forestry.api.core.IErrorLogic;
import forestry.apiculture.gui.IGuiBeeHousingDelegate;
import forestry.apiculture.network.packets.PacketBeeLogicEntityRequest;
import forestry.apiculture.tiles.TileBeeHousingBase;
import forestry.core.entities.EntityMinecartContainerForestry;
import forestry.core.network.IForestryPacketServer;
import forestry.core.network.IStreamableGui;
import forestry.core.network.PacketBufferForestry;
import forestry.core.owner.GameProfileDataSerializer;
import forestry.core.owner.IOwnedTile;
import forestry.core.owner.IOwnerHandler;
import forestry.core.owner.OwnerHandler;
import forestry.core.tiles.IClimatised;
import forestry.core.utils.ClimateUtil;
import forestry.core.utils.NetworkUtil;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.datasync.DataParameter;
import net.minecraft.network.datasync.EntityDataManager;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.Vec3d;
import net.minecraft.world.World;
import net.minecraft.world.biome.Biome;

public abstract class EntityMinecartBeeHousingBase extends EntityMinecartContainerForestry implements IBeeHousing, IOwnedTile, IGuiBeeHousingDelegate, IClimatised, IStreamableGui {
	private static final DataParameter<Optional<GameProfile>> OWNER = EntityDataManager.func_187226_a(EntityMinecartBeeHousingBase.class, GameProfileDataSerializer.INSTANCE);

	private static final Random random = new Random();
	private static final int beeFXInterval = 4;
	private static final int pollenFXInterval = 50;

	private final int beeFXTime = random.nextInt(beeFXInterval);
	private final int pollenFXTime = random.nextInt(pollenFXInterval);

	private final IBeekeepingLogic beeLogic = BeeManager.beeRoot.createBeekeepingLogic(this);
	private final IErrorLogic errorLogic = ForestryAPI.errorStateRegistry.createErrorLogic();
	private final OwnerHandler ownerHandler = new OwnerHandler() {
		@Override
		public void setOwner(GameProfile owner) {
			super.setOwner(owner);
			field_70180_af.func_187227_b(OWNER, Optional.of(owner));
		}

		@Override
		public GameProfile getOwner() {
			Optional<GameProfile> gameProfileOptional = field_70180_af.func_187225_a(OWNER);
			return gameProfileOptional.orElse(null);
		}
	};

	// CLIENT
	private int breedingProgressPercent = 0;
	private boolean needsActiveUpdate = true;

	@SuppressWarnings("unused")
	public EntityMinecartBeeHousingBase(World world) {
		super(world);
	}

	public EntityMinecartBeeHousingBase(World world, double posX, double posY, double posZ) {
		super(world, posX, posY, posZ);
	}

	@Override
	protected void func_70088_a() {
		super.func_70088_a();
		this.field_70180_af.func_187214_a(OWNER, Optional.empty());
	}

	/* IOwnedTile */
	@Override
	public IOwnerHandler getOwnerHandler() {
		return ownerHandler;
	}

	/* IBeeHousing */
	@Override
	public IBeekeepingLogic getBeekeepingLogic() {
		return beeLogic;
	}

	@Override
	public EnumTemperature getTemperature() {
		return EnumTemperature.getFromBiome(getBiome(), field_70170_p, func_180425_c());
	}

	@Override
	public EnumHumidity getHumidity() {
		return EnumHumidity.getFromValue(getBiome().func_76727_i());
	}

	@Override
	public float getExactTemperature() {
		return ClimateUtil.getTemperature(field_70170_p, func_180425_c());
	}

	@Override
	public float getExactHumidity() {
		return ClimateUtil.getHumidity(field_70170_p, func_180425_c());
	}

	@Override
	public int getBlockLightValue() {
		return field_70170_p.func_175671_l(func_180425_c().func_177984_a());
	}

	@Override
	public boolean canBlockSeeTheSky() {
		return field_70170_p.func_175710_j(func_180425_c().func_177984_a());
	}

	@Override
	public boolean isRaining() {
		return field_70170_p.func_175727_C(func_180425_c().func_177984_a());
	}

	@Override
	public World getWorldObj() {
		return field_70170_p;
	}

	@Override
	public Biome getBiome() {
		return field_70170_p.func_180494_b(func_180425_c());
	}

	@Override
	public GameProfile getOwner() {
		return getOwnerHandler().getOwner();
	}

	@Override
	public IErrorLogic getErrorLogic() {
		return errorLogic;
	}

	@Override
	public BlockPos getCoordinates() {
		return func_180425_c();
	}

	@Override
	public Vec3d getBeeFXCoordinates() {
		return new Vec3d(field_70165_t, field_70163_u + 0.25, field_70161_v);
	}

	@Override
	public void writeGuiData(PacketBufferForestry data) {
		data.func_150787_b(beeLogic.getBeeProgressPercent());
	}

	@Override
	public void readGuiData(PacketBufferForestry data) throws IOException {
		breedingProgressPercent = data.func_150792_a();
	}

	@Override
	public int getHealthScaled(int i) {
		return breedingProgressPercent * i / 100;
	}

	@Override
	public void func_70071_h_() {
		super.func_70071_h_();
		if (!field_70170_p.field_72995_K) {
			if (beeLogic.canWork()) {
				beeLogic.doWork();
			}
		} else {
			if (needsActiveUpdate) {
				IForestryPacketServer packet = new PacketBeeLogicEntityRequest(this);
				NetworkUtil.sendToServer(packet);
				needsActiveUpdate = false;
			}

			if (beeLogic.canDoBeeFX()) {
				if (field_70170_p.func_82737_E() % beeFXInterval == beeFXTime) {
					beeLogic.doBeeFX();
				}

				if (field_70170_p.func_82737_E() % pollenFXInterval == pollenFXTime) {
					TileBeeHousingBase.doPollenFX(field_70170_p, field_70165_t - 0.5, field_70163_u - 0.1, field_70161_v - 0.5);
				}
			}
		}
	}

	@Override
	protected void func_70037_a(NBTTagCompound nbtTagCompound) {
		super.func_70037_a(nbtTagCompound);
		beeLogic.readFromNBT(nbtTagCompound);
		ownerHandler.readFromNBT(nbtTagCompound);
	}

	@Override
	protected void func_70014_b(NBTTagCompound nbtTagCompound) {
		super.func_70014_b(nbtTagCompound);
		beeLogic.writeToNBT(nbtTagCompound);
		ownerHandler.writeToNBT(nbtTagCompound);
	}
}
