/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.farming.logic;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.Stack;

import net.minecraft.block.Block;
import net.minecraft.block.BlockOldLog;
import net.minecraft.block.BlockPlanks;
import net.minecraft.block.state.IBlockState;
import net.minecraft.init.Blocks;
import net.minecraft.init.Items;
import net.minecraft.item.ItemStack;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.NonNullList;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;

import forestry.api.farming.FarmDirection;
import forestry.api.farming.ICrop;
import forestry.api.farming.IFarmHousing;
import forestry.api.farming.IFarmProperties;
import forestry.api.farming.IFarmable;
import forestry.api.farming.ISoil;
import forestry.core.utils.BlockUtil;
import forestry.farming.logic.farmables.FarmableCocoa;

public class FarmLogicCocoa extends FarmLogicSoil {
	private static final int[] LAYOUT_POSITIONS = new int[]{4, 1, 3, 0, 2};
	private final IFarmable cocoa = new FarmableCocoa();

	public FarmLogicCocoa(IFarmProperties properties, boolean isManual) {
		super(properties, isManual);
	}

	@Override
	public ItemStack getIconItemStack() {
		return new ItemStack(Items.field_151100_aR, 1, 3);
	}

	@Override
	public String getUnlocalizedName() {
		return "for.farm.cocoa";
	}

	@Override
	public int getFertilizerConsumption() {
		return 120;
	}

	@Override
	public int getWaterConsumption(float hydrationModifier) {
		return (int) (20 * hydrationModifier);
	}

	@Override
	public boolean isAcceptedGermling(ItemStack itemstack) {
		return cocoa.isGermling(itemstack);
	}

	@Override
	public boolean isAcceptedWindfall(ItemStack stack) {
		return false;
	}

	@Override
	public boolean cultivate(World world, IFarmHousing farmHousing, BlockPos pos, FarmDirection direction, int extent) {
		if (maintainSoil(world, farmHousing, pos, direction, extent)) {
			return true;
		}
		BlockPos position = farmHousing.getValidPosition(direction, pos, extent, pos.func_177984_a());
		boolean result = tryPlantingCocoa(world, farmHousing, position, direction);

		farmHousing.increaseExtent(direction, pos, extent);

		return result;
	}

	protected boolean maintainSoil(World world, IFarmHousing farmHousing, BlockPos pos, FarmDirection direction, int extent) {
		if (!farmHousing.canPlantSoil(isManual)) {
			return false;
		}
		BlockPos cornerPos = farmHousing.getFarmCorner(direction);
		int distance = getDistanceValue(direction.getFacing().func_176732_a(EnumFacing.Axis.Y), cornerPos, pos) - 1;
		int layoutExtent = LAYOUT_POSITIONS[distance % LAYOUT_POSITIONS.length];
		for (ISoil soil : getSoils()) {
			NonNullList<ItemStack> resources = NonNullList.func_191196_a();
			resources.add(soil.getResource());

			for (int i = 0; i < extent; i++) {
				BlockPos position = translateWithOffset(pos, direction, i);
				if (!world.func_175667_e(position)) {
					break;
				}

				if (!isValidPosition(direction, position, pos, layoutExtent)
					|| !farmHousing.getFarmInventory().hasResources(resources)) {
					continue;
				}

				BlockPos platformPosition = position.func_177977_b();
				if (!farmHousing.isValidPlatform(world, platformPosition)) {
					break;
				}

				for (int z = 0; z < 3; z++) {
					BlockPos location = position.func_177981_b(z);

					IBlockState state = world.func_180495_p(location);
					if (z == 0 && !world.func_175623_d(location)
						|| z > 0 && isAcceptedSoil(state)
						|| !BlockUtil.isBreakableBlock(state, world, pos)) {
						continue;
					}

					if (!BlockUtil.isReplaceableBlock(state, world, location)) {
						BlockUtil.getBlockDrops(world, location).forEach(farmHousing::addPendingProduce);
						world.func_175698_g(location);
						return trySetSoil(world, farmHousing, location, soil.getResource(), soil.getSoilState());
					}

					if (!isManual) {
						return trySetSoil(world, farmHousing, location, soil.getResource(), soil.getSoilState());
					}
				}
			}
		}

		return false;
	}

	protected int getDistanceValue(EnumFacing facing, BlockPos posA, BlockPos posB) {
		BlockPos delta = posA.func_177973_b(posB);
		int value;
		switch (facing.func_176740_k()) {
			case X:
				value = delta.func_177958_n();
				break;
			case Y:
				value = delta.func_177956_o();
				break;
			case Z:
				value = delta.func_177952_p();
				break;
			default:
				value = 0;
		}
		return Math.abs(value);
	}

	//4, 1, 3, 0, 2
	protected boolean isValidPosition(FarmDirection direction, BlockPos pos, BlockPos logicPos, int layoutExtent) {
		int distance = getDistanceValue(direction.getFacing(), pos, logicPos);
		return (distance % LAYOUT_POSITIONS.length) == (layoutExtent);
	}

	protected boolean trySetSoil(World world, IFarmHousing farmHousing, BlockPos position, ItemStack resource, IBlockState ground) {
		NonNullList<ItemStack> resources = NonNullList.func_191196_a();
		resources.add(resource);
		if (!farmHousing.getFarmInventory().hasResources(resources)) {
			return false;
		}
		if (!BlockUtil.setBlockWithPlaceSound(world, position, ground)) {
			return false;
		}
		farmHousing.getFarmInventory().removeResources(resources);
		return true;
	}

	@Override
	public Collection<ICrop> harvest(World world, IFarmHousing housing, BlockPos pos, FarmDirection direction, int extent) {
		BlockPos position = housing.getValidPosition(direction, pos, extent, pos.func_177984_a());
		Collection<ICrop> crops = getHarvestBlocks(world, position);
		housing.increaseExtent(direction, pos, extent);

		return crops;
	}

	private boolean tryPlantingCocoa(World world, IFarmHousing farmHousing, BlockPos position, FarmDirection farmDirection) {
		BlockPos.MutableBlockPos current = new BlockPos.MutableBlockPos(position);
		IBlockState blockState = world.func_180495_p(current);
		while (isJungleTreeTrunk(blockState)) {
			for (EnumFacing direction : EnumFacing.field_176754_o) {
				BlockPos candidate = new BlockPos(current.func_177958_n() + direction.func_82601_c(), current.func_177956_o(), current.func_177952_p() + direction.func_82599_e());
				if (world.func_175667_e(candidate) && world.func_175623_d(candidate)) {
					return farmHousing.plantGermling(cocoa, world, candidate, farmDirection);
				}
			}

			current.func_189536_c(EnumFacing.UP);
			if (current.func_177956_o() - position.func_177956_o() > 1) {
				break;
			}

			blockState = world.func_180495_p(current);
		}

		return false;
	}

	private static boolean isJungleTreeTrunk(IBlockState blockState) {
		Block block = blockState.func_177230_c();
		return block == Blocks.field_150364_r && blockState.func_177229_b(BlockOldLog.field_176301_b) == BlockPlanks.EnumType.JUNGLE;
	}

	private Collection<ICrop> getHarvestBlocks(World world, BlockPos position) {

		Set<BlockPos> seen = new HashSet<>();
		Stack<ICrop> crops = new Stack<>();

		// Determine what type we want to harvest.
		IBlockState blockState = world.func_180495_p(position);
		Block block = blockState.func_177230_c();

		ICrop crop = null;
		if (!block.isWood(world, position)) {
			crop = cocoa.getCropAt(world, position, blockState);
			if (crop == null) {
				return crops;
			}
		}

		if (crop != null) {
			crops.add(crop);
		}

		List<BlockPos> candidates = processHarvestBlock(world, crops, seen, position, position);
		List<BlockPos> temp = new ArrayList<>();
		while (!candidates.isEmpty() && crops.size() < 20) {
			for (BlockPos candidate : candidates) {
				temp.addAll(processHarvestBlock(world, crops, seen, position, candidate));
			}
			candidates.clear();
			candidates.addAll(temp);
			temp.clear();
		}
		// Log.finest("Logic %s at %s/%s/%s has seen %s blocks.", getClass().getName(), position.x, position.y, position.z, seen.size());

		return crops;
	}

	private List<BlockPos> processHarvestBlock(World world, Stack<ICrop> crops, Set<BlockPos> seen, BlockPos start, BlockPos position) {
		List<BlockPos> candidates = new ArrayList<>();

		// Get additional candidates to return
		for (int i = -1; i < 2; i++) {
			for (int j = 0; j < 2; j++) {
				for (int k = -1; k < 2; k++) {
					BlockPos candidate = position.func_177982_a(i, j, k);
					if (candidate.equals(position)) {
						continue;
					}
					if (Math.abs(candidate.func_177958_n() - start.func_177958_n()) > 5) {
						continue;
					}
					if (Math.abs(candidate.func_177952_p() - start.func_177952_p()) > 5) {
						continue;
					}

					// See whether the given position has already been processed
					if (seen.contains(candidate)) {
						continue;
					}

					if (!world.func_175667_e(candidate)) {
						continue;
					}

					IBlockState blockState = world.func_180495_p(candidate);
					ICrop crop = cocoa.getCropAt(world, candidate, blockState);
					if (crop != null) {
						crops.push(crop);
						candidates.add(candidate);
						seen.add(candidate);
					} else if (blockState.func_177230_c().isWood(world, candidate)) {
						candidates.add(candidate);
						seen.add(candidate);
					}
				}
			}
		}

		return candidates;
	}

}
