package forestry.core.gui.elements;

import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.Supplier;

import net.minecraft.client.Minecraft;
import net.minecraft.client.renderer.BufferBuilder;
import net.minecraft.client.renderer.GlStateManager;
import net.minecraft.client.renderer.Tessellator;
import net.minecraft.client.renderer.texture.TextureAtlasSprite;
import net.minecraft.client.renderer.texture.TextureManager;
import net.minecraft.client.renderer.texture.TextureMap;
import net.minecraft.client.renderer.vertex.DefaultVertexFormats;
import net.minecraft.item.EnumRarity;
import net.minecraft.util.ResourceLocation;

import net.minecraftforge.fluids.Fluid;
import net.minecraftforge.fluids.FluidStack;
import net.minecraftforge.fluids.FluidTankInfo;

import forestry.core.gui.Drawable;
import forestry.core.utils.Translator;

public class TankElement extends GuiElement {
	/* Attributes - Final */
	@Nullable
	private final Drawable background;
	@Nullable
	private final Drawable overlay;
	private final Supplier<FluidTankInfo> tank;

	public TankElement(int xPos, int yPos, @Nullable Drawable background, Supplier<FluidTankInfo> tank) {
		this(xPos, yPos, background, tank, null);
	}

	public TankElement(int xPos, int yPos, @Nullable Drawable background, Supplier<FluidTankInfo> tank, @Nullable Drawable overlay) {
		this(xPos, yPos, background, tank, overlay, 16, 58);
	}

	public TankElement(int xPos, int yPos, @Nullable Drawable background, Supplier<FluidTankInfo> tank, @Nullable Drawable overlay, int width, int height) {
		super(xPos, yPos, width, height);
		this.background = background;
		this.tank = tank;
		this.overlay = overlay;
	}

	@Nullable
	private FluidTankInfo getTank() {
		return tank.get();
	}

	@Override
	public void drawElement(int mouseX, int mouseY) {
		GlStateManager.func_179084_k();
		GlStateManager.func_179141_d();
		if (background != null) {
			GlStateManager.func_179131_c(1.0F, 1.0F, 1.0F, 1.0F);
			background.draw(0, 0);
		}
		FluidTankInfo tankInfo = getTank();
		if (tankInfo == null || tankInfo.capacity <= 0) {
			return;
		}

		FluidStack contents = tankInfo.fluid;
		Minecraft minecraft = Minecraft.func_71410_x();
		TextureManager textureManager = minecraft.func_110434_K();
		if (contents != null && contents.amount > 0 && contents.getFluid() != null) {
			Fluid fluid = contents.getFluid();
			if (fluid != null) {
				TextureMap textureMapBlocks = minecraft.func_147117_R();
				ResourceLocation fluidStill = fluid.getStill();
				TextureAtlasSprite fluidStillSprite = null;
				if (fluidStill != null) {
					fluidStillSprite = textureMapBlocks.getTextureExtry(fluidStill.toString());
				}
				if (fluidStillSprite == null) {
					fluidStillSprite = textureMapBlocks.func_174944_f();
				}

				int fluidColor = fluid.getColor(contents);

				int scaledAmount = contents.amount * height / tankInfo.capacity;
				if (contents.amount > 0 && scaledAmount < 1) {
					scaledAmount = 1;
				}
				if (scaledAmount > height) {
					scaledAmount = height;
				}

				textureManager.func_110577_a(TextureMap.field_110575_b);
				setGLColorFromInt(fluidColor);

				final int xTileCount = width / 16;
				final int xRemainder = width - xTileCount * 16;
				final int yTileCount = scaledAmount / 16;
				final int yRemainder = scaledAmount - yTileCount * 16;

				final int yStart = height;

				for (int xTile = 0; xTile <= xTileCount; xTile++) {
					for (int yTile = 0; yTile <= yTileCount; yTile++) {
						int width = xTile == xTileCount ? xRemainder : 16;
						int height = yTile == yTileCount ? yRemainder : 16;
						int x = xTile * 16;
						int y = yStart - (yTile + 1) * 16;
						if (width > 0 && height > 0) {
							int maskTop = 16 - height;
							int maskRight = 16 - width;

							drawFluidTexture(x, y, fluidStillSprite, maskTop, maskRight, 100);
						}
					}
				}
			}
		}

		if (overlay != null) {
			GlStateManager.func_179097_i();
			overlay.draw(0, 0);
			GlStateManager.func_179126_j();
		}

		GlStateManager.func_179131_c(1, 1, 1, 1);
		GlStateManager.func_179118_c();
	}

	@Override
	public List<String> getTooltip(int mouseX, int mouseY) {
		FluidTankInfo tankInfo = getTank();
		if (tankInfo == null) {
			return Collections.emptyList();
		}
		List<String> toolTip = new ArrayList<>();
		int amount = 0;
		FluidStack fluidStack = tankInfo.fluid;
		if (fluidStack != null) {
			Fluid fluidType = fluidStack.getFluid();
			EnumRarity rarity = fluidType.getRarity();
			if (rarity == null) {
				rarity = EnumRarity.COMMON;
			}
			toolTip.add(rarity.field_77937_e + fluidType.getLocalizedName(fluidStack));
			amount = fluidStack.amount;
		}
		String liquidAmount = Translator.translateToLocalFormatted("for.gui.tooltip.liquid.amount", amount, tankInfo.capacity);
		toolTip.add(liquidAmount);
		return toolTip;
	}

	@Override
	public boolean hasTooltip() {
		return true;
	}

	private static void setGLColorFromInt(int color) {
		float red = (color >> 16 & 0xFF) / 255.0F;
		float green = (color >> 8 & 0xFF) / 255.0F;
		float blue = (color & 0xFF) / 255.0F;

		GlStateManager.func_179131_c(red, green, blue, 1.0F);
	}

	private static void drawFluidTexture(double xCoord, double yCoord, TextureAtlasSprite textureSprite, int maskTop, int maskRight, double zLevel) {
		double uMin = textureSprite.func_94209_e();
		double uMax = textureSprite.func_94212_f();
		double vMin = textureSprite.func_94206_g();
		double vMax = textureSprite.func_94210_h();
		uMax = uMax - maskRight / 16.0 * (uMax - uMin);
		vMax = vMax - maskTop / 16.0 * (vMax - vMin);

		Tessellator tessellator = Tessellator.func_178181_a();
		BufferBuilder buffer = tessellator.func_178180_c();
		buffer.func_181668_a(7, DefaultVertexFormats.field_181707_g);
		buffer.func_181662_b(xCoord, yCoord + 16, zLevel).func_187315_a(uMin, vMax).func_181675_d();
		buffer.func_181662_b(xCoord + 16 - maskRight, yCoord + 16, zLevel).func_187315_a(uMax, vMax).func_181675_d();
		buffer.func_181662_b(xCoord + 16 - maskRight, yCoord + maskTop, zLevel).func_187315_a(uMax, vMin).func_181675_d();
		buffer.func_181662_b(xCoord, yCoord + maskTop, zLevel).func_187315_a(uMin, vMin).func_181675_d();
		tessellator.func_78381_a();
	}
}
