/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.core.utils;

import javax.annotation.Nullable;

import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.IInventory;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.NonNullList;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;

import net.minecraftforge.common.ForgeHooks;
import net.minecraftforge.items.IItemHandler;

import net.minecraftforge.fml.common.Optional;

import forestry.core.circuits.ISocketable;
import forestry.core.inventory.ItemHandlerInventoryManipulator;
import forestry.core.inventory.StandardStackFilters;
import forestry.core.tiles.AdjacentTileCache;
import forestry.modules.ForestryModuleUids;
import forestry.modules.ModuleHelper;
import forestry.plugins.ForestryCompatPlugins;

public abstract class InventoryUtil {
	/**
	 * Attempts to move an ItemStack from one inventory to another.
	 *
	 * @param source The source IInventory.
	 * @param dest   The destination IInventory.
	 * @return true if any items were moved
	 */
	public static boolean moveItemStack(IItemHandler source, IItemHandler dest) {
		ItemHandlerInventoryManipulator manipulator = new ItemHandlerInventoryManipulator(source);
		return manipulator.transferOneStack(dest, StandardStackFilters.ALL);
	}

	/**
	 * Attempts to move an ItemStack from one inventory to another.
	 *
	 * @param source       The source IInventory.
	 * @param destinations The destination IInventory.
	 * @return true if any items were moved
	 */
	public static boolean moveItemStack(IItemHandler source, Iterable<IItemHandler> destinations) {
		for (IItemHandler dest : destinations) {
			if (moveItemStack(source, dest)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Attempts to move a single item from the source inventory into a adjacent Buildcraft pipe.
	 * If the attempt fails, the source Inventory will not be modified.
	 *
	 * @param source    The source inventory
	 * @param tileCache The tile cache of the source block.
	 * @return true if an item was inserted, otherwise false.
	 */
	public static boolean moveOneItemToPipe(IItemHandler source, AdjacentTileCache tileCache) {
		return moveOneItemToPipe(source, tileCache, EnumFacing.values());
	}

	public static boolean moveOneItemToPipe(IItemHandler source, AdjacentTileCache tileCache, EnumFacing[] directions) {
		if (ModuleHelper.isModuleEnabled(ForestryCompatPlugins.ID, ForestryModuleUids.BUILDCRAFT_TRANSPORT)) {
			return internal_moveOneItemToPipe(source, tileCache, directions);
		}

		return false;
	}

	//TODO Buildcraft for 1.9
	@Optional.Method(modid = "buildcraftapi_transport")
	private static boolean internal_moveOneItemToPipe(IItemHandler source, AdjacentTileCache tileCache, EnumFacing[] directions) {
		//		IInventory invClone = new InventoryCopy(source);
		//		ItemStack stackToMove = removeOneItem(invClone);
		//		if (stackToMove == null) {
		//			return false;
		//		}
		//		if (stackToMove.stackSize <= 0) {
		//			return false;
		//		}
		//
		//		List<Map.Entry<EnumFacing, IPipeTile>> pipes = new ArrayList<>();
		//		boolean foundPipe = false;
		//		for (EnumFacing side : directions) {
		//			TileEntity tile = tileCache.getTileOnSide(side);
		//			if (tile instanceof IPipeTile) {
		//				IPipeTile pipe = (IPipeTile) tile;
		//				if (pipe.getPipeType() == IPipeTile.PipeType.ITEM && pipe.isPipeConnected(side.getOpposite())) {
		//					pipes.add(new AbstractMap.SimpleEntry<>(side, pipe));
		//					foundPipe = true;
		//				}
		//			}
		//		}
		//
		//		if (!foundPipe) {
		//			return false;
		//		}
		//
		//		int choice = tileCache.getSource().getWorld().rand.nextInt(pipes.size());
		//		Map.Entry<EnumFacing, IPipeTile> pipe = pipes.get(choice);
		//		if (pipe.getValue().injectItem(stackToMove, false, pipe.getKey().getOpposite(), null) > 0) {
		//			if (removeOneItem(source, stackToMove) != null) {
		//				pipe.getValue().injectItem(stackToMove, true, pipe.getKey().getOpposite(), null);
		//				return true;
		//			}
		//		}
		return false;
	}

	/* REMOVAL */

	/**
	 * Removes a set of items from an inventory.
	 * Removes the exact items first if they exist, and then removes crafting equivalents.
	 * If the inventory doesn't have all the required items, returns false without removing anything.
	 * If stowContainer is true, items with containers will have their container stowed.
	 */
	public static boolean removeSets(IInventory inventory, int count, NonNullList<ItemStack> set, @Nullable EntityPlayer player, boolean stowContainer, boolean oreDictionary, boolean craftingTools, boolean doRemove) {
		NonNullList<ItemStack> stock = getStacks(inventory);

		if (doRemove) {
			NonNullList<ItemStack> removed = removeSets(inventory, count, set, player, stowContainer, oreDictionary, craftingTools);
			return removed != null && removed.size() >= count;
		} else {
			return ItemStackUtil.containsSets(set, stock, oreDictionary, craftingTools) >= count;
		}
	}

	public static boolean removeSets(IInventory inventory, int count, NonNullList<ItemStack> set, NonNullList<String> oreDicts, @Nullable EntityPlayer player, boolean stowContainer, boolean craftingTools, boolean doRemove) {
		NonNullList<ItemStack> stock = getStacks(inventory);

		if (doRemove) {
			NonNullList<ItemStack> removed = removeSets(inventory, count, set, oreDicts, player, stowContainer, craftingTools);
			return removed != null && removed.size() >= count;
		} else {
			return ItemStackUtil.containsSets(set, stock, oreDicts, craftingTools) >= count;
		}
	}

	public static boolean deleteExactSet(IInventory inventory, NonNullList<ItemStack> required) {
		NonNullList<ItemStack> offered = getStacks(inventory);
		NonNullList<ItemStack> condensedRequired = ItemStackUtil.condenseStacks(required);
		NonNullList<ItemStack> condensedOffered = ItemStackUtil.condenseStacks(offered);

		for (ItemStack req : condensedRequired) {
			if (!containsExactStack(req, condensedOffered)) {
				return false;
			}
		}

		for (ItemStack itemStack : condensedRequired) {
			deleteExactStack(inventory, itemStack);
		}
		return true;
	}

	private static boolean containsExactStack(ItemStack req, NonNullList<ItemStack> condensedOffered) {
		for (ItemStack offer : condensedOffered) {
			if (offer.func_190916_E() >= req.func_190916_E() && ItemStackUtil.areItemStacksEqualIgnoreCount(req, offer)) {
				return true;
			}
		}
		return false;
	}

	private static void deleteExactStack(IInventory inventory, ItemStack itemStack) {
		int count = itemStack.func_190916_E();
		for (int j = 0; j < inventory.func_70302_i_(); j++) {
			ItemStack stackInSlot = inventory.func_70301_a(j);
			if (!stackInSlot.func_190926_b()) {
				if (ItemStackUtil.areItemStacksEqualIgnoreCount(itemStack, stackInSlot)) {
					ItemStack removed = inventory.func_70298_a(j, count);
					count -= removed.func_190916_E();
					if (count == 0) {
						return;
					}
				}
			}
		}
	}

	@Nullable
	public static NonNullList<ItemStack> removeSets(IInventory inventory, int count, NonNullList<ItemStack> set, @Nullable EntityPlayer player, boolean stowContainer, boolean oreDictionary, boolean craftingTools) {
		NonNullList<ItemStack> removed = NonNullList.func_191197_a(set.size(), ItemStack.field_190927_a);
		NonNullList<ItemStack> stock = getStacks(inventory);

		if (ItemStackUtil.containsSets(set, stock, oreDictionary, craftingTools) < count) {
			return null;
		}

		for (int i = 0; i < set.size(); i++) {
			ItemStack itemStack = set.get(i);
			if (!itemStack.func_190926_b()) {
				ItemStack stackToRemove = itemStack.func_77946_l();
				stackToRemove.func_190920_e(stackToRemove.func_190916_E() * count);

				// try to remove the exact stack first
				ItemStack removedStack = removeStack(inventory, stackToRemove, player, stowContainer, false, false);
				if (removedStack.func_190926_b()) {
					// remove crafting equivalents next
					removedStack = removeStack(inventory, stackToRemove, player, stowContainer, oreDictionary, craftingTools);
				}

				removed.set(i, removedStack);
			}
		}
		return removed;
	}

	@Nullable
	public static NonNullList<ItemStack> removeSets(IInventory inventory, int count, NonNullList<ItemStack> set, NonNullList<String> oreDicts, @Nullable EntityPlayer player, boolean stowContainer, boolean craftingTools) {
		NonNullList<ItemStack> removed = NonNullList.func_191197_a(set.size(), ItemStack.field_190927_a);
		NonNullList<ItemStack> stock = getStacks(inventory);

		if (ItemStackUtil.containsSets(set, stock, oreDicts, craftingTools) < count) {
			return null;
		}

		for (int i = 0; i < set.size(); i++) {
			ItemStack itemStack = set.get(i);
			if (!itemStack.func_190926_b()) {
				ItemStack stackToRemove = itemStack.func_77946_l();
				stackToRemove.func_190920_e(stackToRemove.func_190916_E() * count);

				// try to remove the exact stack first
				ItemStack removedStack = removeStack(inventory, stackToRemove, null, player, stowContainer, false);
				if (removedStack.func_190926_b()) {
					// remove crafting equivalents next
					removedStack = removeStack(inventory, stackToRemove, oreDicts.get(i), player, stowContainer, craftingTools);
				}

				removed.set(i, removedStack);
			}
		}
		return removed;
	}

	/**
	 * Private Helper for removeSetsFromInventory. Assumes removal is possible.
	 */
	private static ItemStack removeStack(IInventory inventory, ItemStack stackToRemove, @Nullable EntityPlayer player, boolean stowContainer, boolean oreDictionary, boolean craftingTools) {
		for (int j = 0; j < inventory.func_70302_i_(); j++) {
			ItemStack stackInSlot = inventory.func_70301_a(j);
			if (!stackInSlot.func_190926_b()) {
				if (ItemStackUtil.isCraftingEquivalent(stackToRemove, stackInSlot, oreDictionary, craftingTools)) {
					ItemStack removed = inventory.func_70298_a(j, stackToRemove.func_190916_E());
					stackToRemove.func_190918_g(removed.func_190916_E());

					if (stowContainer && stackToRemove.func_77973_b().hasContainerItem(stackToRemove)) {
						stowContainerItem(removed, inventory, j, player);
					}

					if (stackToRemove.func_190926_b()) {
						return removed;
					}
				}
			}
		}
		return ItemStack.field_190927_a;
	}

	private static ItemStack removeStack(IInventory inventory, ItemStack stackToRemove, @Nullable String oreDictOfStack, @Nullable EntityPlayer player, boolean stowContainer, boolean craftingTools) {
		for (int j = 0; j < inventory.func_70302_i_(); j++) {
			ItemStack stackInSlot = inventory.func_70301_a(j);
			if (!stackInSlot.func_190926_b()) {
				if (ItemStackUtil.isCraftingEquivalent(stackToRemove, stackInSlot, oreDictOfStack, craftingTools)) {
					ItemStack removed = inventory.func_70298_a(j, stackToRemove.func_190916_E());
					stackToRemove.func_190918_g(removed.func_190916_E());

					if (stowContainer && stackToRemove.func_77973_b().hasContainerItem(stackToRemove)) {
						stowContainerItem(removed, inventory, j, player);
					}

					if (stackToRemove.func_190926_b()) {
						return removed;
					}
				}
			}
		}
		return ItemStack.field_190927_a;
	}

	/* CONTAINS */

	public static boolean contains(IInventory inventory, NonNullList<ItemStack> query) {
		return contains(inventory, query, 0, inventory.func_70302_i_());
	}

	public static boolean contains(IInventory inventory, NonNullList<ItemStack> query, int startSlot, int slots) {
		NonNullList<ItemStack> stock = getStacks(inventory, startSlot, slots);
		return ItemStackUtil.containsSets(query, stock) > 0;
	}

	public static boolean containsPercent(IInventory inventory, float percent) {
		return containsPercent(inventory, percent, 0, inventory.func_70302_i_());
	}

	public static boolean containsPercent(IInventory inventory, float percent, int slot1, int length) {
		int amount = 0;
		int stackMax = 0;
		for (ItemStack itemStack : getStacks(inventory, slot1, length)) {
			if (itemStack == null) {
				stackMax += 64;
				continue;
			}

			amount += itemStack.func_190916_E();
			stackMax += itemStack.func_77976_d();
		}
		if (stackMax == 0) {
			return false;
		}
		return (float) amount / (float) stackMax >= percent;
	}

	public static boolean isEmpty(IInventory inventory, int slotStart, int slotCount) {
		for (int i = slotStart; i < slotStart + slotCount; i++) {
			if (!inventory.func_70301_a(i).func_190926_b()) {
				return false;
			}
		}
		return true;
	}

	public static NonNullList<ItemStack> getStacks(IInventory inventory) {
		NonNullList<ItemStack> stacks = NonNullList.func_191197_a(inventory.func_70302_i_(), ItemStack.field_190927_a);
		for (int i = 0; i < inventory.func_70302_i_(); i++) {
			stacks.set(i, inventory.func_70301_a(i));
		}
		return stacks;
	}

	public static NonNullList<ItemStack> getStacks(IInventory inventory, int slot1, int length) {
		NonNullList<ItemStack> result = NonNullList.func_191197_a(length, ItemStack.field_190927_a);
		for (int i = slot1; i < slot1 + length; i++) {
			result.set(i - slot1, inventory.func_70301_a(i));
		}
		return result;
	}

	public static NonNullList<String> getOreDictAsList(String[][] oreDicts) {
		NonNullList<String> result = NonNullList.func_191197_a(9, "");
		if (oreDicts == null || oreDicts.length == 0) {
			return result;
		}
		for (int i = 0; i < oreDicts.length; i++) {
			for (int d = 0; d < oreDicts[i].length; d++) {
				result.set(i * 3 + d, oreDicts[d][i]);
			}
		}
		return result;
	}

	public static boolean tryAddStacksCopy(IInventory inventory, NonNullList<ItemStack> stacks, int startSlot, int slots, boolean all) {

		for (ItemStack stack : stacks) {
			if (stack == null || stack.func_190926_b()) {
				continue;
			}

			if (!tryAddStack(inventory, stack.func_77946_l(), startSlot, slots, all)) {
				return false;
			}
		}

		return true;
	}

	public static boolean tryAddStack(IInventory inventory, ItemStack stack, boolean all) {
		return tryAddStack(inventory, stack, 0, inventory.func_70302_i_(), all, true);
	}

	public static boolean tryAddStack(IInventory inventory, ItemStack stack, boolean all, boolean doAdd) {
		return tryAddStack(inventory, stack, 0, inventory.func_70302_i_(), all, doAdd);
	}

	/**
	 * Tries to add a stack to the specified slot range.
	 */
	public static boolean tryAddStack(IInventory inventory, ItemStack stack, int startSlot, int slots, boolean all) {
		return tryAddStack(inventory, stack, startSlot, slots, all, true);
	}

	public static boolean tryAddStack(IInventory inventory, ItemStack stack, int startSlot, int slots, boolean all, boolean doAdd) {
		int added = addStack(inventory, stack, startSlot, slots, false);
		boolean success = all ? added == stack.func_190916_E() : added > 0;

		if (success && doAdd) {
			addStack(inventory, stack, startSlot, slots, true);
		}

		return success;
	}

	public static int addStack(IInventory inventory, ItemStack stack, boolean doAdd) {
		return addStack(inventory, stack, 0, inventory.func_70302_i_(), doAdd);
	}

	public static int addStack(IInventory inventory, ItemStack stack, int startSlot, int slots, boolean doAdd) {
		if (stack.func_190926_b()) {
			return 0;
		}

		int added = 0;
		// Add to existing stacks first
		for (int i = startSlot; i < startSlot + slots; i++) {

			ItemStack inventoryStack = inventory.func_70301_a(i);
			// Empty slot. Add
			if (inventoryStack.func_190926_b()) {
				continue;
			}

			// Already occupied by different item, skip this slot.
			if (!inventoryStack.func_77985_e()) {
				continue;
			}
			if (!inventoryStack.func_77969_a(stack)) {
				continue;
			}
			if (!ItemStack.func_77970_a(inventoryStack, stack)) {
				continue;
			}

			int remain = stack.func_190916_E() - added;
			int space = inventoryStack.func_77976_d() - inventoryStack.func_190916_E();
			// No space left, skip this slot.
			if (space <= 0) {
				continue;
			}
			// Enough space
			if (space >= remain) {
				if (doAdd) {
					inventoryStack.func_190917_f(remain);
				}
				return stack.func_190916_E();
			}

			// Not enough space
			if (doAdd) {
				inventoryStack.func_190920_e(inventoryStack.func_77976_d());
			}

			added += space;
		}

		if (added >= stack.func_190916_E()) {
			return added;
		}

		for (int i = startSlot; i < startSlot + slots; i++) {
			if (inventory.func_70301_a(i).func_190926_b()) {
				if (doAdd) {
					inventory.func_70299_a(i, stack.func_77946_l());
					inventory.func_70301_a(i).func_190920_e(stack.func_190916_E() - added);
				}
				return stack.func_190916_E();
			}
		}

		return added;
	}

	public static boolean stowInInventory(ItemStack itemstack, IInventory inventory, boolean doAdd) {
		return stowInInventory(itemstack, inventory, doAdd, 0, inventory.func_70302_i_());
	}

	public static boolean stowInInventory(ItemStack itemstack, IInventory inventory, boolean doAdd, int slot1, int count) {

		boolean added = false;

		for (int i = slot1; i < slot1 + count; i++) {
			ItemStack inventoryStack = inventory.func_70301_a(i);

			// Grab those free slots
			if (inventoryStack.func_190926_b()) {
				if (doAdd) {
					inventory.func_70299_a(i, itemstack.func_77946_l());
					itemstack.func_190920_e(0);
				}
				return true;
			}

			// Already full
			if (inventoryStack.func_190916_E() >= inventoryStack.func_77976_d()) {
				continue;
			}

			// Not same type
			if (!inventoryStack.func_77969_a(itemstack)) {
				continue;
			}
			if (!ItemStack.func_77970_a(inventoryStack, itemstack)) {
				continue;
			}

			int space = inventoryStack.func_77976_d() - inventoryStack.func_190916_E();

			// Enough space to add all
			if (space > itemstack.func_190916_E()) {
				if (doAdd) {
					inventoryStack.func_190917_f(itemstack.func_190916_E());
					itemstack.func_190920_e(0);
				}
				return true;
				// Only part can be added
			} else {
				if (doAdd) {
					inventoryStack.func_190920_e(inventoryStack.func_77976_d());
					itemstack.func_190918_g(space);
				}
				added = true;
			}

		}

		return added;
	}

	public static void stowContainerItem(ItemStack itemstack, IInventory stowing, int slotIndex, @Nullable EntityPlayer player) {
		if (!itemstack.func_77973_b().hasContainerItem(itemstack)) {
			return;
		}

		ItemStack container = ForgeHooks.getContainerItem(itemstack);
		if (!container.func_190926_b()) {
			if (!tryAddStack(stowing, container, slotIndex, 1, true)) {
				if (!tryAddStack(stowing, container, true) && player != null) {
					player.func_71019_a(container, true);
				}
			}
		}
	}

	public static void deepCopyInventoryContents(IInventory source, IInventory destination) {
		if (source.func_70302_i_() != destination.func_70302_i_()) {
			throw new IllegalArgumentException("Inventory sizes do not match. Source: " + source + ", Destination: " + destination);
		}

		for (int i = 0; i < source.func_70302_i_(); i++) {
			ItemStack stack = source.func_70301_a(i);
			if (!stack.func_190926_b()) {
				stack = stack.func_77946_l();
			}
			destination.func_70299_a(i, stack);
		}
	}

	public static void dropInventory(IInventory inventory, World world, double x, double y, double z) {
		// Release inventory
		for (int slot = 0; slot < inventory.func_70302_i_(); slot++) {
			ItemStack itemstack = inventory.func_70301_a(slot);
			dropItemStackFromInventory(itemstack, world, x, y, z);
			inventory.func_70299_a(slot, ItemStack.field_190927_a);
		}
	}

	public static void dropInventory(IInventory inventory, World world, BlockPos pos) {
		dropInventory(inventory, world, pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p());
	}

	public static void dropSockets(ISocketable socketable, World world, double x, double y, double z) {
		for (int slot = 0; slot < socketable.getSocketCount(); slot++) {
			ItemStack itemstack = socketable.getSocket(slot);
			dropItemStackFromInventory(itemstack, world, x, y, z);
			socketable.setSocket(slot, ItemStack.field_190927_a);
		}
	}

	public static void dropSockets(ISocketable socketable, World world, BlockPos pos) {
		dropSockets(socketable, world, pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p());
	}

	public static void dropItemStackFromInventory(ItemStack itemStack, World world, double x, double y, double z) {
		if (itemStack.func_190926_b()) {
			return;
		}

		float f = world.field_73012_v.nextFloat() * 0.8F + 0.1F;
		float f1 = world.field_73012_v.nextFloat() * 0.8F + 0.1F;
		float f2 = world.field_73012_v.nextFloat() * 0.8F + 0.1F;

		while (!itemStack.func_190926_b()) {
			int stackPartial = world.field_73012_v.nextInt(21) + 10;
			if (stackPartial > itemStack.func_190916_E()) {
				stackPartial = itemStack.func_190916_E();
			}
			ItemStack drop = itemStack.func_77979_a(stackPartial);
			EntityItem entityitem = new EntityItem(world, x + f, y + f1, z + f2, drop);
			float accel = 0.05F;
			entityitem.field_70159_w = (float) world.field_73012_v.nextGaussian() * accel;
			entityitem.field_70181_x = (float) world.field_73012_v.nextGaussian() * accel + 0.2F;
			entityitem.field_70179_y = (float) world.field_73012_v.nextGaussian() * accel;
			world.func_72838_d(entityitem);
		}
	}

	/* NBT */

	/**
	 * The database has an inventory large enough that int must be used here instead of byte
	 * TODO in 1.13 - remove migrations from this.
	 */
	public static void readFromNBT(IInventory inventory, NBTTagCompound nbttagcompound) {
		if (!nbttagcompound.func_74764_b(inventory.func_70005_c_())) {
			return;
		}

		NBTTagList nbttaglist = nbttagcompound.func_150295_c(inventory.func_70005_c_(), 10);

		for (int j = 0; j < nbttaglist.func_74745_c(); ++j) {
			NBTTagCompound nbttagcompound2 = nbttaglist.func_150305_b(j);
			int index;
			//type of byte tag is 1
			if (nbttagcompound2.func_150297_b("Slot", 1)) {
				index = nbttagcompound2.func_74771_c("Slot");
			} else {
				index = nbttagcompound2.func_74762_e("Slot");
			}
			inventory.func_70299_a(index, new ItemStack(nbttagcompound2));
		}
	}

	public static void writeToNBT(IInventory inventory, NBTTagCompound nbttagcompound) {
		NBTTagList nbttaglist = new NBTTagList();
		for (int i = 0; i < inventory.func_70302_i_(); i++) {
			if (!inventory.func_70301_a(i).func_190926_b()) {
				NBTTagCompound nbttagcompound2 = new NBTTagCompound();
				nbttagcompound2.func_74768_a("Slot", i);
				inventory.func_70301_a(i).func_77955_b(nbttagcompound2);
				nbttaglist.func_74742_a(nbttagcompound2);
			}
		}
		nbttagcompound.func_74782_a(inventory.func_70005_c_(), nbttaglist);
	}
}
