/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.worktable.recipes;

import javax.annotation.Nullable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import net.minecraft.inventory.InventoryCrafting;
import net.minecraft.item.ItemStack;
import net.minecraft.item.crafting.IRecipe;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.nbt.NBTTagString;
import net.minecraft.util.ResourceLocation;
import net.minecraft.world.World;

import net.minecraftforge.fml.common.registry.ForgeRegistries;

import forestry.api.core.INbtReadable;
import forestry.api.core.INbtWritable;
import forestry.core.network.IStreamable;
import forestry.core.network.PacketBufferForestry;
import forestry.core.utils.InventoryUtil;
import forestry.core.utils.NBTUtilForestry;
import forestry.worktable.inventory.InventoryCraftingForestry;

public final class MemorizedRecipe implements INbtWritable, INbtReadable, IStreamable {
	private InventoryCraftingForestry craftMatrix = new InventoryCraftingForestry();
	private List<IRecipe> recipes = new ArrayList<>();
	private int selectedRecipe;
	private long lastUsed;
	private boolean locked;

	public MemorizedRecipe(PacketBufferForestry data) throws IOException {
		readData(data);
	}

	public MemorizedRecipe(NBTTagCompound nbt) {
		readFromNBT(nbt);
	}

	public MemorizedRecipe(InventoryCraftingForestry craftMatrix, List<IRecipe> recipes) {
		InventoryUtil.deepCopyInventoryContents(craftMatrix, this.craftMatrix);
		this.recipes = recipes;
	}

	public InventoryCraftingForestry getCraftMatrix() {
		return craftMatrix;
	}

	public void setCraftMatrix(InventoryCraftingForestry craftMatrix) {
		this.craftMatrix = craftMatrix;
	}

	public void incrementRecipe() {
		selectedRecipe++;
		if (selectedRecipe >= recipes.size()) {
			selectedRecipe = 0;
		}
	}

	public void decrementRecipe() {
		selectedRecipe--;
		if (selectedRecipe < 0) {
			selectedRecipe = recipes.size() - 1;
		}
	}

	public boolean hasRecipeConflict() {
		return recipes.size() > 1;
	}

	public void removeRecipeConflicts() {
		IRecipe recipe = getSelectedRecipe();
		recipes.clear();
		recipes.add(recipe);
		selectedRecipe = 0;
	}

	public ItemStack getOutputIcon() {
		IRecipe selectedRecipe = getSelectedRecipe();
		if (selectedRecipe != null) {
			ItemStack recipeOutput = selectedRecipe.func_77572_b(craftMatrix);
			if (!recipeOutput.func_190926_b()) {
				return recipeOutput;
			}
		}
		return ItemStack.field_190927_a;
	}

	public ItemStack getCraftingResult(InventoryCrafting inventoryCrafting, World world) {
		IRecipe selectedRecipe = getSelectedRecipe();
		if (selectedRecipe != null && selectedRecipe.func_77569_a(inventoryCrafting, world)) {
			ItemStack recipeOutput = selectedRecipe.func_77572_b(inventoryCrafting);
			if (!recipeOutput.func_190926_b()) {
				return recipeOutput;
			}
		}
		return ItemStack.field_190927_a;
	}

	@Nullable
	public IRecipe getSelectedRecipe() {
		if (recipes.isEmpty()) {
			return null;
		} else {
			return recipes.get(selectedRecipe);
		}
	}

	public boolean hasRecipe(@Nullable IRecipe recipe) {
		return this.recipes.contains(recipe);
	}

	public void updateLastUse(long lastUsed) {
		this.lastUsed = lastUsed;
	}

	public long getLastUsed() {
		return lastUsed;
	}

	public void toggleLock() {
		locked = !locked;
	}

	public boolean isLocked() {
		return locked;
	}

	/* INbtWritable */
	@Override
	public final void readFromNBT(NBTTagCompound nbttagcompound) {
		InventoryUtil.readFromNBT(craftMatrix, nbttagcompound);
		lastUsed = nbttagcompound.func_74763_f("LastUsed");
		locked = nbttagcompound.func_74767_n("Locked");

		if (nbttagcompound.func_74764_b("SelectedRecipe")) {
			selectedRecipe = nbttagcompound.func_74762_e("SelectedRecipe");
		}

		recipes.clear();
		NBTTagList recipesNbt = nbttagcompound.func_150295_c("Recipes", NBTUtilForestry.EnumNBTType.STRING.ordinal());
		for (int i = 0; i < recipesNbt.func_74745_c(); i++) {
			String recipeKey = recipesNbt.func_150307_f(i);
			ResourceLocation key = new ResourceLocation(recipeKey);
			IRecipe recipe = ForgeRegistries.RECIPES.getValue(key);
			if (recipe != null) {
				recipes.add(recipe);
			}
		}

		if (selectedRecipe > recipes.size()) {
			selectedRecipe = 0;
		}
	}

	@Override
	public NBTTagCompound writeToNBT(NBTTagCompound nbttagcompound) {
		InventoryUtil.writeToNBT(craftMatrix, nbttagcompound);
		nbttagcompound.func_74772_a("LastUsed", lastUsed);
		nbttagcompound.func_74757_a("Locked", locked);
		nbttagcompound.func_74768_a("SelectedRecipe", selectedRecipe);

		NBTTagList recipesNbt = new NBTTagList();
		for (IRecipe recipe : recipes) {
			ResourceLocation recipeKey = ForgeRegistries.RECIPES.getKey(recipe);
			if (recipeKey != null) {
				recipesNbt.func_74742_a(new NBTTagString(recipeKey.toString()));
			}
		}
		nbttagcompound.func_74782_a("Recipes", recipesNbt);

		return nbttagcompound;
	}

	/* IStreamable */
	@Override
	public void writeData(PacketBufferForestry data) {
		data.writeInventory(craftMatrix);
		data.writeBoolean(locked);
		data.func_150787_b(selectedRecipe);

		data.func_150787_b(recipes.size());
		for (IRecipe recipe : recipes) {
			ResourceLocation recipeId = ForgeRegistries.RECIPES.getKey(recipe);
			if (recipeId != null) {
				data.func_180714_a(recipeId.toString());
			}
		}
	}

	@Override
	public void readData(PacketBufferForestry data) throws IOException {
		data.readInventory(craftMatrix);
		locked = data.readBoolean();
		selectedRecipe = data.func_150792_a();

		recipes.clear();
		int recipeCount = data.func_150792_a();
		for (int i = 0; i < recipeCount; i++) {
			String recipeId = data.readString();
			IRecipe recipe = ForgeRegistries.RECIPES.getValue(new ResourceLocation(recipeId));
			if (recipe != null) {
				recipes.add(recipe);
			}
		}
	}
}
