package forestry.database.gui;

import javax.annotation.Nullable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import net.minecraft.client.gui.GuiButton;
import net.minecraft.client.gui.GuiTextField;
import net.minecraft.client.renderer.GlStateManager;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.IInventory;
import net.minecraft.inventory.Slot;
import net.minecraft.item.ItemStack;
import net.minecraft.util.ResourceLocation;

import forestry.core.config.Constants;
import forestry.core.gui.ContainerForestry;
import forestry.core.gui.Drawable;
import forestry.core.gui.GuiAnalyzerProvider;
import forestry.core.gui.slots.SlotFilteredInventory;
import forestry.core.gui.widgets.IScrollable;
import forestry.core.gui.widgets.WidgetScrollBar;
import forestry.database.DatabaseHelper;
import forestry.database.DatabaseItem;
import forestry.database.gui.buttons.DatabaseButton;
import forestry.database.gui.buttons.GuiDatabaseButton;
import forestry.database.gui.widgets.WidgetDatabaseSlot;
import forestry.database.tiles.TileDatabase;

public class GuiDatabase extends GuiAnalyzerProvider<ContainerDatabase> implements IScrollable {
	/* Attributes - Constants */
	private static final ResourceLocation CREATIVE_TABS = new ResourceLocation(Constants.TEXTURE_PATH_GUI + "/container/creative_inventory/tabs.png");
	private static final Drawable SCROLLBAR_SLIDER = new Drawable(CREATIVE_TABS, 232, 0, 12, 15);
	/*  Attributes - Final */
	public final TileDatabase tile;
	private final ArrayList<WidgetDatabaseSlot> slots;
	private final ArrayList<DatabaseItem> sorted = new ArrayList<>();
	/* Attributes - Gui Elements */
	@Nullable
	private GuiTextField searchField;
	private WidgetScrollBar scrollBar;
	/* Attributes - State */
	private boolean markedForSorting;
	@Nullable
	private DatabaseItem selectedItem;

	/* Constructors */
	public GuiDatabase(TileDatabase tile, EntityPlayer player) {
		super(Constants.TEXTURE_PATH_GUI + "/database_inventory.png", new ContainerDatabase(tile, player.field_71071_by), tile, 7, 140, 20, true, tile.getInternalInventory().func_70302_i_(), 0);
		this.tile = tile;

		slots = new ArrayList<>();
		field_146999_f = 218;
		field_147000_g = 202;
		//Start at index 36, because all slots before 36 are player inventory slots
		Iterator<Slot> slotIterator = container.field_75151_b.listIterator(ContainerForestry.PLAYER_INV_SLOTS);
		while (slotIterator.hasNext()) {
			Slot slot = slotIterator.next();
			if (slot instanceof SlotFilteredInventory) {
				SlotFilteredInventory slotDatabase = (SlotFilteredInventory) slot;
				slotDatabase.setChangeWatcher(this);
			}
		}
		for (int i = 0; i < 24; i++) {
			WidgetDatabaseSlot slot = new WidgetDatabaseSlot(widgetManager);
			slots.add(slot);
			widgetManager.add(slot);
		}
		widgetManager.add(this.scrollBar = new WidgetScrollBar(widgetManager, 196, 19, 12, 90, SCROLLBAR_SLIDER));
		this.scrollBar.setParameters(this, 0, tile.func_70302_i_() / 4 - 6, 1);
		analyzer.init();
		analyzer.setSelectedSlot(-1);
	}


	/* Methods */
	@Nullable
	public DatabaseItem getSelectedItem() {
		return selectedItem;
	}

	/**
	 * @return the count of all valid item in {@link #sorted}
	 */
	public int getSize() {
		return sorted.size() - 1;//subtract one because the last entry is the empty item
	}

	public int getRealSize() {
		return sorted.size();
	}

	public ItemStack getSelectedItemStack() {
		if (selectedItem == null) {
			return ItemStack.field_190927_a;
		}
		return selectedItem.itemStack;
	}

	public void markForSorting() {
		markedForSorting = true;
	}

	private void updateItems(String searchText) {
		if (markedForSorting) {
			sorted.clear();
			List<DatabaseItem> items = new ArrayList<>();
			boolean firstEmpty = false;
			for (int invIndex = 0; invIndex < tile.func_70302_i_(); invIndex++) {
				ItemStack stack = tile.func_70301_a(invIndex).func_77946_l();
				if (!stack.func_190926_b()) {
					items.add(new DatabaseItem(stack, invIndex));
					continue;
				}
				if (!firstEmpty) {
					firstEmpty = true;
					items.add(new DatabaseItem(stack, invIndex));
				}
			}
			DatabaseHelper.update(searchText, items, sorted);
			analyzer.updateSelected();
			updateViewedItems();

			markedForSorting = false;
		}
	}

	@Nullable
	public DatabaseItem getItem(int index) {
		if (sorted.isEmpty() || sorted.size() <= index || index < 0) {
			return null;
		}
		return sorted.get(index);
	}

	private void updateViewedItems() {
		int currentRow = scrollBar.getValue();
		if (currentRow < 0) {
			currentRow = 0;
		}
		//The inventory index of the first slot.
		int slotStart = currentRow * 4;
		//The inventory index of the last slot.
		int slotEnd = (currentRow + 6) * 4;
		if (slotEnd > tile.func_70302_i_()) {
			slotEnd = tile.func_70302_i_();
		}
		//The row of the first slot
		byte startRow = (byte) (currentRow % 2);
		//The index of the empty slot in the list.
		int emptySlot = sorted.size() - 1;
		for (int invIndex = 0; invIndex < tile.func_70302_i_(); invIndex++) {
			if (invIndex >= slotStart && invIndex < slotEnd) {
				int x = invIndex % 4;
				int y = invIndex / 4 - currentRow;
				int yOffset;
				int xOffset;
				if (startRow == 0) {
					yOffset = 25;
					xOffset = 17;
					if (y % 2 == 1) {
						xOffset = 38;
						yOffset = 38;
						y--;
					}
				} else {
					yOffset = 25;
					xOffset = 38;
					if (y % 2 == 1) {
						yOffset = 38;
						xOffset = 17;
						y--;
					}
				}
				int xPos = xOffset + x * 42;
				int yPos = yOffset + y / 2 * 25;
				//If the index is above the count of the valid items in the list, set the index to the same value like the index of the empty slot.
				int index = invIndex;
				if (sorted.size() <= index || sorted.isEmpty()) {
					index = emptySlot;
				}
				WidgetDatabaseSlot slot = slots.get(invIndex - slotStart);
				slot.update(xPos, yPos, index, index != emptySlot);
			}
		}
	}

	/* Methods - Implement GuiScreen */
	@Override
	public void func_73866_w_() {
		super.func_73866_w_();

		this.searchField = new GuiTextField(0, this.field_146289_q, this.field_147003_i + 101, this.field_147009_r + 6, 80, this.field_146289_q.field_78288_b);
		this.searchField.func_146203_f(50);
		this.searchField.func_146185_a(false);
		this.searchField.func_146193_g(16777215);

		func_189646_b(new GuiDatabaseButton<>(0, field_147003_i - 18, field_147009_r, DatabaseHelper.ascending, this, DatabaseButton.SORT_DIRECTION_BUTTON));

		updateViewedItems();
	}

	@Override
	public void func_73863_a(int mouseX, int mouseY, float partialTicks) {
		String searchText = searchField != null ? searchField.func_146179_b() : "";
		updateItems(searchText);
		super.func_73863_a(mouseX, mouseY, partialTicks);
	}

	@Override
	protected void func_73869_a(char typedChar, int keyCode) throws IOException {
		if (searchField != null && this.searchField.func_146201_a(typedChar, keyCode)) {
			scrollBar.setValue(0);
			markForSorting();
		} else {
			super.func_73869_a(typedChar, keyCode);
		}
	}

	@Override
	protected void func_73864_a(int mouseX, int mouseY, int mouseButton) throws IOException {
		Slot slot = getSlotAtPosition(mouseX, mouseY);
		if (slot != null && slot.getSlotIndex() == -1) {
			return;
		}
		super.func_73864_a(mouseX, mouseY, mouseButton);
		if (searchField != null) {
			searchField.func_146192_a(mouseX, mouseY, mouseButton);
		}
	}

	@Override
	protected void func_146284_a(GuiButton button) throws IOException {
		super.func_146284_a(button);
		if (button instanceof GuiDatabaseButton) {
			GuiDatabaseButton databaseButton = (GuiDatabaseButton) button;
			databaseButton.onPressed();
		}
	}

	/* Methods - Implement GuiContainer */
	@Override
	protected void func_146976_a(float f, int mouseX, int mouseY) {
		super.func_146976_a(f, mouseX, mouseY);

		if (searchField != null) {
			GlStateManager.func_179131_c(1.0F, 1.0F, 1.0F, 1.0F);
			GlStateManager.func_179140_f();
			this.searchField.func_146194_f();
		}
	}

	/* Methods - Implement GuiForestry */
	@Override
	protected void drawBackground() {
		bindTexture(textureFile);

		func_73729_b(field_147003_i, field_147009_r, 0, 0, field_146999_f, field_147000_g);
	}

	@Override
	protected void addLedgers() {
	}

	/* Methods - Implement GuiForestryTitled */

	@Override
	protected boolean centeredTitle() {
		return false;
	}

	/* Methods - Implement IGeneticAnalyzerProvider */
	@Override
	protected void drawSelectedSlot(int selectedSlot) {
		//Currently not used
	}

	@Override
	public ItemStack getSpecimen(int index) {
		DatabaseItem item = getSelectedItem();
		if (item == null) {
			return ItemStack.field_190927_a;
		}
		return item.itemStack;
	}

	@Override
	public boolean onUpdateSelected() {
		int index = selectedItem == null ? -1 : sorted.indexOf(selectedItem);
		if (index >= 0) {
			analyzer.setSelectedSlot(index);
			return true;
		}
		return false;
	}

	@Override
	public void onSelection(int index, boolean changed) {
		if (index < 0) {
			selectedItem = null;
		} else {
			this.selectedItem = sorted != null && index >= sorted.size() ? null : sorted.get(index);
		}
	}

	@Override
	public int getSelectedSlot(int index) {
		DatabaseItem item = getItem(index);
		if (item == null) {
			return -1;
		}
		return 1 + item.invIndex;
	}

	/* Methods - Implement IScrollable */
	@Override
	public void onScroll(int value) {
		updateViewedItems();
	}

	@Override
	public boolean isFocused(int mouseX, int mouseY) {
		return func_146978_c(0, 0, field_146999_f, field_147000_g, mouseX + field_147003_i, mouseY + field_147009_r);
	}

	/* Methods - Implement ISlotChangeWatcher */
	@Override
	public void onSlotChanged(IInventory inventory, int slot) {
		super.onSlotChanged(inventory, slot);
		markForSorting();
	}
}
