package forestry.core.climate;

import javax.annotation.Nullable;

import net.minecraft.entity.Entity;
import net.minecraft.entity.player.EntityPlayerMP;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;
import net.minecraft.world.biome.Biome;

import net.minecraftforge.fml.common.FMLCommonHandler;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import forestry.api.climate.ClimateManager;
import forestry.api.climate.IClimateListener;
import forestry.api.climate.IClimateProvider;
import forestry.api.climate.IClimateState;
import forestry.api.climate.IWorldClimateHolder;
import forestry.api.core.BiomeHelper;
import forestry.api.core.EnumHumidity;
import forestry.api.core.EnumTemperature;
import forestry.api.core.ILocatable;
import forestry.core.network.packets.PacketClimateListenerUpdate;
import forestry.core.network.packets.PacketClimateListenerUpdateEntity;
import forestry.core.network.packets.PacketClimateListenerUpdateEntityRequest;
import forestry.core.network.packets.PacketClimateListenerUpdateRequest;
import forestry.core.render.ParticleRender;
import forestry.core.utils.NetworkUtil;
import forestry.core.utils.TickHelper;

public class ClimateListener implements IClimateListener {
	public static final int SERVER_UPDATE = 250;

	private final Object locationProvider;
	@Nullable
	protected World world;
	@Nullable
	protected BlockPos pos;
	private IClimateState cachedState = AbsentClimateState.INSTANCE;
	private IClimateState cachedClientState = AbsentClimateState.INSTANCE;
	@SideOnly(Side.CLIENT)
	private TickHelper tickHelper;
	@SideOnly(Side.CLIENT)
	protected boolean needsClimateUpdate;
	//The total world time at the moment the cached state has been updated
	private long cacheTime = 0;
	private long lastUpdate = 0;

	public ClimateListener(Object locationProvider) {
		this.locationProvider = locationProvider;
		if(FMLCommonHandler.instance().getSide() == Side.CLIENT){
			tickHelper = new TickHelper();
			needsClimateUpdate = true;
		}
	}

	@SideOnly(Side.CLIENT)
	@Override
	public void updateClientSide(boolean spawnParticles) {
		if(spawnParticles) {
			tickHelper.onTick();
			if (cachedState.isPresent() && tickHelper.updateOnInterval(20)) {
				World worldObj = getWorldObj();
				BlockPos coordinates = getCoordinates();
				ParticleRender.addTransformParticles(worldObj, coordinates, worldObj.field_73012_v);
			}
		}
		if (needsClimateUpdate) {
			if (locationProvider instanceof Entity) {
				NetworkUtil.sendToServer(new PacketClimateListenerUpdateEntityRequest((Entity) locationProvider));
			} else {
				NetworkUtil.sendToServer(new PacketClimateListenerUpdateRequest(getCoordinates()));
			}
			needsClimateUpdate = false;
		}
	}

	private void updateState(boolean syncToClient) {
		IWorldClimateHolder climateHolder = ClimateManager.climateRoot.getWorldClimate(getWorldObj());
		long totalTime = getWorldObj().func_82737_E();
		if (cacheTime + SERVER_UPDATE > totalTime && climateHolder.getLastUpdate(getCoordinates()) == lastUpdate) {
			return;
		}
		lastUpdate = climateHolder.getLastUpdate(getCoordinates());
		cachedState = climateHolder.getState(getCoordinates());
		cacheTime = totalTime;
		if (syncToClient) {
			syncToClient();
		}
	}

	private IClimateState getState() {
		return getState(true);
	}

	private IClimateState getState(boolean update) {
		return getState(update, true);
	}

	private IClimateState getState(boolean update, boolean syncToClient) {
		World worldObj = getWorldObj();
		if (!worldObj.field_72995_K && update) {
			updateState(syncToClient);
		}
		return cachedState;
	}

	private IClimateProvider getDefaultProvider() {
		IClimateProvider provider;
		if (locationProvider instanceof IClimateProvider) {
			provider = (IClimateProvider) locationProvider;
		} else {
			provider = ClimateRoot.getInstance().getDefaultClimate(getWorldObj(), getCoordinates());
		}
		return provider;
	}

	@Override
	public Biome getBiome() {
		IClimateProvider provider = getDefaultProvider();
		return provider.getBiome();
	}

	@Override
	public EnumTemperature getTemperature() {
		Biome biome = getBiome();
		if (BiomeHelper.isBiomeHellish(biome)) {
			return EnumTemperature.HELLISH;
		}
		return EnumTemperature.getFromValue(getExactTemperature());
	}

	@Override
	public EnumHumidity getHumidity() {
		return EnumHumidity.getFromValue(getExactHumidity());
	}

	@Override
	public float getExactTemperature() {
		IClimateState climateState = getState();
		float temperature;
		if (climateState.isPresent()) {
			temperature = climateState.getTemperature();
		} else {
			Biome biome = getBiome();
			temperature = biome.func_180626_a(getCoordinates());
		}
		return temperature;
	}

	@Override
	public float getExactHumidity() {
		IClimateState climateState = getState();
		float humidity;
		if (climateState.isPresent()) {
			humidity = climateState.getHumidity();
		} else {
			Biome biome = getBiome();
			humidity = biome.func_76727_i();
		}
		return humidity;
	}

	@Override
	public IClimateState getClimateState() {
		return ClimateStateHelper.of(getExactTemperature(), getExactHumidity());
	}

	@SideOnly(Side.CLIENT)
	@Override
	public void setClimateState(IClimateState climateState) {
		this.cachedState = climateState;
	}

	@Override
	public void syncToClient() {
		if (!cachedState.equals(cachedClientState)) {
			World worldObj = getWorldObj();
			if (!worldObj.field_72995_K) {
				BlockPos coordinates = getCoordinates();
				if (locationProvider instanceof Entity) {
					NetworkUtil.sendNetworkPacket(new PacketClimateListenerUpdateEntity((Entity) locationProvider, cachedState), coordinates, worldObj);
				} else {
					NetworkUtil.sendNetworkPacket(new PacketClimateListenerUpdate(getCoordinates(), cachedState), coordinates, getWorldObj());
				}
			}
			cachedClientState = cachedState;
		}
	}

	@Override
	public void syncToClient(EntityPlayerMP player) {
		World worldObj = getWorldObj();
		if (!worldObj.field_72995_K) {
			IClimateState climateState = getState(true, false);
			if (locationProvider instanceof Entity) {
				NetworkUtil.sendToPlayer(new PacketClimateListenerUpdateEntity((Entity) locationProvider, climateState), player);
			} else {
				NetworkUtil.sendToPlayer(new PacketClimateListenerUpdate(getCoordinates(), climateState), player);
			}
		}
	}

	@Override
	public BlockPos getCoordinates() {
		if (this.pos == null) {
			initLocation();
		}
		return this.pos;
	}

	@Override
	public World getWorldObj() {
		if (this.world == null) {
			initLocation();
		}
		return this.world;
	}

	@Override
	public void markLocatableDirty() {
		this.world = null;
		this.pos = null;
		World worldObj = getWorldObj();
		if (!worldObj.field_72995_K) {
			updateState(true);
		}
	}

	private void initLocation() {
		if ((this.locationProvider instanceof ILocatable)) {
			ILocatable provider = (ILocatable) this.locationProvider;
			this.world = provider.getWorldObj();
			this.pos = provider.getCoordinates();
		} else if ((this.locationProvider instanceof TileEntity)) {
			TileEntity provider = (TileEntity) this.locationProvider;
			this.world = provider.func_145831_w();
			this.pos = provider.func_174877_v();
		} else {
			throw new IllegalStateException("no / incompatible location provider");
		}
	}
}
