/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.apiculture.worldgen;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Random;

import net.minecraft.block.Block;
import net.minecraft.block.state.IBlockState;
import net.minecraft.init.Blocks;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;
import net.minecraft.world.biome.Biome;

import net.minecraftforge.common.BiomeDictionary;

import forestry.api.apiculture.IBeeGenome;
import forestry.api.apiculture.hives.HiveManager;
import forestry.api.apiculture.hives.IHiveDescription;
import forestry.api.apiculture.hives.IHiveGen;
import forestry.api.apiculture.hives.IHiveRegistry;
import forestry.api.core.BiomeHelper;
import forestry.api.core.EnumHumidity;
import forestry.api.core.EnumTemperature;
import forestry.api.genetics.AlleleManager;
import forestry.api.genetics.EnumTolerance;
import forestry.apiculture.HiveConfig;
import forestry.apiculture.ModuleApiculture;
import forestry.apiculture.genetics.BeeDefinition;
import forestry.core.config.Constants;

public enum HiveDescription implements IHiveDescription {

	FOREST(IHiveRegistry.HiveType.FOREST, 3.0f, BeeDefinition.FOREST, HiveManager.genHelper.tree()) {
		@Override
		public void postGen(World world, Random rand, BlockPos pos) {
			super.postGen(world, rand, pos);
			postGenFlowers(world, rand, pos, flowerStates);
		}
	},
	MEADOWS(IHiveRegistry.HiveType.MEADOWS, 1.0f, BeeDefinition.MEADOWS, HiveManager.genHelper.ground(Blocks.field_150346_d, Blocks.field_150349_c)) {
		@Override
		public void postGen(World world, Random rand, BlockPos pos) {
			super.postGen(world, rand, pos);
			postGenFlowers(world, rand, pos, flowerStates);
		}
	},
	DESERT(IHiveRegistry.HiveType.DESERT, 1.0f, BeeDefinition.MODEST, HiveManager.genHelper.ground(Blocks.field_150346_d, Blocks.field_150349_c, Blocks.field_150354_m, Blocks.field_150322_A)) {
		@Override
		public void postGen(World world, Random rand, BlockPos pos) {
			super.postGen(world, rand, pos);
			postGenFlowers(world, rand, pos, cactusStates);
		}
	},
	JUNGLE(IHiveRegistry.HiveType.JUNGLE, 6.0f, BeeDefinition.TROPICAL, HiveManager.genHelper.tree()),
	END(IHiveRegistry.HiveType.END, 2.0f, BeeDefinition.ENDED, HiveManager.genHelper.ground(Blocks.field_150377_bs, Blocks.field_185772_cY)) {
		@Override
		public boolean isGoodBiome(Biome biome) {
			return BiomeDictionary.hasType(biome, BiomeDictionary.Type.END);
		}
	},
	SNOW(IHiveRegistry.HiveType.SNOW, 2.0f, BeeDefinition.WINTRY, HiveManager.genHelper.ground(Blocks.field_150346_d, Blocks.field_150349_c, Blocks.field_150433_aE)) {
		@Override
		public void postGen(World world, Random rand, BlockPos pos) {
			BlockPos posAbove = pos.func_177984_a();
			if (world.func_175623_d(posAbove)) {
				world.func_180501_a(posAbove, Blocks.field_150431_aC.func_176223_P(), Constants.FLAG_BLOCK_SYNC);
			}

			postGenFlowers(world, rand, pos, flowerStates);
		}
	},
	SWAMP(IHiveRegistry.HiveType.SWAMP, 2.0f, BeeDefinition.MARSHY, HiveManager.genHelper.ground(Blocks.field_150346_d, Blocks.field_150349_c)) {
		@Override
		public void postGen(World world, Random rand, BlockPos pos) {
			super.postGen(world, rand, pos);

			postGenFlowers(world, rand, pos, mushroomStates);
		}
	};

	private static final IHiveGen groundGen = HiveManager.genHelper.ground(Blocks.field_150346_d, Blocks.field_150349_c, Blocks.field_150433_aE, Blocks.field_150354_m, Blocks.field_150322_A);
	private static final List<IBlockState> flowerStates = new ArrayList<>();
	private static final List<IBlockState> mushroomStates = new ArrayList<>();
	private static final List<IBlockState> cactusStates = Collections.singletonList(Blocks.field_150434_aF.func_176223_P());

	static {
		flowerStates.addAll(Blocks.field_150328_O.func_176194_O().func_177619_a());
		flowerStates.addAll(Blocks.field_150327_N.func_176194_O().func_177619_a());
		mushroomStates.add(Blocks.field_150337_Q.func_176223_P());
		mushroomStates.add(Blocks.field_150338_P.func_176223_P());
	}

	private final IBlockState blockState;
	private final float genChance;
	private final IBeeGenome beeGenome;
	private final IHiveGen hiveGen;
	private final IHiveRegistry.HiveType hiveType;

	HiveDescription(IHiveRegistry.HiveType hiveType, float genChance, BeeDefinition beeTemplate, IHiveGen hiveGen) {
		this.blockState = ModuleApiculture.getBlocks().beehives.getStateForType(hiveType);
		this.genChance = genChance;
		this.beeGenome = beeTemplate.getGenome();
		this.hiveGen = hiveGen;
		this.hiveType = hiveType;
	}

	@Override
	public IHiveGen getHiveGen() {
		return hiveGen;
	}

	@Override
	public IBlockState getBlockState() {
		return blockState;
	}

	@Override
	public boolean isGoodBiome(Biome biome) {
		return !BiomeHelper.isBiomeHellish(biome) && !HiveConfig.isBlacklisted(hiveType, biome);
	}

	@Override
	public boolean isGoodHumidity(EnumHumidity humidity) {
		EnumHumidity idealHumidity = beeGenome.getPrimary().getHumidity();
		EnumTolerance humidityTolerance = beeGenome.getToleranceHumid();
		return AlleleManager.climateHelper.isWithinLimits(humidity, idealHumidity, humidityTolerance);
	}

	@Override
	public boolean isGoodTemperature(EnumTemperature temperature) {
		EnumTemperature idealTemperature = beeGenome.getPrimary().getTemperature();
		EnumTolerance temperatureTolerance = beeGenome.getToleranceTemp();
		return AlleleManager.climateHelper.isWithinLimits(temperature, idealTemperature, temperatureTolerance);
	}

	@Override
	public float getGenChance() {
		return genChance;
	}

	@Override
	public void postGen(World world, Random rand, BlockPos pos) {}

	protected static void postGenFlowers(World world, Random rand, BlockPos hivePos, List<IBlockState> flowerStates) {
		int plantedCount = 0;
		for (int i = 0; i < 10; i++) {
			int xOffset = rand.nextInt(8) - 4;
			int zOffset = rand.nextInt(8) - 4;
			BlockPos blockPos = hivePos.func_177982_a(xOffset, 0, zOffset);
			if ((xOffset == 0 && zOffset == 0) || !world.func_175667_e(blockPos)) {
				continue;
			}

			blockPos = groundGen.getPosForHive(world, blockPos.func_177958_n(), blockPos.func_177952_p());
			if (blockPos == null) {
				continue;
			}

			IBlockState state = flowerStates.get(rand.nextInt(flowerStates.size()));
			Block block = state.func_177230_c();
			if (!block.func_176196_c(world, blockPos)) {
				continue;
			}

			world.func_175656_a(blockPos, state);
			plantedCount++;

			if (plantedCount >= 3) {
				break;
			}
		}
	}
}
