package forestry.energy;

import javax.annotation.Nullable;
import java.io.IOException;

import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.math.MathHelper;

import net.minecraftforge.common.capabilities.Capability;
import net.minecraftforge.energy.CapabilityEnergy;
import net.minecraftforge.energy.EnergyStorage;
import net.minecraftforge.energy.IEnergyStorage;

import forestry.api.core.INbtReadable;
import forestry.api.core.INbtWritable;
import forestry.core.network.IStreamable;
import forestry.core.network.PacketBufferForestry;
import forestry.energy.compat.EnergyStorageWrapper;
import forestry.energy.compat.tesla.TeslaConsumerWrapper;
import forestry.energy.compat.tesla.TeslaHelper;
import forestry.energy.compat.tesla.TeslaHolderWrapper;
import forestry.energy.compat.tesla.TeslaProducerWrapper;

import net.darkhax.tesla.api.ITeslaConsumer;
import net.darkhax.tesla.api.ITeslaHolder;
import net.darkhax.tesla.api.ITeslaProducer;

public class EnergyManager extends EnergyStorage implements IStreamable, INbtReadable, INbtWritable {
	private EnergyTransferMode externalMode = EnergyTransferMode.BOTH;

	public EnergyManager(int maxTransfer, int capacity) {
		super(EnergyHelper.scaleForDifficulty(capacity), EnergyHelper.scaleForDifficulty(maxTransfer), EnergyHelper.scaleForDifficulty(maxTransfer));
	}

	public void setExternalMode(EnergyTransferMode externalMode) {
		this.externalMode = externalMode;
	}

	public EnergyTransferMode getExternalMode() {
		return externalMode;
	}

	@Override
	public void readFromNBT(NBTTagCompound nbt) {
		final int energy;
		if (nbt.func_74764_b("EnergyManager")) { // legacy
			NBTTagCompound energyManagerNBT = nbt.func_74775_l("EnergyManager");
			NBTTagCompound energyStorageNBT = energyManagerNBT.func_74775_l("EnergyStorage");
			energy = energyStorageNBT.func_74762_e("Energy");
		} else {
			energy = nbt.func_74762_e("Energy");
		}

		setEnergyStored(energy);
	}

	@Override
	public NBTTagCompound writeToNBT(NBTTagCompound nbt) {
		nbt.func_74768_a("Energy", energy);
		return nbt;
	}

	@Override
	public void writeData(PacketBufferForestry data) {
		data.func_150787_b(this.energy);
	}

	@Override
	public void readData(PacketBufferForestry data) {
		int energyStored = data.func_150792_a();
		setEnergyStored(energyStored);
	}

	public int getMaxEnergyReceived() {
		return this.maxReceive;
	}

	/**
	 * Drains an amount of energy, due to decay from lack of work or other factors
	 */
	public void drainEnergy(int amount) {
		setEnergyStored(energy - amount);
	}

	/**
	 * Creates an amount of energy, generated by engines
	 */
	public void generateEnergy(int amount) {
		setEnergyStored(energy + amount);
	}

	public void setEnergyStored(int energyStored) {
		this.energy = energyStored;
		if (this.energy > capacity) {
			this.energy = capacity;
		} else if (this.energy < 0) {
			this.energy = 0;
		}
	}

	public boolean hasCapability(Capability<?> capability) {
		return capability == CapabilityEnergy.ENERGY ||
				capability == TeslaHelper.TESLA_PRODUCER && externalMode.canExtract() ||
				capability == TeslaHelper.TESLA_CONSUMER && externalMode.canReceive() ||
				capability == TeslaHelper.TESLA_HOLDER;
	}

	@Nullable
	public <T> T getCapability(Capability<T> capability) {
		if (capability == CapabilityEnergy.ENERGY) {
			IEnergyStorage energyStorage = new EnergyStorageWrapper(this, externalMode);
			return CapabilityEnergy.ENERGY.cast(energyStorage);
		} else {
			Capability<ITeslaProducer> teslaProducer = TeslaHelper.TESLA_PRODUCER;
			Capability<ITeslaConsumer> teslaConsumer = TeslaHelper.TESLA_CONSUMER;
			Capability<ITeslaHolder> teslaHolder = TeslaHelper.TESLA_HOLDER;

			if (capability == teslaProducer && externalMode.canExtract()) {
				return teslaProducer.cast(new TeslaProducerWrapper(this));
			} else if (capability == teslaConsumer && externalMode.canReceive()) {
				return teslaConsumer.cast(new TeslaConsumerWrapper(this));
			} else if (capability == teslaHolder) {
				return teslaHolder.cast(new TeslaHolderWrapper(this));
			} else {
				return null;
			}
		}
	}

	public int calculateRedstone(){
		return MathHelper.func_76141_d(((float)energy / (float)capacity) * 14.0F) + (energy > 0 ? 1 : 0);
	}

}
