/*******************************************************************************
 * Copyright 2011-2014 SirSengir
 *
 * This work (the API) is licensed under the "MIT" License, see LICENSE.txt for details.
 ******************************************************************************/
package forestry.api.multiblock;

import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.NetworkManager;
import net.minecraft.network.play.server.SPacketUpdateTileEntity;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.math.BlockPos;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

/**
 * Base logic class for Multiblock-connected tile entities.
 * Most multiblock components should derive from this.
 * Supply it an IMultiblockLogic from MultiblockManager.logicFactory
 */
public abstract class MultiblockTileEntityBase<T extends IMultiblockLogic> extends TileEntity implements IMultiblockComponent {
	private final T multiblockLogic;

	public MultiblockTileEntityBase(T multiblockLogic) {
		this.multiblockLogic = multiblockLogic;
	}

	@Override
	public BlockPos getCoordinates() {
		return func_174877_v();
	}

	@Override
	public T getMultiblockLogic() {
		return multiblockLogic;
	}

	@Override
	public abstract void onMachineAssembled(IMultiblockController multiblockController, BlockPos minCoord, BlockPos maxCoord);

	@Override
	public abstract void onMachineBroken();

	@Override
	public void func_145839_a(NBTTagCompound data) {
		super.func_145839_a(data);
		multiblockLogic.readFromNBT(data);
	}

	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound data) {
		data = super.func_189515_b(data);
		multiblockLogic.writeToNBT(data);
		return data;
	}

	@Override
	public final void func_145843_s() {
		super.func_145843_s();
		multiblockLogic.invalidate(field_145850_b, this);
	}

	@Override
	public final void onChunkUnload() {
		super.onChunkUnload();
		multiblockLogic.onChunkUnload(field_145850_b, this);
	}

	@Override
	public final void func_145829_t() {
		super.func_145829_t();
		multiblockLogic.validate(field_145850_b, this);
	}

	/* Network Communication */

	@Override
	public SPacketUpdateTileEntity func_189518_D_() {
		return new SPacketUpdateTileEntity(func_174877_v(), 0, func_189517_E_());
	}

	@Override
	public NBTTagCompound func_189517_E_() {
		NBTTagCompound updateTag = super.func_189517_E_();
		multiblockLogic.encodeDescriptionPacket(updateTag);
		this.encodeDescriptionPacket(updateTag);
		return updateTag;
	}

	@Override
	@SideOnly(Side.CLIENT)
	public final void onDataPacket(NetworkManager network, SPacketUpdateTileEntity packet) {
		super.onDataPacket(network, packet);
		NBTTagCompound nbtData = packet.func_148857_g();
		multiblockLogic.decodeDescriptionPacket(nbtData);
		this.decodeDescriptionPacket(nbtData);
	}

	@Override
	public void handleUpdateTag(NBTTagCompound tag) {
		super.handleUpdateTag(tag);
		multiblockLogic.decodeDescriptionPacket(tag);
		this.decodeDescriptionPacket(tag);
	}

	/**
	 * Used to write tileEntity-specific data to the descriptionPacket
	 */
	protected void encodeDescriptionPacket(NBTTagCompound packetData) {

	}

	/**
	 * Used to read tileEntity-specific data from the descriptionPacket (onDataPacket)
	 */
	protected void decodeDescriptionPacket(NBTTagCompound packetData) {

	}
}
