/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.greenhouse.tiles;

import java.io.IOException;

import net.minecraft.block.Block;
import net.minecraft.client.gui.inventory.GuiContainer;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.Container;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;

import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import forestry.api.circuits.ChipsetManager;
import forestry.api.circuits.CircuitSocketType;
import forestry.api.circuits.ICircuitBoard;
import forestry.api.circuits.ICircuitSocketType;
import forestry.api.core.ICamouflageHandler;
import forestry.api.core.ICamouflagedTile;
import forestry.core.circuits.ISocketable;
import forestry.core.inventory.InventoryAdapter;
import forestry.core.network.IStreamableGui;
import forestry.core.network.PacketBufferForestry;
import forestry.core.network.packets.PacketActiveUpdate;
import forestry.core.tiles.IActivatable;
import forestry.core.tiles.TileForestry;
import forestry.core.utils.ItemStackUtil;
import forestry.core.utils.NetworkUtil;
import forestry.greenhouse.api.climate.IClimateSource;
import forestry.greenhouse.api.climate.IClimateSourceOwner;
import forestry.greenhouse.blocks.BlockClimatiser;
import forestry.greenhouse.camouflage.CamouflageHandlerType;
import forestry.greenhouse.climate.ClimateSource;
import forestry.greenhouse.climate.ClimateSourceClimatiser;
import forestry.greenhouse.climate.ClimateSourceType;
import forestry.greenhouse.gui.ContainerClimatiser;
import forestry.greenhouse.gui.GuiClimatiser;
import forestry.greenhouse.multiblock.GreenhouseController;
import forestry.greenhouse.network.packets.PacketCamouflageSelectionServer;

public class TileClimatiser extends TileForestry implements IActivatable, IStreamableGui, IClimateSourceOwner, ICamouflagedTile, ICamouflageHandler, ISocketable {

	public static final ClimatiserDefinition HEATER = new ClimatiserDefinition(0.075F, 2.5F, ClimateSourceType.TEMPERATURE);
	public static final ClimatiserDefinition FAN = new ClimatiserDefinition(-0.075F, 2.5F, ClimateSourceType.TEMPERATURE);
	public static final ClimatiserDefinition HUMIDIFIER = new ClimatiserDefinition(0.075F, 2.5F, ClimateSourceType.HUMIDITY);
	public static final ClimatiserDefinition DEHUMIDIFIER = new ClimatiserDefinition(-0.075F, 2.5F, ClimateSourceType.HUMIDITY);

	private final InventoryAdapter sockets = new InventoryAdapter(1, "sockets");
	private final ClimateSource source;
	private ItemStack camouflageBlock;
	private boolean active;

	public TileClimatiser(ClimatiserDefinition definition) {
		this(definition.change, definition.boundModifier, definition.type);
	}

	protected TileClimatiser(float change, float range, ClimateSourceType type) {
		this(new ClimateSourceClimatiser(type, change, range));
	}

	protected TileClimatiser(ClimateSource source) {
		this.camouflageBlock = getDefaultCamouflageBlock();
		this.source = source;
		this.source.setOwner(this);
	}

	@Override
	public ItemStack getCamouflageBlock() {
		return camouflageBlock;
	}

	@Override
	public ItemStack getDefaultCamouflageBlock() {
		return GreenhouseController.createDefaultCamouflageBlock();
	}

	/* ICamouflageHandler */
	@Override
	public boolean setCamouflageBlock(ItemStack camouflageBlock, boolean sendClientUpdate) {
		if (!ItemStackUtil.isIdenticalItem(camouflageBlock, this.camouflageBlock)) {
			this.camouflageBlock = camouflageBlock;

			if (sendClientUpdate && field_145850_b != null && field_145850_b.field_72995_K) {
				NetworkUtil.sendToServer(new PacketCamouflageSelectionServer(this, CamouflageHandlerType.TILE));
				field_145850_b.func_175704_b(field_174879_c, field_174879_c);
				func_70296_d();
			}
			return true;
		}
		return false;
	}

	/* IActivatable */
	@Override
	public boolean isActive() {
		return active;
	}

	@Override
	public void setActive(boolean active) {
		if (this.active == active) {
			return;
		}

		this.active = active;

		if (field_145850_b != null) {
			if (field_145850_b.field_72995_K) {
				field_145850_b.func_175704_b(getCoordinates(), getCoordinates());
			} else {
				NetworkUtil.sendNetworkPacket(new PacketActiveUpdate(this), getCoordinates(), field_145850_b);
			}
		}
	}

	@Override
	public void onChunkUnload() {
		source.onChunkUnload();

		super.onChunkUnload();
	}

	@Override
	public void func_145843_s() {
		source.invalidate();

		super.func_145843_s();
	}

	@Override
	protected void updateServerSide() {
		source.update();

		super.updateServerSide();
	}

	/* SAVING & LOADING */
	@Override
	public void func_145839_a(NBTTagCompound data) {
		super.func_145839_a(data);

		if (data.func_74764_b("Camouflage")) {
			camouflageBlock = new ItemStack(data.func_74775_l("Camouflage"));
		}
		setActive(data.func_74767_n("Active"));

		sockets.readFromNBT(data);

		ItemStack chip = sockets.func_70301_a(0);
		if (!chip.func_190926_b()) {
			ICircuitBoard chipset = ChipsetManager.circuitRegistry.getCircuitBoard(chip);
			if (chipset != null) {
				chipset.onLoad(this);
			}
		}

		source.readFromNBT(data);
	}

	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound data) {
		data = super.func_189515_b(data);
		if (!camouflageBlock.func_190926_b()) {
			NBTTagCompound nbtTag = new NBTTagCompound();
			camouflageBlock.func_77955_b(nbtTag);
			data.func_74782_a("Camouflage", nbtTag);
		}
		data.func_74757_a("Active", active);

		sockets.writeToNBT(data);

		source.writeToNBT(data);
		return data;
	}

	@Override
	public NBTTagCompound func_189517_E_() {
		return func_189515_b(new NBTTagCompound());
	}

	@Override
	public String getUnlocalizedTitle() {
		Block block = func_145838_q();
		String blockUnlocalizedName = block.func_149739_a();
		if (block instanceof BlockClimatiser) {
			blockUnlocalizedName += '.' + ((BlockClimatiser) block).getNameFromMeta(func_145832_p());
		}
		return blockUnlocalizedName + ".name";
	}

	@Override
	public IClimateSource getClimateSource() {
		return source;
	}

	@Override
	public boolean isCircuitable() {
		return true;
	}

	@SideOnly(Side.CLIENT)
	@Override
	public GuiContainer getGui(EntityPlayer player, int data) {
		return new GuiClimatiser(player, this);
	}

	@Override
	public Container getContainer(EntityPlayer player, int data) {
		return new ContainerClimatiser(player.field_71071_by, this);
	}

	@Override
	public void writeGuiData(PacketBufferForestry data) {
		sockets.writeData(data);
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void readGuiData(PacketBufferForestry data) throws IOException {
		sockets.readData(data);
	}

	/* ISocketable */
	@Override
	public int getSocketCount() {
		return sockets.func_70302_i_();
	}

	@Override
	public ItemStack getSocket(int slot) {
		return sockets.func_70301_a(slot);
	}

	@Override
	public void setSocket(int slot, ItemStack stack) {
		if (!stack.func_190926_b() && !ChipsetManager.circuitRegistry.isChipset(stack)) {
			return;
		}

		// Dispose correctly of old chipsets
		if (!sockets.func_70301_a(slot).func_190926_b()) {
			if (ChipsetManager.circuitRegistry.isChipset(sockets.func_70301_a(slot))) {
				ICircuitBoard chipset = ChipsetManager.circuitRegistry.getCircuitBoard(sockets.func_70301_a(slot));
				if (chipset != null) {
					chipset.onRemoval(this);
				}
			}
		}

		sockets.func_70299_a(slot, stack);
		if (stack.func_190926_b()) {
			return;
		}

		ICircuitBoard chipset = ChipsetManager.circuitRegistry.getCircuitBoard(stack);
		if (chipset != null) {
			chipset.onInsertion(this);
		}
	}

	@Override
	public ICircuitSocketType getSocketType() {
		return CircuitSocketType.GREENHOUSE_CLIMATISER;
	}

	public static class ClimatiserDefinition {
		float change, boundModifier;
		ClimateSourceType type;

		public ClimatiserDefinition(float change, float boundModifier, ClimateSourceType type) {
			this.change = change;
			this.boundModifier = boundModifier;
			this.type = type;
		}
	}
}
