/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.core.tiles;

import com.google.common.base.Preconditions;

import javax.annotation.Nullable;
import java.io.IOException;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.ISidedInventory;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.play.server.SPacketUpdateTileEntity;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.ITickable;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.text.ITextComponent;
import net.minecraft.util.text.TextComponentTranslation;
import net.minecraft.world.World;

import net.minecraftforge.common.capabilities.Capability;
import net.minecraftforge.items.CapabilityItemHandler;
import net.minecraftforge.items.wrapper.InvWrapper;
import net.minecraftforge.items.wrapper.SidedInvWrapper;

import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import forestry.api.core.IErrorLogic;
import forestry.api.core.IErrorLogicSource;
import forestry.api.core.ILocatable;
import forestry.core.errors.ErrorLogic;
import forestry.core.gui.IGuiHandlerTile;
import forestry.core.inventory.FakeInventoryAdapter;
import forestry.core.inventory.IInventoryAdapter;
import forestry.core.network.IStreamable;
import forestry.core.network.PacketBufferForestry;
import forestry.core.network.packets.PacketTileStream;
import forestry.core.utils.NBTUtilForestry;
import forestry.core.utils.NetworkUtil;
import forestry.core.utils.TickHelper;

//@Optional.Interface(iface = "buildcraft.api.statements.ITriggerProvider", modid = "BuildCraftAPI|statements")
public abstract class TileForestry extends TileEntity implements IStreamable, IErrorLogicSource, ISidedInventory, IFilterSlotDelegate, ITitled, ILocatable, IGuiHandlerTile, ITickable {
	private final ErrorLogic errorHandler = new ErrorLogic();
	private final AdjacentTileCache tileCache = new AdjacentTileCache(this);

	private IInventoryAdapter inventory = FakeInventoryAdapter.instance();

	private final TickHelper tickHelper = new TickHelper();
	private boolean needsNetworkUpdate = false;

	protected AdjacentTileCache getTileCache() {
		return tileCache;
	}

	public void onNeighborTileChange(World world, BlockPos pos, BlockPos neighbor) {
		tileCache.onNeighborChange();
	}

	@Override
	public void func_145843_s() {
		tileCache.purge();
		super.func_145843_s();
	}

	@Override
	public void func_145829_t() {
		tileCache.purge();
		super.func_145829_t();
	}

	// / UPDATING
	@Override
	public final void func_73660_a() {
		tickHelper.onTick();

		if (!field_145850_b.field_72995_K) {
			updateServerSide();
		} else {
			updateClientSide();
		}

		if (needsNetworkUpdate) {
			needsNetworkUpdate = false;
			sendNetworkUpdate();
		}
	}

	@SideOnly(Side.CLIENT)
	protected void updateClientSide() {
	}

	protected void updateServerSide() {
	}

	protected final boolean updateOnInterval(int tickInterval) {
		return tickHelper.updateOnInterval(tickInterval);
	}

	// / SAVING & LOADING
	@Override
	public void func_145839_a(NBTTagCompound data) {
		super.func_145839_a(data);
		inventory.readFromNBT(data);
	}


	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound data) {
		data = super.func_189515_b(data);
		inventory.writeToNBT(data);
		return data;
	}

	@Nullable
	@Override
	public SPacketUpdateTileEntity func_189518_D_() {
		return new SPacketUpdateTileEntity(this.func_174877_v(), 0, func_189517_E_());
	}


	@Override
	public NBTTagCompound func_189517_E_() {
		NBTTagCompound tag = super.func_189517_E_();
		return NBTUtilForestry.writeStreamableToNbt(this, tag);
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void handleUpdateTag(NBTTagCompound tag) {
		super.handleUpdateTag(tag);
		NBTUtilForestry.readStreamableFromNbt(this, tag);
	}

	/* INetworkedEntity */
	protected final void sendNetworkUpdate() {
		PacketTileStream packet = new PacketTileStream(this);
		NetworkUtil.sendNetworkPacket(packet, field_174879_c, field_145850_b);
	}

	/* IStreamable */
	@Override
	public void writeData(PacketBufferForestry data) {

	}

	@Override
	@SideOnly(Side.CLIENT)
	public void readData(PacketBufferForestry data) throws IOException {

	}

	public void onRemoval() {
	}

	@Override
	public World getWorldObj() {
		return field_145850_b;
	}

	/* ITriggerProvider */
	// TODO: buildcraft for 1.9
	//	@Optional.Method(modid = "BuildCraftAPI|statements")
	//	@Override
	//	public Collection<ITriggerInternal> getInternalTriggers(IStatementContainer container) {
	//		return null;
	//	}
	//
	//	@Optional.Method(modid = "BuildCraftAPI|statements")
	//	@Override
	//	public Collection<ITriggerExternal> getExternalTriggers(EnumFacing side, TileEntity tile) {
	//		return null;
	//	}

	// / REDSTONE INFO
	protected boolean isRedstoneActivated() {
		return field_145850_b.func_175687_A(func_174877_v()) > 0;
	}

	protected final void setNeedsNetworkUpdate() {
		needsNetworkUpdate = true;
	}

	@Override
	public final IErrorLogic getErrorLogic() {
		return errorHandler;
	}

	/* NAME */

	/**
	 * Gets the tile's unlocalized name, based on the block at the location of this entity (client-only).
	 */
	@Override
	public String getUnlocalizedTitle() {
		String blockUnlocalizedName = func_145838_q().func_149739_a();
		return blockUnlocalizedName + '.' + func_145832_p() + ".name";
	}

	/* INVENTORY BASICS */
	public IInventoryAdapter getInternalInventory() {
		return inventory;
	}

	protected final void setInternalInventory(IInventoryAdapter inv) {
		Preconditions.checkNotNull(inv);
		this.inventory = inv;
	}

	/* ISidedInventory */

	@Override
	public boolean func_191420_l() {
		return getInternalInventory().func_191420_l();
	}

	@Override
	public final int func_70302_i_() {
		return getInternalInventory().func_70302_i_();
	}

	@Override
	public final ItemStack func_70301_a(int slotIndex) {
		return getInternalInventory().func_70301_a(slotIndex);
	}

	@Override
	public ItemStack func_70298_a(int slotIndex, int amount) {
		return getInternalInventory().func_70298_a(slotIndex, amount);
	}

	@Override
	public ItemStack func_70304_b(int slotIndex) {
		return getInternalInventory().func_70304_b(slotIndex);
	}

	@Override
	public void func_70299_a(int slotIndex, ItemStack itemstack) {
		getInternalInventory().func_70299_a(slotIndex, itemstack);
	}

	@Override
	public final int func_70297_j_() {
		return getInternalInventory().func_70297_j_();
	}

	@Override
	public final void func_174889_b(EntityPlayer player) {
		getInternalInventory().func_174889_b(player);
	}

	@Override
	public final void func_174886_c(EntityPlayer player) {
		getInternalInventory().func_174886_c(player);
	}

	@Override
	public String func_70005_c_() {
		return getUnlocalizedTitle();
	}

	@Override
	public ITextComponent func_145748_c_() {
		return new TextComponentTranslation(getUnlocalizedTitle());
	}

	@Override
	public final boolean func_70300_a(EntityPlayer player) {
		return getInternalInventory().func_70300_a(player);
	}

	@Override
	public boolean func_145818_k_() {
		return getInternalInventory().func_145818_k_();
	}

	@Override
	public final boolean func_94041_b(int slotIndex, ItemStack itemStack) {
		return getInternalInventory().func_94041_b(slotIndex, itemStack);
	}

	@Override
	public final boolean canSlotAccept(int slotIndex, ItemStack itemStack) {
		return getInternalInventory().canSlotAccept(slotIndex, itemStack);
	}

	@Override
	public boolean isLocked(int slotIndex) {
		return getInternalInventory().isLocked(slotIndex);
	}

	@Override
	public int[] func_180463_a(EnumFacing side) {
		return getInternalInventory().func_180463_a(side);
	}

	@Override
	public final boolean func_180462_a(int slotIndex, ItemStack itemStack, EnumFacing side) {
		return getInternalInventory().func_180462_a(slotIndex, itemStack, side);
	}

	@Override
	public final boolean func_180461_b(int slotIndex, ItemStack itemStack, EnumFacing side) {
		return getInternalInventory().func_180461_b(slotIndex, itemStack, side);
	}

	@Override
	public final BlockPos getCoordinates() {
		return func_174877_v();
	}

	@Override
	public int func_174887_a_(int id) {
		return 0;
	}

	@Override
	public int func_174890_g() {
		return 0;
	}

	@Override
	public void func_174885_b(int id, int value) {
	}

	@Override
	public void func_174888_l() {
	}

	@Override
	@Nullable
	public <T> T getCapability(Capability<T> capability, @Nullable EnumFacing facing) {
		if (capability == CapabilityItemHandler.ITEM_HANDLER_CAPABILITY) {
			if (facing != null) {
				SidedInvWrapper sidedInvWrapper = new SidedInvWrapper(inventory, facing);
				return CapabilityItemHandler.ITEM_HANDLER_CAPABILITY.cast(sidedInvWrapper);
			} else {
				InvWrapper invWrapper = new InvWrapper(inventory);
				return CapabilityItemHandler.ITEM_HANDLER_CAPABILITY.cast(invWrapper);
			}

		}
		return super.getCapability(capability, facing);
	}

	@Override
	public boolean hasCapability(Capability<?> capability, @Nullable EnumFacing facing) {
		return capability == CapabilityItemHandler.ITEM_HANDLER_CAPABILITY ||
				super.hasCapability(capability, facing);
	}
}
