/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.core.blocks;

import java.util.Random;

import net.minecraft.block.Block;
import net.minecraft.block.SoundType;
import net.minecraft.block.material.Material;
import net.minecraft.block.properties.PropertyInteger;
import net.minecraft.block.state.BlockStateContainer;
import net.minecraft.block.state.IBlockState;
import net.minecraft.creativetab.CreativeTabs;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.init.Blocks;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.IStringSerializable;
import net.minecraft.util.NonNullList;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;

import net.minecraftforge.common.IPlantable;

import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import forestry.api.core.IItemModelRegister;
import forestry.api.core.IModelManager;
import forestry.core.CreativeTabForestry;
import forestry.core.ModuleCore;
import forestry.core.config.Constants;

/**
 * bog earth, which becomes peat
 */
public class BlockBogEarth extends Block implements IItemModelRegister, IBlockWithMeta {
	private static final int maturityDelimiter = 3; //maturity at which bogEarth becomes peat
	public static final PropertyInteger MATURITY = PropertyInteger.func_177719_a("maturity", 0, maturityDelimiter);

	public enum SoilType implements IStringSerializable {
		BOG_EARTH("bog_earth"),
		PEAT("peat");


		private final String name;

		SoilType(String name) {
			this.name = name;
		}

		@Override
		public String func_176610_l() {
			return name;
		}

		public static SoilType fromMaturity(int maturity) {
			if (maturity >= maturityDelimiter) {
				return PEAT;
			} else {
				return BOG_EARTH;
			}
		}
	}

	public BlockBogEarth() {
		super(Material.field_151578_c);
		func_149675_a(true);
		func_149711_c(0.5f);
		func_149672_a(SoundType.field_185849_b);
		func_149647_a(CreativeTabForestry.tabForestry);

		func_180632_j(this.field_176227_L.func_177621_b().func_177226_a(MATURITY, 0));
	}

	@Override
	public int func_176201_c(IBlockState state) {
		return state.func_177229_b(MATURITY);
	}

	@Override
	public IBlockState func_176203_a(int meta) {
		return func_176223_P().func_177226_a(MATURITY, meta);
	}

	@Override
	protected BlockStateContainer func_180661_e() {
		return new BlockStateContainer(this, MATURITY);
	}

	@Override
	public int func_149738_a(World world) {
		return 500;
	}
	
	@Override
	public void getDrops(NonNullList<ItemStack> drops, IBlockAccess world, BlockPos pos, IBlockState state, int fortune) {
		Integer maturity = state.func_177229_b(MATURITY);
		SoilType type = SoilType.fromMaturity(maturity);

		if (type == SoilType.PEAT) {
			drops.add(ModuleCore.getItems().peat.getItemStack(2));
			drops.add(new ItemStack(Blocks.field_150346_d));
		} else {
			drops.add(new ItemStack(this, 1, SoilType.BOG_EARTH.ordinal()));
		}
	}

	@Override
	public void func_180650_b(World world, BlockPos pos, IBlockState state, Random rand) {
		if (world.field_72995_K || world.field_73012_v.nextInt(13) != 0) {
			return;
		}

		Integer maturity = state.func_177229_b(MATURITY);
		SoilType type = SoilType.fromMaturity(maturity);

		if (type == SoilType.BOG_EARTH) {
			if (isMoistened(world, pos)) {
				world.func_180501_a(pos, state.func_177226_a(MATURITY, maturity + 1), Constants.FLAG_BLOCK_SYNC);
			}
		}
	}

	private static boolean isMoistened(World world, BlockPos pos) {
		for (BlockPos.MutableBlockPos waterPos : BlockPos.func_177975_b(pos.func_177982_a(-2, -2, -2), pos.func_177982_a(2, 2, 2))) {
			IBlockState blockState = world.func_180495_p(waterPos);
			Block block = blockState.func_177230_c();
			if (block == Blocks.field_150355_j || block == Blocks.field_150358_i) {
				return true;
			}
		}

		return false;
	}

	@Override
	public boolean canSustainPlant(IBlockState state, IBlockAccess world, BlockPos pos, EnumFacing direction, IPlantable plantable) {
		return false;
	}

	@Override
	public boolean canSilkHarvest(World world, BlockPos pos, IBlockState state, EntityPlayer player) {
		return false;
	}

	public static SoilType getTypeFromState(IBlockState state) {
		Integer maturity = state.func_177229_b(MATURITY);
		return SoilType.fromMaturity(maturity);
	}

	@Override
	public void func_149666_a(CreativeTabs tab, NonNullList<ItemStack> list) {
		list.add(new ItemStack(this, 1, 0));
	}

	/* MODELS */
	@Override
	@SideOnly(Side.CLIENT)
	public void registerModel(Item item, IModelManager manager) {
		manager.registerItemModel(item, 0, "soil/bog");
		manager.registerItemModel(item, 1, "soil/bog");
		manager.registerItemModel(item, 2, "soil/bog");
		manager.registerItemModel(item, 3, "soil/peat");
	}

	@Override
	public String getNameFromMeta(int meta) {
		SoilType type = SoilType.fromMaturity(meta);
		return type.func_176610_l();
	}

	@Override
	public int func_180651_a(IBlockState state) {
		return 0;
	}

	public ItemStack get(SoilType soilType, int amount) {
		return new ItemStack(this, amount, soilType.ordinal());
	}
}
