/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.energy.tiles;

import java.io.IOException;
import java.util.Collection;

import net.minecraft.client.gui.inventory.GuiContainer;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.Container;
import net.minecraft.inventory.IInventory;
import net.minecraft.inventory.ISidedInventory;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.EnumFacing;

import net.minecraftforge.items.IItemHandler;
import net.minecraftforge.items.wrapper.InvWrapper;

import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import forestry.api.fuels.FuelManager;
import forestry.core.ModuleCore;
import forestry.core.blocks.BlockBase;
import forestry.core.config.Constants;
import forestry.core.errors.EnumErrorCode;
import forestry.core.inventory.AdjacentInventoryCache;
import forestry.core.inventory.IInventoryAdapter;
import forestry.core.inventory.wrappers.InventoryMapper;
import forestry.core.network.PacketBufferForestry;
import forestry.core.tiles.TemperatureState;
import forestry.core.tiles.TileEngine;
import forestry.core.utils.InventoryUtil;
import forestry.energy.gui.ContainerEnginePeat;
import forestry.energy.gui.GuiEnginePeat;
import forestry.energy.inventory.InventoryEnginePeat;

public class TileEnginePeat extends TileEngine implements ISidedInventory {
	private ItemStack fuel = ItemStack.field_190927_a;
	private int burnTime;
	private int totalBurnTime;
	private int ashProduction;
	private final int ashForItem;
	private final AdjacentInventoryCache inventoryCache = new AdjacentInventoryCache(this, getTileCache());

	public TileEnginePeat() {
		super("engine.copper", Constants.ENGINE_COPPER_HEAT_MAX, 200000);

		ashForItem = Constants.ENGINE_COPPER_ASH_FOR_ITEM;
		setInternalInventory(new InventoryEnginePeat(this));
	}

	private int getFuelSlot() {
		IInventoryAdapter inventory = getInternalInventory();
		if (inventory.func_70301_a(InventoryEnginePeat.SLOT_FUEL).func_190926_b()) {
			return -1;
		}

		if (determineFuelValue(inventory.func_70301_a(InventoryEnginePeat.SLOT_FUEL)) > 0) {
			return InventoryEnginePeat.SLOT_FUEL;
		}

		return -1;
	}

	private int getFreeWasteSlot() {
		IInventoryAdapter inventory = getInternalInventory();
		for (int i = InventoryEnginePeat.SLOT_WASTE_1; i <= InventoryEnginePeat.SLOT_WASTE_COUNT; i++) {
			ItemStack waste = inventory.func_70301_a(i);
			if (waste.func_190926_b()) {
				return i;
			}

			if (waste.func_77973_b() != ModuleCore.getItems().ash) {
				continue;
			}

			if (waste.func_190916_E() < waste.func_77976_d()) {
				return i;
			}
		}

		return -1;
	}

	@Override
	public void updateServerSide() {
		super.updateServerSide();

		if (!updateOnInterval(40)) {
			return;
		}

		dumpStash();

		int fuelSlot = getFuelSlot();
		boolean hasFuel = fuelSlot >= 0 && determineBurnDuration(getInternalInventory().func_70301_a(fuelSlot)) > 0;
		getErrorLogic().setCondition(!hasFuel, EnumErrorCode.NO_FUEL);
	}

	@Override
	public void burn() {

		currentOutput = 0;

		if (burnTime > 0) {
			burnTime--;
			addAsh(1);

			if (isRedstoneActivated()) {
				currentOutput = determineFuelValue(fuel);
				energyManager.generateEnergy(currentOutput);
				field_145850_b.func_175666_e(field_174879_c, func_145838_q());
			}
		} else if (isRedstoneActivated()) {
			int fuelSlot = getFuelSlot();
			int wasteSlot = getFreeWasteSlot();

			if (fuelSlot >= 0 && wasteSlot >= 0) {
				IInventoryAdapter inventory = getInternalInventory();
				ItemStack fuelStack = inventory.func_70301_a(fuelSlot);
				burnTime = totalBurnTime = determineBurnDuration(fuelStack);
				if (burnTime > 0 && !fuelStack.func_190926_b()) {
					fuel = fuelStack.func_77946_l();
					func_70298_a(fuelSlot, 1);
				}
			}
		}
	}

	@Override
	public int dissipateHeat() {
		if (heat <= 0) {
			return 0;
		}

		int loss = 0;

		if (!isBurning()) {
			loss += 1;
		}

		TemperatureState tempState = getTemperatureState();
		if (tempState == TemperatureState.OVERHEATING || tempState == TemperatureState.OPERATING_TEMPERATURE) {
			loss += 1;
		}

		heat -= loss;
		return loss;
	}

	@Override
	public int generateHeat() {

		int heatToAdd = 0;

		if (isBurning()) {
			heatToAdd++;
			if ((double) energyManager.getEnergyStored() / (double) energyManager.getMaxEnergyStored() > 0.5) {
				heatToAdd++;
			}
		}

		addHeat(heatToAdd);
		return heatToAdd;
	}

	private void addAsh(int amount) {

		ashProduction += amount;
		if (ashProduction < ashForItem) {
			return;
		}

		// If we have reached the necessary amount, we need to add ash
		int wasteSlot = getFreeWasteSlot();
		if (wasteSlot >= 0) {
			IInventoryAdapter inventory = getInternalInventory();
			ItemStack wasteStack = inventory.func_70301_a(wasteSlot);
			if (wasteStack.func_190926_b()) {
				inventory.func_70299_a(wasteSlot, ModuleCore.getItems().ash.getItemStack());
			} else {
				wasteStack.func_190917_f(1);
			}
		}
		// Reset
		ashProduction = 0;
		// try to dump stash
		dumpStash();
	}

	/**
	 * Returns the fuel value (power per cycle) an item of the passed ItemStack provides
	 */
	private static int determineFuelValue(ItemStack fuel) {
		if (FuelManager.copperEngineFuel.containsKey(fuel)) {
			return FuelManager.copperEngineFuel.get(fuel).getPowerPerCycle();
		} else {
			return 0;
		}
	}

	/**
	 * Returns the fuel value (power per cycle) an item of the passed ItemStack provides
	 */
	private static int determineBurnDuration(ItemStack fuel) {
		if (FuelManager.copperEngineFuel.containsKey(fuel)) {
			return FuelManager.copperEngineFuel.get(fuel).getBurnDuration();
		} else {
			return 0;
		}
	}

	/* AUTO-EJECTING */
	private IInventory getWasteInventory() {
		return new InventoryMapper(this, InventoryEnginePeat.SLOT_WASTE_1, InventoryEnginePeat.SLOT_WASTE_COUNT);
	}

	private void dumpStash() {
		IInventory wasteInventory = getWasteInventory();

		IItemHandler wasteItemHandler = new InvWrapper(wasteInventory);

		if (!InventoryUtil.moveOneItemToPipe(wasteItemHandler, getTileCache())) {
			EnumFacing powerSide = field_145850_b.func_180495_p(func_174877_v()).func_177229_b(BlockBase.FACING);
			Collection<IItemHandler> inventories = inventoryCache.getAdjacentInventoriesOtherThan(powerSide);
			InventoryUtil.moveItemStack(wasteItemHandler, inventories);
		}
	}

	// / STATE INFORMATION
	@Override
	public boolean isBurning() {
		return mayBurn() && burnTime > 0;
	}

	@Override
	public int getBurnTimeRemainingScaled(int i) {
		if (totalBurnTime == 0) {
			return 0;
		}

		return burnTime * i / totalBurnTime;
	}

	@Override
	public boolean hasFuelMin(float percentage) {
		int fuelSlot = this.getFuelSlot();
		if (fuelSlot < 0) {
			return false;
		}

		IInventoryAdapter inventory = getInternalInventory();
		return (float) inventory.func_70301_a(fuelSlot).func_190916_E() / (float) inventory.func_70301_a(fuelSlot).func_77976_d() > percentage;
	}

	// / LOADING AND SAVING
	@Override
	public void func_145839_a(NBTTagCompound nbttagcompound) {
		super.func_145839_a(nbttagcompound);

		if (nbttagcompound.func_74764_b("EngineFuelItemStack")) {
			NBTTagCompound fuelItemNbt = nbttagcompound.func_74775_l("EngineFuelItemStack");
			fuel = new ItemStack(fuelItemNbt);
		}

		burnTime = nbttagcompound.func_74762_e("EngineBurnTime");
		totalBurnTime = nbttagcompound.func_74762_e("EngineTotalTime");
		if (nbttagcompound.func_74764_b("AshProduction")) {
			ashProduction = nbttagcompound.func_74762_e("AshProduction");
		}
	}


	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound nbttagcompound) {
		nbttagcompound = super.func_189515_b(nbttagcompound);

		if (!fuel.func_190926_b()) {
			nbttagcompound.func_74782_a("EngineFuelItemStack", fuel.serializeNBT());
		}

		nbttagcompound.func_74768_a("EngineBurnTime", burnTime);
		nbttagcompound.func_74768_a("EngineTotalTime", totalBurnTime);
		nbttagcompound.func_74768_a("AshProduction", ashProduction);
		return nbttagcompound;
	}

	@Override
	public void writeGuiData(PacketBufferForestry data) {
		super.writeGuiData(data);
		data.writeInt(burnTime);
		data.writeInt(totalBurnTime);
	}

	@Override
	public void readGuiData(PacketBufferForestry data) throws IOException {
		super.readGuiData(data);
		burnTime = data.readInt();
		totalBurnTime = data.readInt();
	}

	/* ITriggerProvider */
	// TODO: buildcraft for 1.9
	//	@Optional.Method(modid = "BuildCraftAPI|statements")
	//	@Override
	//	public Collection<ITriggerExternal> getExternalTriggers(EnumFacing side, TileEntity tile) {
	//		LinkedList<ITriggerExternal> res = new LinkedList<>();
	//		res.add(FactoryTriggers.lowFuel25);
	//		res.add(FactoryTriggers.lowFuel10);
	//		return res;
	//	}

	@Override
	@SideOnly(Side.CLIENT)
	public GuiContainer getGui(EntityPlayer player, int data) {
		return new GuiEnginePeat(player.field_71071_by, this);
	}

	@Override
	public Container getContainer(EntityPlayer player, int data) {
		return new ContainerEnginePeat(player.field_71071_by, this);
	}
}
