/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.storage;

import com.google.common.base.Preconditions;

import javax.annotation.Nullable;
import java.awt.Color;
import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

import net.minecraft.init.Blocks;
import net.minecraft.init.Items;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.util.ResourceLocation;

import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.common.config.Property;
import net.minecraftforge.fluids.Fluid;
import net.minecraftforge.fluids.FluidRegistry;
import net.minecraftforge.fluids.FluidStack;
import net.minecraftforge.oredict.OreDictionary;

import net.minecraftforge.fml.common.event.FMLInterModComms.IMCMessage;

import forestry.Forestry;
import forestry.api.core.ForestryAPI;
import forestry.api.modules.ForestryModule;
import forestry.api.recipes.RecipeManagers;
import forestry.api.storage.BackpackManager;
import forestry.api.storage.IBackpackDefinition;
import forestry.api.storage.IBackpackFilterConfigurable;
import forestry.api.storage.StorageManager;
import forestry.apiculture.ModuleApiculture;
import forestry.apiculture.blocks.BlockRegistryApiculture;
import forestry.core.IPickupHandler;
import forestry.core.IResupplyHandler;
import forestry.core.ModuleCore;
import forestry.core.config.Constants;
import forestry.core.config.LocalizedConfiguration;
import forestry.core.items.ItemRegistryCore;
import forestry.core.recipes.RecipeUtil;
import forestry.core.utils.IMCUtil;
import forestry.core.utils.ItemStackUtil;
import forestry.core.utils.Log;
import forestry.core.utils.OreDictUtil;
import forestry.core.utils.Translator;
import forestry.lepidopterology.ModuleLepidopterology;
import forestry.lepidopterology.blocks.BlockRegistryLepidopterology;
import forestry.modules.BlankForestryModule;
import forestry.modules.ForestryModuleUids;
import forestry.storage.items.ItemRegistryBackpacks;

@ForestryModule(moduleID = ForestryModuleUids.BACKPACKS, containerID = Constants.MOD_ID, name = "Backpack", author = "SirSengir", url = Constants.URL, unlocalizedDescription = "for.module.backpacks.description", lootTable = "storage")
public class ModuleBackpacks extends BlankForestryModule {

	private static final String CONFIG_CATEGORY = "backpacks";

	@Nullable
	private static ItemRegistryBackpacks items;

	private final Map<String, List<String>> backpackAcceptedOreDictRegexpDefaults = new HashMap<>();
	private final Map<String, List<String>> backpackRejectedOreDictRegexpDefaults = new HashMap<>();
	private final Map<String, List<String>> backpackAcceptedItemDefaults = new HashMap<>();
	private final Map<String, List<String>> backpackRejectedItemDefaults = new HashMap<>();

	private final List<String> forestryBackpackUids = Arrays.asList(
			BackpackManager.MINER_UID,
			BackpackManager.DIGGER_UID,
			BackpackManager.FORESTER_UID,
			BackpackManager.HUNTER_UID,
			BackpackManager.ADVENTURER_UID,
			BackpackManager.BUILDER_UID
	);

	public static ItemRegistryBackpacks getItems() {
		Preconditions.checkState(items != null);
		return items;
	}

	@Override
	public void setupAPI() {

		StorageManager.crateRegistry = new CrateRegistry();

		BackpackManager.backpackInterface = new BackpackInterface();

		BackpackDefinition definition;

		if (ForestryAPI.enabledModules.contains(new ResourceLocation(Constants.MOD_ID, ForestryModuleUids.APICULTURE))) {
			Predicate<ItemStack> filter = BackpackManager.backpackInterface.createNaturalistBackpackFilter("rootBees");
			definition = new BackpackDefinition(new Color(0xc4923d), Color.WHITE, filter);
			BackpackManager.backpackInterface.registerBackpackDefinition("apiarist", definition);
		}

		if (ForestryAPI.enabledModules.contains(new ResourceLocation(Constants.MOD_ID, ForestryModuleUids.LEPIDOPTEROLOGY))) {
			Predicate<ItemStack> filter = BackpackManager.backpackInterface.createNaturalistBackpackFilter("rootButterflies");
			definition = new BackpackDefinition(new Color(0x995b31), Color.WHITE, filter);
			BackpackManager.backpackInterface.registerBackpackDefinition("lepidopterist", definition);
		}

		definition = new BackpackDefinition(new Color(0x36187d), Color.WHITE);
		BackpackManager.backpackInterface.registerBackpackDefinition(BackpackManager.MINER_UID, definition);

		definition = new BackpackDefinition(new Color(0x363cc5), Color.WHITE);
		BackpackManager.backpackInterface.registerBackpackDefinition(BackpackManager.DIGGER_UID, definition);

		definition = new BackpackDefinition(new Color(0x347427), Color.WHITE);
		BackpackManager.backpackInterface.registerBackpackDefinition(BackpackManager.FORESTER_UID, definition);

		definition = new BackpackDefinition(new Color(0x412215), Color.WHITE);
		BackpackManager.backpackInterface.registerBackpackDefinition(BackpackManager.HUNTER_UID, definition);

		definition = new BackpackDefinition(new Color(0x7fb8c2), Color.WHITE);
		BackpackManager.backpackInterface.registerBackpackDefinition(BackpackManager.ADVENTURER_UID, definition);

		definition = new BackpackDefinition(new Color(0xdd3a3a), Color.WHITE);
		BackpackManager.backpackInterface.registerBackpackDefinition(BackpackManager.BUILDER_UID, definition);
	}

	@Override
	public void registerItemsAndBlocks() {
		items = new ItemRegistryBackpacks();
	}

	@Override
	public void preInit() {
		registerFenceAndFenceGatesToOreDict();
		MinecraftForge.EVENT_BUS.register(this);
	}

	@Override
	public void postInit() {
		final String newConfig = CONFIG_CATEGORY + ".cfg";

		File configFile = new File(Forestry.instance.getConfigFolder(), newConfig);
		LocalizedConfiguration config = new LocalizedConfiguration(configFile, "2.0.0");
		if (!config.getDefinedConfigVersion().equals(config.getLoadedConfigVersion())) {
			boolean deleted = configFile.delete();
			if (deleted) {
				config = new LocalizedConfiguration(configFile, "2.0.0");
			}
		}

		setDefaultsForConfig();

		for (String backpackUid : forestryBackpackUids) {
			handleBackpackConfig(config, backpackUid);
		}

		config.save();
	}

	private void setDefaultsForConfig() {
		ItemRegistryCore coreItems = ModuleCore.getItems();

		backpackAcceptedOreDictRegexpDefaults.put(BackpackManager.MINER_UID, Arrays.asList(
				"obsidian",
				"ore[A-Z].*",
				"dust[A-Z].*",
				"gem[A-Z].*",
				"ingot[A-Z].*",
				"nugget[A-Z].*",
				"crushed[A-Z].*",
				"cluster[A-Z].*",
				"denseore[A-Z].*"
		));

		backpackAcceptedOreDictRegexpDefaults.put(BackpackManager.DIGGER_UID, Arrays.asList(
				"cobblestone",
				"dirt",
				"gravel",
				"netherrack",
				"stone",
				"stone[A-Z].*",
				"sand"
		));

		backpackAcceptedOreDictRegexpDefaults.put(BackpackManager.HUNTER_UID, Arrays.asList(
				"bone",
				"egg",
				"enderpearl",
				"feather",
				"fish[A-Z].*",
				"gunpowder",
				"leather",
				"slimeball",
				"string"
		));

		backpackAcceptedOreDictRegexpDefaults.put(BackpackManager.FORESTER_UID, Arrays.asList(
				"logWood",
				"stickWood",
				"woodStick",
				"saplingTree",
				"treeSapling",
				"vine",
				"sugarcane",
				"blockCactus",
				"crop[A-Z].*",
				"seed[A-Z].*",
				"tree[A-Z].*"
		));

		backpackAcceptedOreDictRegexpDefaults.put(BackpackManager.BUILDER_UID, Arrays.asList(
				"block[A-Z].*",
				"paneGlass[A-Z].*",
				"slabWood[A-Z].*",
				"stainedClay[A-Z].*",
				"stainedGlass[A-Z].*",
				"stone",
				"sandstone",
				OreDictUtil.PLANK_WOOD,
				OreDictUtil.STAIR_WOOD,
				OreDictUtil.SLAB_WOOD,
				OreDictUtil.FENCE_WOOD,
				OreDictUtil.FENCE_GATE_WOOD,
				OreDictUtil.TRAPDOOR_WOOD,
				"glass",
				"paneGlass",
				"torch",
				"chest",
				"chest[A-Z].*",
				"workbench",
				"doorWood"
		));

		backpackAcceptedItemDefaults.put(BackpackManager.MINER_UID, getItemStrings(Arrays.asList(
				new ItemStack(Blocks.field_150365_q),
				new ItemStack(Items.field_151044_h),
				coreItems.bronzePickaxe.getItemStack(),
				coreItems.kitPickaxe.getItemStack(),
				coreItems.brokenBronzePickaxe.getItemStack()
		)));

		backpackAcceptedItemDefaults.put(BackpackManager.DIGGER_UID, getItemStrings(Arrays.asList(
				new ItemStack(Blocks.field_150346_d, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Items.field_151145_ak),
				new ItemStack(Blocks.field_150322_A, 1, 0),
				new ItemStack(Items.field_151119_aD),
				new ItemStack(Items.field_151126_ay),
				new ItemStack(Blocks.field_150425_aM),
				coreItems.bronzeShovel.getItemStack(),
				coreItems.kitShovel.getItemStack(),
				coreItems.brokenBronzeShovel.getItemStack()
		)));

		backpackAcceptedItemDefaults.put(BackpackManager.FORESTER_UID, getItemStrings(Arrays.asList(
				new ItemStack(Blocks.field_150337_Q),
				new ItemStack(Blocks.field_150338_P),
				new ItemStack(Blocks.field_150328_O, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_150327_N),
				new ItemStack(Blocks.field_150329_H, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_150398_cm, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_150423_aK),
				new ItemStack(Blocks.field_150440_ba),
				new ItemStack(Items.field_151153_ao),
				new ItemStack(Items.field_151075_bm),
				new ItemStack(Items.field_151014_N),
				new ItemStack(Items.field_151080_bb),
				new ItemStack(Items.field_151081_bc),
				new ItemStack(Items.field_185163_cU),
				new ItemStack(Items.field_185164_cV),
				new ItemStack(Items.field_185161_cS),
				new ItemStack(Items.field_151034_e)
		)));

		backpackAcceptedItemDefaults.put(BackpackManager.HUNTER_UID, getItemStrings(Arrays.asList(
				new ItemStack(Items.field_151065_br),
				new ItemStack(Items.field_151072_bj),
				new ItemStack(Items.field_151078_bh),
				new ItemStack(Items.field_151144_bL, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Items.field_151073_bk),
				new ItemStack(Items.field_151074_bl),
				new ItemStack(Items.field_151032_g),
				new ItemStack(Items.field_185166_h),
				new ItemStack(Items.field_185167_i),
				new ItemStack(Items.field_151147_al),
				new ItemStack(Items.field_151157_am),
				new ItemStack(Items.field_151082_bd),
				new ItemStack(Items.field_151083_be),
				new ItemStack(Items.field_151076_bf),
				new ItemStack(Items.field_151077_bg),
				new ItemStack(Items.field_179561_bm),
				new ItemStack(Items.field_179557_bn),
				new ItemStack(Items.field_179558_bo),
				new ItemStack(Items.field_179559_bp),
				new ItemStack(Items.field_179556_br),
				new ItemStack(Items.field_179555_bs),
				new ItemStack(Items.field_151070_bp),
				new ItemStack(Items.field_151071_bq),
				new ItemStack(Items.field_151100_aR, 1, 0),
				new ItemStack(Blocks.field_150407_cf),
				new ItemStack(Blocks.field_150325_L, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Items.field_151061_bv),
				new ItemStack(Items.field_151064_bs),
				new ItemStack(Items.field_151060_bw),
				new ItemStack(Items.field_151115_aP, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Items.field_179566_aV, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Items.field_151058_ca),
				new ItemStack(Items.field_151112_aM),
				new ItemStack(Items.field_151057_cb),
				new ItemStack(Items.field_151141_av),
				new ItemStack(Items.field_151125_bZ),
				new ItemStack(Items.field_151136_bY),
				new ItemStack(Items.field_151138_bX)
		)));

		backpackAcceptedItemDefaults.put(BackpackManager.BUILDER_UID, getItemStrings(Arrays.asList(
				new ItemStack(Blocks.field_150429_aA),
				new ItemStack(Blocks.field_150379_bu),
				new ItemStack(Blocks.field_180398_cJ),
				new ItemStack(Blocks.field_185764_cQ),
				new ItemStack(Blocks.field_150417_aV, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_150336_V),
				new ItemStack(Blocks.field_150435_aG),
				new ItemStack(Blocks.field_150405_ch, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_150406_ce, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_150403_cj),
				new ItemStack(Blocks.field_150385_bj),
				new ItemStack(Blocks.field_150386_bk),
				new ItemStack(Blocks.field_150462_ai),
				new ItemStack(Blocks.field_150460_al),
				new ItemStack(Blocks.field_150442_at),
				new ItemStack(Blocks.field_150367_z),
				new ItemStack(Blocks.field_150409_cd),
				new ItemStack(Blocks.field_150468_ap),
				new ItemStack(Blocks.field_150411_aY),
				new ItemStack(Blocks.field_150371_ca, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_150370_cb),
				new ItemStack(Blocks.field_150372_bz),
				new ItemStack(Blocks.field_180396_cN),
				new ItemStack(Blocks.field_150463_bK, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_150430_aB),
				new ItemStack(Blocks.field_150471_bO),
				new ItemStack(Blocks.field_150333_U, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_180389_cP, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_150376_bx, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_185767_cT),
				new ItemStack(Blocks.field_185768_cU),
				new ItemStack(Blocks.field_185769_cV),
				new ItemStack(Blocks.field_185771_cX),
				new ItemStack(Blocks.field_185772_cY),
				new ItemStack(Blocks.field_150404_cg, 1, OreDictionary.WILDCARD_VALUE),
				new ItemStack(Blocks.field_180400_cw),
				new ItemStack(Blocks.field_150456_au),
				new ItemStack(Blocks.field_150452_aw),
				new ItemStack(Blocks.field_150445_bS),
				new ItemStack(Blocks.field_150443_bT),
				new ItemStack(Items.field_151155_ap),
				new ItemStack(Items.field_151160_bD),
				new ItemStack(Items.field_179572_au),
				new ItemStack(Items.field_179568_as),
				new ItemStack(Items.field_179571_av),
				new ItemStack(Items.field_151139_aw),
				new ItemStack(Items.field_179567_at),
				new ItemStack(Items.field_179570_aq),
				new ItemStack(Items.field_179569_ar)
		)));

		if (ForestryAPI.enabledModules.contains(new ResourceLocation(Constants.MOD_ID, ForestryModuleUids.APICULTURE))) {
			BlockRegistryApiculture beeBlocks = ModuleApiculture.getBlocks();
			backpackAcceptedItemDefaults.put(BackpackManager.BUILDER_UID, getItemStrings(Arrays.asList(
					new ItemStack(beeBlocks.candle, 1, OreDictionary.WILDCARD_VALUE),
					new ItemStack(beeBlocks.stump, 1, OreDictionary.WILDCARD_VALUE)
			)));
		}

		// include everything added via the API
		BackpackInterface backpackInterface = (BackpackInterface) BackpackManager.backpackInterface;
		backpackAcceptedItemDefaults.putAll(backpackInterface.getBackpackAcceptedItems());
	}

	// Should be ore dicted in Forge at some point.
	private static void registerFenceAndFenceGatesToOreDict() {
		OreDictionary.registerOre(OreDictUtil.FENCE_WOOD, Blocks.field_180407_aO);
		OreDictionary.registerOre(OreDictUtil.FENCE_WOOD, Blocks.field_180408_aP);
		OreDictionary.registerOre(OreDictUtil.FENCE_WOOD, Blocks.field_180404_aQ);
		OreDictionary.registerOre(OreDictUtil.FENCE_WOOD, Blocks.field_180403_aR);
		OreDictionary.registerOre(OreDictUtil.FENCE_WOOD, Blocks.field_180406_aS);
		OreDictionary.registerOre(OreDictUtil.FENCE_WOOD, Blocks.field_180405_aT);
		OreDictionary.registerOre(OreDictUtil.FENCE_GATE_WOOD, Blocks.field_180390_bo);
		OreDictionary.registerOre(OreDictUtil.FENCE_GATE_WOOD, Blocks.field_180391_bp);
		OreDictionary.registerOre(OreDictUtil.FENCE_GATE_WOOD, Blocks.field_180392_bq);
		OreDictionary.registerOre(OreDictUtil.FENCE_GATE_WOOD, Blocks.field_180386_br);
		OreDictionary.registerOre(OreDictUtil.FENCE_GATE_WOOD, Blocks.field_180385_bs);
		OreDictionary.registerOre(OreDictUtil.FENCE_GATE_WOOD, Blocks.field_180387_bt);
	}

	private static List<String> getItemStrings(List<ItemStack> itemStacks) {
		List<String> itemStrings = new ArrayList<>(itemStacks.size());
		for (ItemStack itemStack : itemStacks) {
			String itemString = ItemStackUtil.getStringForItemStack(itemStack);
			if (itemString != null) {
				itemStrings.add(itemString);
			}
		}
		return itemStrings;
	}

	private void handleBackpackConfig(LocalizedConfiguration config, String backpackUid) {
		BackpackDefinition backpackDefinition = (BackpackDefinition) BackpackManager.backpackInterface.getBackpackDefinition(backpackUid);
		if (backpackDefinition == null) {
			return;
		}

		Predicate<ItemStack> filter = backpackDefinition.getFilter();
		if (filter instanceof IBackpackFilterConfigurable) {
			IBackpackFilterConfigurable backpackFilter = (IBackpackFilterConfigurable) filter;
			backpackFilter.clear();

			// accepted items
			{
				String[] defaultValidItems = new String[0];
				List<String> defaultAcceptedItemNames = backpackAcceptedItemDefaults.get(backpackUid);
				if (defaultAcceptedItemNames != null) {
					Collections.sort(defaultAcceptedItemNames);
					defaultValidItems = defaultAcceptedItemNames.toArray(new String[defaultAcceptedItemNames.size()]);
				}

				Property backpackConf = config.get("backpacks." + backpackUid, "item.stacks.accepted", defaultValidItems);
				backpackConf.setComment(Translator.translateToLocalFormatted("for.config.backpacks.item.stacks.format", backpackUid));

				String[] backpackItemList = backpackConf.getStringList();
				List<ItemStack> backpackItems = ItemStackUtil.parseItemStackStrings(backpackItemList, OreDictionary.WILDCARD_VALUE);
				for (ItemStack backpackItem : backpackItems) {
					backpackFilter.acceptItem(backpackItem);
				}
			}

			// rejected items
			{
				String[] defaultRejectedItems = new String[0];
				List<String> defaultRejectedItemNames = backpackRejectedItemDefaults.get(backpackUid);
				if (defaultRejectedItemNames != null) {
					Collections.sort(defaultRejectedItemNames);
					defaultRejectedItems = defaultRejectedItemNames.toArray(new String[defaultRejectedItemNames.size()]);
				}

				Property backpackConf = config.get("backpacks." + backpackUid, "item.stacks.rejected", defaultRejectedItems);
				backpackConf.setComment(Translator.translateToLocalFormatted("for.config.backpacks.item.stacks.format", backpackUid));

				String[] backpackItemList = backpackConf.getStringList();
				List<ItemStack> backpackItems = ItemStackUtil.parseItemStackStrings(backpackItemList, OreDictionary.WILDCARD_VALUE);
				for (ItemStack backpackItem : backpackItems) {
					backpackFilter.rejectItem(backpackItem);
				}
			}

			// accepted oreDict
			{
				String[] defaultOreRegexpNames =new String[0];
				List<String> defaultOreRegexpList = backpackAcceptedOreDictRegexpDefaults.get(backpackUid);
				if (defaultOreRegexpList != null) {
					Collections.sort(defaultOreRegexpList);
					defaultOreRegexpNames = defaultOreRegexpList.toArray(new String[defaultOreRegexpList.size()]);
				}

				Property backpackConf = config.get("backpacks." + backpackUid, "ore.dict.accepted", defaultOreRegexpNames);
				backpackConf.setComment(Translator.translateToLocalFormatted("for.config.backpacks.ore.dict.format", backpackUid));

				for (String name : OreDictionary.getOreNames()) {
					if (name == null) {
						Log.error("Found a null oreName in the ore dictionary");
					} else {
						for (String regex : backpackConf.getStringList()) {
							if (name.matches(regex)) {
								backpackFilter.acceptOreDictName(name);
							}
						}
					}
				}
			}

			// rejected oreDict
			{
				String[] defaultOreRegexpNames = new String[0];
				List<String> defaultOreRegexpList = backpackRejectedOreDictRegexpDefaults.get(backpackUid);
				if (defaultOreRegexpList != null) {
					Collections.sort(defaultOreRegexpList);
					defaultOreRegexpNames = defaultOreRegexpList.toArray(new String[defaultOreRegexpList.size()]);
				}

				Property backpackConf = config.get("backpacks." + backpackUid, "ore.dict.rejected", defaultOreRegexpNames);
				backpackConf.setComment(Translator.translateToLocalFormatted("for.config.backpacks.ore.dict.format", backpackUid));

				for (String name : OreDictionary.getOreNames()) {
					if (name == null) {
						Log.error("Found a null oreName in the ore dictionary");
					} else {
						for (String regex : backpackConf.getStringList()) {
							if (name.matches(regex)) {
								backpackFilter.rejectOreDictName(name);
							}
						}
					}
				}
			}
		}
	}

	@Override
	public boolean processIMCMessage(IMCMessage message) {
		if (message.key.equals("add-backpack-items")) {
			String[] tokens = message.getStringValue().split("@");
			if (tokens.length != 2) {
				IMCUtil.logInvalidIMCMessage(message);
				return true;
			}

			String backpackUid = tokens[0];
			String itemStackStrings = tokens[1];

			IBackpackDefinition backpackDefinition = BackpackManager.backpackInterface.getBackpackDefinition(backpackUid);
			if (backpackDefinition == null) {
				String errorMessage = IMCUtil.getInvalidIMCMessageText(message);
				Log.error("{} For non-existent backpack {}.", errorMessage, backpackUid);
				return true;
			}

			List<ItemStack> itemStacks = ItemStackUtil.parseItemStackStrings(itemStackStrings, 0);
			for (ItemStack itemStack : itemStacks) {
				BackpackManager.backpackInterface.addItemToForestryBackpack(backpackUid, itemStack);
			}

			return true;
		}
		return false;
	}

	@Override
	public IPickupHandler getPickupHandler() {
		return new PickupHandlerStorage();
	}

	@Override
	public IResupplyHandler getResupplyHandler() {
		return new ResupplyHandler();
	}

	@Override
	public void registerRecipes() {
		ItemRegistryBackpacks items = getItems();
		if (items.apiaristBackpack != null && ForestryAPI.enabledModules.contains(new ResourceLocation(Constants.MOD_ID, ForestryModuleUids.APICULTURE))) {
			BlockRegistryApiculture beeBlocks = ModuleApiculture.getBlocks();
			addBackpackRecipe("bee", items.apiaristBackpack, "stickWood", beeBlocks.beeChest);
		}

		if (items.lepidopteristBackpack != null && ForestryAPI.enabledModules.contains(new ResourceLocation(Constants.MOD_ID, ForestryModuleUids.LEPIDOPTEROLOGY))) {
			BlockRegistryLepidopterology butterflyBlocks = ModuleLepidopterology.getBlocks();
			ItemStack chest = new ItemStack(butterflyBlocks.butterflyChest);
			addBackpackRecipe("butterfly", items.lepidopteristBackpack, "stickWood", chest);
		}

		addBackpackRecipe("mining", items.minerBackpack, "ingotIron");
		addBackpackRecipe("digging", items.diggerBackpack, "stone");
		addBackpackRecipe("foresting", items.foresterBackpack, "logWood");
		addBackpackRecipe("hunting", items.hunterBackpack, Items.field_151008_G);
		addBackpackRecipe("adventuring", items.adventurerBackpack, Items.field_151103_aS);
		addBackpackRecipe("building", items.builderBackpack, Items.field_151119_aD);

		// / CARPENTER
		if (ForestryAPI.enabledModules.contains(new ResourceLocation(Constants.MOD_ID, ForestryModuleUids.FACTORY))) {
			// / BACKPACKS WOVEN
			addT2BackpackRecipe(items.minerBackpack, items.minerBackpackT2);
			addT2BackpackRecipe(items.diggerBackpack, items.diggerBackpackT2);
			addT2BackpackRecipe(items.foresterBackpack, items.foresterBackpackT2);
			addT2BackpackRecipe(items.hunterBackpack, items.hunterBackpackT2);
			addT2BackpackRecipe(items.adventurerBackpack, items.adventurerBackpackT2);
			addT2BackpackRecipe(items.builderBackpack, items.builderBackpackT2);
		}
	}

	private static void addBackpackRecipe(String recipeName, Item backpack, Object material) {
		addBackpackRecipe(recipeName, backpack, material, "chestWood");
	}

	private static void addBackpackRecipe(String recipeName, Item backpack, Object material, Object chest) {
		RecipeUtil.addRecipe("backpack_" + recipeName, backpack,
				"X#X",
				"VYV",
				"X#X",
				'#', Blocks.field_150325_L,
				'X', Items.field_151007_F,
				'V', material,
				'Y', chest);
	}

	private static void addT2BackpackRecipe(Item backpackT1, Item backpackT2) {
		ItemRegistryCore coreItems = ModuleCore.getItems();

		ItemStack wovenSilk = coreItems.craftingMaterial.getWovenSilk();
		RecipeManagers.carpenterManager.addRecipe(200, new FluidStack(FluidRegistry.WATER, Fluid.BUCKET_VOLUME), ItemStack.field_190927_a, new ItemStack(backpackT2),
				"WXW",
				"WTW",
				"WWW",
				'X', "gemDiamond",
				'W', wovenSilk,
				'T', backpackT1);
	}

	@Override
	public void addLootPoolNames(Set<String> lootPoolNames) {
		lootPoolNames.add("forestry_storage_items");
	}
}
