/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.factory.tiles;

import javax.annotation.Nullable;
import java.io.IOException;
import java.util.ArrayList;

import net.minecraft.client.gui.inventory.GuiContainer;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.Container;
import net.minecraft.inventory.IContainerListener;
import net.minecraft.inventory.ISidedInventory;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.EnumFacing;

import net.minecraftforge.common.capabilities.Capability;
import net.minecraftforge.fluids.FluidRegistry;
import net.minecraftforge.fluids.capability.CapabilityFluidHandler;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import forestry.api.core.IErrorLogic;
import forestry.api.fuels.FuelManager;
import forestry.api.fuels.MoistenerFuel;
import forestry.api.recipes.IMoistenerRecipe;
import forestry.core.config.Constants;
import forestry.core.errors.EnumErrorCode;
import forestry.core.fluids.FilteredTank;
import forestry.core.fluids.FluidHelper;
import forestry.core.fluids.TankManager;
import forestry.core.inventory.IInventoryAdapter;
import forestry.core.network.PacketBufferForestry;
import forestry.core.render.TankRenderInfo;
import forestry.core.tiles.ILiquidTankTile;
import forestry.core.tiles.IRenderableTile;
import forestry.core.tiles.TileBase;
import forestry.core.utils.InventoryUtil;
import forestry.core.utils.ItemStackUtil;
import forestry.factory.gui.ContainerMoistener;
import forestry.factory.gui.GuiMoistener;
import forestry.factory.inventory.InventoryMoistener;
import forestry.factory.recipes.MoistenerRecipeManager;

public class TileMoistener extends TileBase implements ISidedInventory, ILiquidTankTile, IRenderableTile {
	private final FilteredTank resourceTank;
	private final TankManager tankManager;
	@Nullable
	private IMoistenerRecipe currentRecipe;

	private int burnTime = 0;
	private int totalTime = 0;
	private int productionTime = 0;
	private int timePerItem = 0;
	@Nullable
	private ItemStack currentProduct;
	@Nullable
	private ItemStack pendingProduct;

	public TileMoistener() {
		setInternalInventory(new InventoryMoistener(this));
		resourceTank = new FilteredTank(Constants.PROCESSOR_TANK_CAPACITY).setFilters(FluidRegistry.WATER);
		tankManager = new TankManager(this, resourceTank);
	}

	/* LOADING & SAVING */
	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound nbttagcompound) {
		nbttagcompound = super.func_189515_b(nbttagcompound);

		nbttagcompound.func_74768_a("BurnTime", burnTime);
		nbttagcompound.func_74768_a("TotalTime", totalTime);
		nbttagcompound.func_74768_a("ProductionTime", productionTime);

		tankManager.writeToNBT(nbttagcompound);

		// Write pending product
		if (pendingProduct != null) {
			NBTTagCompound nbttagcompoundP = new NBTTagCompound();
			pendingProduct.func_77955_b(nbttagcompoundP);
			nbttagcompound.func_74782_a("PendingProduct", nbttagcompoundP);
		}
		if (currentProduct != null) {
			NBTTagCompound nbttagcompoundP = new NBTTagCompound();
			currentProduct.func_77955_b(nbttagcompoundP);
			nbttagcompound.func_74782_a("CurrentProduct", nbttagcompoundP);
		}
		return nbttagcompound;
	}

	@Override
	public void func_145839_a(NBTTagCompound nbttagcompound) {
		super.func_145839_a(nbttagcompound);

		burnTime = nbttagcompound.func_74762_e("BurnTime");
		totalTime = nbttagcompound.func_74762_e("TotalTime");
		productionTime = nbttagcompound.func_74762_e("ProductionTime");

		tankManager.readFromNBT(nbttagcompound);

		// Load pending product
		if (nbttagcompound.func_74764_b("PendingProduct")) {
			NBTTagCompound nbttagcompoundP = nbttagcompound.func_74775_l("PendingProduct");
			pendingProduct = new ItemStack(nbttagcompoundP);
		}
		if (nbttagcompound.func_74764_b("CurrentProduct")) {
			NBTTagCompound nbttagcompoundP = nbttagcompound.func_74775_l("CurrentProduct");
			currentProduct = new ItemStack(nbttagcompoundP);
		}

		checkRecipe();
	}

	@Override
	public void writeData(PacketBufferForestry data) {
		super.writeData(data);
		tankManager.writeData(data);
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void readData(PacketBufferForestry data) throws IOException {
		super.readData(data);
		tankManager.readData(data);
	}

	@Override
	public void updateServerSide() {

		if (updateOnInterval(20)) {
			// Check if we have suitable water container waiting in the item slot
			FluidHelper.drainContainers(tankManager, this, InventoryMoistener.SLOT_PRODUCT);
		}

		// Let's get to work
		int lightvalue = field_145850_b.func_175671_l(func_174877_v().func_177984_a());

		IErrorLogic errorLogic = getErrorLogic();

		// Not working in broad daylight
		boolean gloomy = lightvalue <= 11;
		if (errorLogic.setCondition(!gloomy, EnumErrorCode.NOT_DARK)) {
			return;
		}

		// The darker, the better
		int speed;
		if (lightvalue >= 9) {
			speed = 1;
		} else if (lightvalue >= 7) {
			speed = 2;
		} else if (lightvalue >= 5) {
			speed = 3;
		} else {
			speed = 4;
		}

		// Already running
		if (burnTime > 0 && pendingProduct == null) {
			// Not working if there is no water available.
			boolean hasLiquid = resourceTank.getFluidAmount() > 0;
			if (errorLogic.setCondition(!hasLiquid, EnumErrorCode.NO_RESOURCE_LIQUID)) {
				return;
			}

			checkRecipe();

			if (currentRecipe == null) {
				return;
			}

			resourceTank.drain(1, true);
			burnTime -= speed;
			productionTime -= speed;

			if (productionTime <= 0) {
				pendingProduct = currentProduct;
				func_70298_a(InventoryMoistener.SLOT_RESOURCE, 1);
				resetRecipe();
				tryAddPending();
			}

		} else if (pendingProduct != null) {
			tryAddPending();
		}
		// Try to start process
		else { // Make sure we have a new item in the working slot.
			if (rotateWorkingSlot()) {
				checkRecipe();

				// Let's see if we have a valid resource in the working slot
				if (func_70301_a(InventoryMoistener.SLOT_WORKING).func_190926_b()) {
					return;
				}

				if (FuelManager.moistenerResource.containsKey(func_70301_a(InventoryMoistener.SLOT_WORKING))) {
					MoistenerFuel res = FuelManager.moistenerResource.get(func_70301_a(InventoryMoistener.SLOT_WORKING));
					burnTime = totalTime = res.getMoistenerValue();
				}
			} else {
				rotateReservoir();
			}
		}

		errorLogic.setCondition(currentRecipe == null, EnumErrorCode.NO_RECIPE);
	}

	private boolean tryAddPending() {
		if (pendingProduct == null) {
			return false;
		}

		boolean added = InventoryUtil.tryAddStack(this, pendingProduct, InventoryMoistener.SLOT_PRODUCT, 1, true);
		getErrorLogic().setCondition(!added, EnumErrorCode.NO_SPACE_INVENTORY);

		if (added) {
			pendingProduct = null;
		}

		return added;
	}

	public void checkRecipe() {
		IMoistenerRecipe sameRec = MoistenerRecipeManager.findMatchingRecipe(getInternalInventory().func_70301_a(InventoryMoistener.SLOT_RESOURCE));
		if (currentRecipe != sameRec) {
			currentRecipe = sameRec;
			resetRecipe();
		}

		getErrorLogic().setCondition(currentRecipe == null, EnumErrorCode.NO_RECIPE);
	}

	private void resetRecipe() {
		if (currentRecipe == null) {
			currentProduct = null;
			productionTime = 0;
			timePerItem = 0;
		} else {
			currentProduct = currentRecipe.getProduct();
			productionTime = currentRecipe.getTimePerItem();
			timePerItem = currentRecipe.getTimePerItem();
		}
	}

	private int getFreeSlot(ItemStack deposit, int startSlot, int endSlot, boolean emptyOnly) {
		int slot = -1;

		for (int i = startSlot; i < endSlot; i++) {
			ItemStack slotStack = func_70301_a(i);
			// Empty slots are okay.
			if (slotStack.func_190926_b()) {
				if (slot < 0) {
					slot = i;
				}
				continue;
			}

			if (emptyOnly) {
				continue;
			}

			// Wrong item or full
			if (!slotStack.func_77969_a(deposit) || slotStack.func_190916_E() >= slotStack.func_77976_d()) {
				continue;
			}

			slot = i;
		}

		return slot;
	}

	private int getFreeStashSlot(ItemStack deposit, boolean emptyOnly) {
		return getFreeSlot(deposit, 0, InventoryMoistener.SLOT_RESERVOIR_1, emptyOnly);
	}

	private int getFreeReservoirSlot(ItemStack deposit) {
		return getFreeSlot(deposit, InventoryMoistener.SLOT_RESERVOIR_1, InventoryMoistener.SLOT_RESERVOIR_1 + InventoryMoistener.SLOT_RESERVOIR_COUNT, false);

	}

	private int getNextResourceSlot(int startSlot, int endSlot) {
		// Let's look for a new resource to put into the working slot.
		int stage = -1;
		int resourceSlot = -1;

		IInventoryAdapter inventory = getInternalInventory();
		for (int i = startSlot; i < endSlot; i++) {
			ItemStack slotStack = inventory.func_70301_a(i);
			if (slotStack.func_190926_b()) {
				continue;
			}

			if (!FuelManager.moistenerResource.containsKey(slotStack)) {
				continue;
			}

			MoistenerFuel res = FuelManager.moistenerResource.get(slotStack);
			if (stage < 0 || res.getStage() < stage) {
				stage = res.getStage();
				resourceSlot = i;
			}
		}

		return resourceSlot;
	}

	private boolean rotateWorkingSlot() {
		IErrorLogic errorLogic = getErrorLogic();

		// Put working slot contents into inventory if space is available
		if (!func_70301_a(InventoryMoistener.SLOT_WORKING).func_190926_b()) {
			// Get the result of the consumed item in the working slot
			ItemStack deposit;
			if (FuelManager.moistenerResource.containsKey(func_70301_a(InventoryMoistener.SLOT_WORKING))) {
				MoistenerFuel res = FuelManager.moistenerResource.get(func_70301_a(InventoryMoistener.SLOT_WORKING));
				deposit = res.getProduct().func_77946_l();
			} else {
				deposit = func_70301_a(InventoryMoistener.SLOT_WORKING).func_77946_l();
			}

			int targetSlot = getFreeReservoirSlot(deposit);
			// We stop the whole thing, if we don't have any room anymore.
			if (errorLogic.setCondition(targetSlot < 0, EnumErrorCode.NO_SPACE_INVENTORY)) {
				return false;
			}

			if (func_70301_a(targetSlot).func_190926_b()) {
				func_70299_a(targetSlot, deposit);
			} else {
				func_70301_a(targetSlot).func_190917_f(1);
			}

			func_70298_a(InventoryMoistener.SLOT_WORKING, 1);
		}

		if (!func_70301_a(InventoryMoistener.SLOT_WORKING).func_190926_b()) {
			return true;
		}

		// Let's look for a new resource to put into the working slot.
		int resourceSlot = getNextResourceSlot(InventoryMoistener.SLOT_RESERVOIR_1, InventoryMoistener.SLOT_RESERVOIR_1 + InventoryMoistener.SLOT_RESERVOIR_COUNT);
		// Nothing found, stop.
		if (errorLogic.setCondition(resourceSlot < 0, EnumErrorCode.NO_RESOURCE)) {
			return false;
		}

		func_70299_a(InventoryMoistener.SLOT_WORKING, func_70298_a(resourceSlot, 1));
		return true;
	}

	private void rotateReservoir() {
		ArrayList<Integer> slotsToShift = new ArrayList<>();

		for (int i = InventoryMoistener.SLOT_RESERVOIR_1; i < InventoryMoistener.SLOT_RESERVOIR_1 + InventoryMoistener.SLOT_RESERVOIR_COUNT; i++) {
			if (func_70301_a(i).func_190926_b()) {
				continue;
			}

			if (!FuelManager.moistenerResource.containsKey(func_70301_a(i))) {
				slotsToShift.add(i);
			}
		}

		// Move consumed items back to stash
		int shiftedSlots = 0;
		for (int slot : slotsToShift) {
			ItemStack slotStack = func_70301_a(slot);
			int targetSlot = getFreeStashSlot(slotStack, true);
			if (targetSlot < 0) {
				continue;
			}

			func_70299_a(targetSlot, slotStack);
			func_70299_a(slot, ItemStack.field_190927_a);
			shiftedSlots++;
		}

		// Grab new items from stash
		for (int i = 0; i < (!slotsToShift.isEmpty() ? shiftedSlots : 2); i++) {
			int resourceSlot = getNextResourceSlot(0, InventoryMoistener.SLOT_RESERVOIR_1);
			// Stop if no resources are available
			if (resourceSlot < 0) {
				break;
			}
			int targetSlot = getFreeReservoirSlot(func_70301_a(resourceSlot));
			// No free target slot, stop
			if (targetSlot < 0) {
				break;
			}
			// Else shift
			if (func_70301_a(targetSlot).func_190926_b()) {
				func_70299_a(targetSlot, func_70301_a(resourceSlot));
				func_70299_a(resourceSlot, ItemStack.field_190927_a);
			} else {
				ItemStackUtil.mergeStacks(func_70301_a(resourceSlot), func_70301_a(targetSlot));
			}
		}
	}

	public boolean isWorking() {
		return burnTime > 0 && resourceTank.getFluidAmount() > 0;
	}

	public boolean hasFuelMin(float percentage) {
		int max = 0;
		int avail = 0;
		IInventoryAdapter inventory = getInternalInventory();

		for (int i = InventoryMoistener.SLOT_STASH_1; i < InventoryMoistener.SLOT_RESERVOIR_1; i++) {
			if (inventory.func_70301_a(i).func_190926_b()) {
				max += 64;
				continue;
			}
			if (FuelManager.moistenerResource.containsKey(inventory.func_70301_a(i))) {
				MoistenerFuel res = FuelManager.moistenerResource.get(inventory.func_70301_a(i));
				if (res.getItem().func_77969_a(inventory.func_70301_a(i))) {
					max += 64;
					avail += inventory.func_70301_a(i).func_190916_E();
				}
			}
		}

		return (float) avail / (float) max > percentage;
	}

	public boolean hasResourcesMin(float percentage) {
		IInventoryAdapter inventory = getInternalInventory();
		if (inventory.func_70301_a(InventoryMoistener.SLOT_RESOURCE).func_190926_b()) {
			return false;
		}

		return (float) inventory.func_70301_a(InventoryMoistener.SLOT_RESOURCE).func_190916_E() / (float) inventory.func_70301_a(InventoryMoistener.SLOT_RESOURCE).func_77976_d() > percentage;
	}

	public boolean isProducing() {
		return productionTime > 0;
	}

	public int getProductionProgressScaled(int i) {
		if (timePerItem == 0) {
			return 0;
		}

		return productionTime * i / timePerItem;

	}

	public int getConsumptionProgressScaled(int i) {
		if (totalTime == 0) {
			return 0;
		}

		return burnTime * i / totalTime;

	}

	public int getResourceScaled(int i) {
		return resourceTank.getFluidAmount() * i / Constants.PROCESSOR_TANK_CAPACITY;
	}

	/* IRenderableTile */
	@Override
	public TankRenderInfo getResourceTankInfo() {
		return new TankRenderInfo(resourceTank);
	}

	@Override
	public TankRenderInfo getProductTankInfo() {
		return TankRenderInfo.EMPTY;
	}


	@Override
	public TankManager getTankManager() {
		return tankManager;
	}

	/* SMP GUI */
	public void getGUINetworkData(int i, int j) {
		switch (i) {
			case 0:
				burnTime = j;
				break;
			case 1:
				totalTime = j;
				break;
			case 2:
				productionTime = j;
				break;
			case 3:
				timePerItem = j;
				break;
		}
	}

	public void sendGUINetworkData(Container container, IContainerListener iCrafting) {
		iCrafting.func_71112_a(container, 0, burnTime);
		iCrafting.func_71112_a(container, 1, totalTime);
		iCrafting.func_71112_a(container, 2, productionTime);
		iCrafting.func_71112_a(container, 3, timePerItem);
	}

	@Override
	@SideOnly(Side.CLIENT)
	public GuiContainer getGui(EntityPlayer player, int data) {
		return new GuiMoistener(player.field_71071_by, this);
	}

	@Override
	public Container getContainer(EntityPlayer player, int data) {
		return new ContainerMoistener(player.field_71071_by, this);
	}

	@Override
	public boolean hasCapability(Capability<?> capability, @Nullable EnumFacing facing) {
		return capability == CapabilityFluidHandler.FLUID_HANDLER_CAPABILITY || super.hasCapability(capability, facing);
	}

	@Override
	@Nullable
	public <T> T getCapability(Capability<T> capability, @Nullable EnumFacing facing) {
		if (capability == CapabilityFluidHandler.FLUID_HANDLER_CAPABILITY) {
			return CapabilityFluidHandler.FLUID_HANDLER_CAPABILITY.cast(tankManager);
		}
		return super.getCapability(capability, facing);
	}
}
