package forestry.arboriculture.blocks;

import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.List;

import net.minecraft.block.state.BlockStateContainer;
import net.minecraft.block.state.IBlockState;
import net.minecraft.client.renderer.block.model.ModelResourceLocation;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumHand;
import net.minecraft.util.NonNullList;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;

import com.mojang.authlib.GameProfile;

import net.minecraftforge.client.model.ModelLoader;

import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import forestry.api.arboriculture.EnumGermlingType;
import forestry.api.arboriculture.IFruitProvider;
import forestry.api.arboriculture.ITree;
import forestry.api.arboriculture.ITreeGenome;
import forestry.api.arboriculture.TreeManager;
import forestry.api.core.IModelManager;
import forestry.arboriculture.genetics.TreeDefinition;

/**
 * Genetic leaves with no tile entity, used for worldgen trees.
 * Similar to decorative leaves, but these will drop saplings and can be used for pollination.
 */
public abstract class BlockDefaultLeaves extends BlockAbstractLeaves {
	private static final int VARIANTS_PER_BLOCK = 4;

	public static List<BlockDefaultLeaves> create() {
		List<BlockDefaultLeaves> blocks = new ArrayList<>();
		final int blockCount = PropertyTreeType.getBlockCount(VARIANTS_PER_BLOCK);
		for (int blockNumber = 0; blockNumber < blockCount; blockNumber++) {
			final PropertyTreeType variant = PropertyTreeType.create("variant", blockNumber, VARIANTS_PER_BLOCK);
			BlockDefaultLeaves block = new BlockDefaultLeaves(blockNumber) {
				@Override
				public PropertyTreeType getVariant() {
					return variant;
				}
			};
			blocks.add(block);
		}
		return blocks;
	}

	private final int blockNumber;

	public BlockDefaultLeaves(int blockNumber) {
		this.blockNumber = blockNumber;
		PropertyTreeType variant = getVariant();
		func_180632_j(this.field_176227_L.func_177621_b()
				.func_177226_a(variant, variant.getFirstType())
				.func_177226_a(field_176236_b, false)
				.func_177226_a(field_176237_a, true));
	}

	public int getBlockNumber() {
		return blockNumber;
	}

	protected abstract PropertyTreeType getVariant();

	@Nullable
	public TreeDefinition getTreeDefinition(IBlockState blockState) {
		if (blockState.func_177230_c() == this) {
			return blockState.func_177229_b(getVariant());
		} else {
			return null;
		}
	}

	@Override
	public int func_180651_a(IBlockState state) {
		TreeDefinition treeDefinition = getTreeDefinition(state);
		if (treeDefinition == null) {
			return 0;
		}
		return treeDefinition.getMetadata() - blockNumber * VARIANTS_PER_BLOCK;
	}

	@Override
	public IBlockState func_176203_a(int meta) {
		return this.func_176223_P()
				.func_177226_a(getVariant(), getTreeType(meta))
				.func_177226_a(field_176237_a, (meta & 4) == 0)
				.func_177226_a(field_176236_b, (meta & 8) > 0);
	}

	@Override
	public int func_176201_c(IBlockState state) {
		int i = func_180651_a(state);

		if (!state.func_177229_b(field_176237_a)) {
			i |= 4;
		}

		if (state.func_177229_b(field_176236_b)) {
			i |= 8;
		}

		return i;
	}

	protected BlockStateContainer func_180661_e() {
		return new BlockStateContainer(this, getVariant(), field_176236_b, field_176237_a);
	}

	public TreeDefinition getTreeType(int meta) {
		int variantCount = getVariant().func_177700_c().size();
		int variantMeta = (meta % variantCount) + blockNumber * VARIANTS_PER_BLOCK;
		return TreeDefinition.byMetadata(variantMeta);
	}

	@Override
	protected void getLeafDrop(NonNullList<ItemStack> drops, World world, @Nullable GameProfile playerProfile, BlockPos pos, float saplingModifier, int fortune) {
		ITree tree = getTree(world, pos);
		if (tree == null) {
			return;
		}

		// Add saplings
		List<ITree> saplings = tree.getSaplings(world, playerProfile, pos, saplingModifier);
		for (ITree sapling : saplings) {
			if (sapling != null) {
				drops.add(TreeManager.treeRoot.getMemberStack(sapling, EnumGermlingType.SAPLING));
			}
		}

		// Add fruits
		ITreeGenome genome = tree.getGenome();
		IFruitProvider fruitProvider = genome.getFruitProvider();
		if (fruitProvider.isFruitLeaf(genome, world, pos)) {
			NonNullList<ItemStack> produceStacks = tree.produceStacks(world, pos, Integer.MAX_VALUE);
			drops.addAll(produceStacks);
		}
	}

	@Override
	public IBlockState getStateForPlacement(World world, BlockPos pos, EnumFacing facing, float hitX, float hitY, float hitZ, int meta, EntityLivingBase placer, EnumHand hand) {
		TreeDefinition type = getTreeType(meta);
		return func_176223_P().func_177226_a(getVariant(), type);
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void registerModel(Item item, IModelManager manager) {
		for (IBlockState state : field_176227_L.func_177619_a()) {
			int meta = func_176201_c(state);
			ModelLoader.setCustomModelResourceLocation(item, meta, new ModelResourceLocation("forestry:leaves.default." + blockNumber, "inventory"));
		}
	}

	@Override
	protected ITree getTree(IBlockAccess world, BlockPos pos) {
		IBlockState blockState = world.func_180495_p(pos);
		TreeDefinition treeDefinition = getTreeDefinition(blockState);
		if (treeDefinition != null) {
			return treeDefinition.getIndividual();
		} else {
			return null;
		}
	}

	@Override
	@SideOnly(Side.CLIENT)
	public int colorMultiplier(IBlockState state, @Nullable IBlockAccess worldIn, @Nullable BlockPos pos, int tintIndex) {
		TreeDefinition treeDefinition = getTreeDefinition(state);
		if (treeDefinition == null) {
			treeDefinition = TreeDefinition.Oak;
		}
		ITreeGenome genome = treeDefinition.getGenome();

		if (tintIndex == 0) {
			return genome.getPrimary().getLeafSpriteProvider().getColor(false);
		} else {
			IFruitProvider fruitProvider = genome.getFruitProvider();
			return fruitProvider.getDecorativeColor();
		}
	}
}
