package forestry.core.render;

import java.util.List;

import forestry.api.apiculture.IBeeGenome;
import forestry.api.apiculture.IBeeHousing;
import forestry.api.apiculture.IHiveTile;
import forestry.apiculture.entities.ParticleBeeExplore;
import forestry.apiculture.entities.ParticleBeeRoundTrip;
import forestry.apiculture.entities.ParticleBeeTargetEntity;
import forestry.apiculture.genetics.alleles.AlleleEffect;
import forestry.core.config.Config;
import forestry.core.entities.ParticleHoneydust;
import forestry.core.entities.ParticleIgnition;
import forestry.core.entities.ParticleSmoke;
import forestry.core.entities.ParticleSnow;
import forestry.core.utils.VectUtil;
import net.minecraft.client.Minecraft;
import net.minecraft.client.particle.Particle;
import net.minecraft.client.particle.ParticleManager;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.util.EnumParticleTypes;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.Vec3d;
import net.minecraft.util.math.Vec3i;
import net.minecraft.world.World;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

@SideOnly(Side.CLIENT)
public class ParticleRender {
	public static boolean shouldSpawnParticle(World world) {
		if (!Config.enableParticleFX) {
			return false;
		}

		Minecraft mc = Minecraft.func_71410_x();
		int particleSetting = mc.field_71474_y.field_74362_aa;

		if (particleSetting == 2) { // minimal
			return world.field_73012_v.nextInt(10) == 0;
		} else if (particleSetting == 1) { // decreased
			return world.field_73012_v.nextInt(3) != 0;
		} else { // all
			return true;
		}
	}

	public static void addBeeHiveFX(IBeeHousing housing, IBeeGenome genome, List<BlockPos> flowerPositions) {
		World world = housing.getWorldObj();
		if (!shouldSpawnParticle(world)) {
			return;
		}

		ParticleManager effectRenderer = Minecraft.func_71410_x().field_71452_i;

		Vec3d particleStart = housing.getBeeFXCoordinates();

		// Avoid rendering bee particles that are too far away, they're very small.
		// At 32+ distance, have no bee particles. Make more particles up close.
		BlockPos playerPosition = Minecraft.func_71410_x().field_71439_g.func_180425_c();
		double playerDistanceSq = playerPosition.func_177957_d(particleStart.field_72450_a, particleStart.field_72448_b, particleStart.field_72449_c);
		if (world.field_73012_v.nextInt(1024) < playerDistanceSq) {
			return;
		}

		int color = genome.getPrimary().getSpriteColour(0);

		if (!flowerPositions.isEmpty()) {
			int randomInt = world.field_73012_v.nextInt(100);

			if (housing instanceof IHiveTile) {
				if (((IHiveTile) housing).isAngry() || randomInt >= 85) {
					List<EntityLivingBase> entitiesInRange = AlleleEffect.getEntitiesInRange(genome, housing, EntityLivingBase.class);
					if (!entitiesInRange.isEmpty()) {
						EntityLivingBase entity = entitiesInRange.get(world.field_73012_v.nextInt(entitiesInRange.size()));
						Particle particle = new ParticleBeeTargetEntity(world, particleStart, entity, color);
						effectRenderer.func_78873_a(particle);
						return;
					}
				}
			}

			if (randomInt < 75) {
				BlockPos destination = flowerPositions.get(world.field_73012_v.nextInt(flowerPositions.size()));
				Particle particle = new ParticleBeeRoundTrip(world, particleStart, destination, color);
				effectRenderer.func_78873_a(particle);
			} else {
				Vec3i area = AlleleEffect.getModifiedArea(genome, housing);
				Vec3i offset = housing.getCoordinates().func_177982_a(-area.func_177958_n() / 2, -area.func_177956_o() / 4, -area.func_177952_p() / 2);
				BlockPos destination = VectUtil.getRandomPositionInArea(world.field_73012_v, area).func_177971_a(offset);
				Particle particle = new ParticleBeeExplore(world, particleStart, destination, color);
				effectRenderer.func_78873_a(particle);
			}
		}
	}

	public static void addEntityHoneyDustFX(World world, double x, double y, double z) {
		if (!shouldSpawnParticle(world)) {
			return;
		}

		ParticleManager effectRenderer = Minecraft.func_71410_x().field_71452_i;
		effectRenderer.func_78873_a(new ParticleHoneydust(world, x, y, z, 0, 0, 0));
	}

	public static void addEntityExplodeFX(World world, double x, double y, double z) {
		if (!shouldSpawnParticle(world)) {
			return;
		}

		ParticleManager effectRenderer = Minecraft.func_71410_x().field_71452_i;
		Particle Particle = effectRenderer.func_178927_a(EnumParticleTypes.EXPLOSION_NORMAL.func_179348_c(), x, y, z, 0, 0, 0);
		effectRenderer.func_78873_a(Particle);
	}

	public static void addEntitySnowFX(World world, double x, double y, double z) {
		if (!shouldSpawnParticle(world)) {
			return;
		}

		ParticleManager effectRenderer = Minecraft.func_71410_x().field_71452_i;
		effectRenderer.func_78873_a(new ParticleSnow(world, x + world.field_73012_v.nextGaussian(), y, z + world.field_73012_v.nextGaussian()));
	}

	public static void addEntityIgnitionFX(World world, double x, double y, double z) {
		if (!shouldSpawnParticle(world)) {
			return;
		}

		ParticleManager effectRenderer = Minecraft.func_71410_x().field_71452_i;
		effectRenderer.func_78873_a(new ParticleIgnition(world, x, y, z));
	}

	public static void addEntitySmokeFX(World world, double x, double y, double z) {
		if (!shouldSpawnParticle(world)) {
			return;
		}

		ParticleManager effectRenderer = Minecraft.func_71410_x().field_71452_i;
		effectRenderer.func_78873_a(new ParticleSmoke(world, x, y, z));
	}

	public static void addEntityPotionFX(World world, double x, double y, double z, int color) {
		if (!shouldSpawnParticle(world)) {
			return;
		}

		float red = (color >> 16 & 255) / 255.0F;
		float green = (color >> 8 & 255) / 255.0F;
		float blue = (color & 255) / 255.0F;

		ParticleManager effectRenderer = Minecraft.func_71410_x().field_71452_i;
		Particle particle = effectRenderer.func_178927_a(EnumParticleTypes.SPELL.func_179348_c(), x, y, z, 0, 0, 0);
		if (particle != null) {
			particle.func_70538_b(red, green, blue);
			effectRenderer.func_78873_a(particle);
		}
	}
}
