/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.core.genetics;

import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import com.mojang.authlib.GameProfile;
import forestry.api.genetics.AlleleManager;
import forestry.api.genetics.IAllele;
import forestry.api.genetics.IAlleleSpecies;
import forestry.api.genetics.IBreedingTracker;
import forestry.api.genetics.IMutation;
import forestry.api.genetics.ISpeciesRoot;
import forestry.core.genetics.mutations.EnumMutateChance;
import forestry.core.items.ItemForestry;
import forestry.core.utils.NetworkUtil;
import forestry.core.utils.PlayerUtil;
import forestry.core.utils.Translator;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.ActionResult;
import net.minecraft.util.EnumActionResult;
import net.minecraft.util.EnumHand;
import net.minecraft.util.text.TextComponentTranslation;
import net.minecraft.util.text.TextFormatting;
import net.minecraft.world.World;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

public class ItemResearchNote extends ItemForestry {

	public enum EnumNoteType {
		NONE, MUTATION, SPECIES;

		public static final EnumNoteType[] VALUES = values();

		@Nullable
		private static IMutation getEncodedMutation(ISpeciesRoot root, NBTTagCompound compound) {
			IAllele allele0 = AlleleManager.alleleRegistry.getAllele(compound.func_74779_i("AL0"));
			IAllele allele1 = AlleleManager.alleleRegistry.getAllele(compound.func_74779_i("AL1"));
			if (allele0 == null || allele1 == null) {
				return null;
			}

			IAllele result = null;
			if (compound.func_74764_b("RST")) {
				result = AlleleManager.alleleRegistry.getAllele(compound.func_74779_i("RST"));
			}

			IMutation encoded = null;
			for (IMutation mutation : root.getCombinations(allele0)) {
				if (mutation.isPartner(allele1)) {
					if (result == null
							|| mutation.getTemplate()[0].getUID().equals(result.getUID())) {
						encoded = mutation;
						break;
					}
				}
			}

			return encoded;
		}

		public List<String> getTooltip(NBTTagCompound compound) {
			List<String> tooltips = new ArrayList<>();

			if (this == NONE) {
				return tooltips;
			}

			if (this == MUTATION) {
				ISpeciesRoot root = AlleleManager.alleleRegistry.getSpeciesRoot(compound.func_74779_i("ROT"));
				if (root == null) {
					return tooltips;
				}

				IMutation encoded = getEncodedMutation(root, compound);
				if (encoded == null) {
					return tooltips;
				}

				String species1 = encoded.getAllele0().getName();
				String species2 = encoded.getAllele1().getName();
				String mutationChanceKey = EnumMutateChance.rateChance(encoded.getBaseChance()).toString().toLowerCase(Locale.ENGLISH);
				String mutationChance = Translator.translateToLocal("for.researchNote.chance." + mutationChanceKey);
				String speciesResult = encoded.getTemplate()[root.getSpeciesChromosomeType().ordinal()].getName();

				tooltips.add(Translator.translateToLocal("for.researchNote.discovery.0"));
				tooltips.add(Translator.translateToLocal("for.researchNote.discovery.1").replace("%SPEC1", species1).replace("%SPEC2", species2));
				tooltips.add(Translator.translateToLocalFormatted("for.researchNote.discovery.2", mutationChance));
				tooltips.add(Translator.translateToLocalFormatted("for.researchNote.discovery.3", speciesResult));

				if (!encoded.getSpecialConditions().isEmpty()) {
					for (String line : encoded.getSpecialConditions()) {
						tooltips.add(TextFormatting.GOLD + line);
					}
				}
			} else if (this == SPECIES) {
				IAlleleSpecies allele0 = (IAlleleSpecies) AlleleManager.alleleRegistry.getAllele(compound.func_74779_i("AL0"));
				if (allele0 == null) {
					return tooltips;
				}
				ISpeciesRoot root = AlleleManager.alleleRegistry.getSpeciesRoot(compound.func_74779_i("ROT"));
				if (root == null) {
					return tooltips;
				}

				tooltips.add("researchNote.discovered.0");
				tooltips.add(Translator.translateToLocalFormatted("for.researchNote.discovered.1", allele0.getName(), allele0.getBinomial()));
			}

			return tooltips;
		}

		public boolean registerResults(World world, EntityPlayer player, NBTTagCompound compound) {
			if (this == NONE) {
				return false;
			}

			if (this == MUTATION) {
				ISpeciesRoot root = AlleleManager.alleleRegistry.getSpeciesRoot(compound.func_74779_i("ROT"));
				if (root == null) {
					return false;
				}

				IMutation encoded = getEncodedMutation(root, compound);
				if (encoded == null) {
					return false;
				}

				IBreedingTracker tracker = encoded.getRoot().getBreedingTracker(world, player.func_146103_bH());
				if (tracker.isResearched(encoded)) {
					player.func_145747_a(new TextComponentTranslation("for.chat.cannotmemorizeagain"));
					return false;
				}

				IAlleleSpecies species0 = encoded.getAllele0();
				IAlleleSpecies species1 = encoded.getAllele1();
				IAlleleSpecies speciesResult = (IAlleleSpecies) encoded.getTemplate()[root.getSpeciesChromosomeType().ordinal()];

				tracker.registerSpecies(species0);
				tracker.registerSpecies(species1);
				tracker.registerSpecies(speciesResult);

				tracker.researchMutation(encoded);
				player.func_145747_a(new TextComponentTranslation("for.chat.memorizednote"));

				player.func_145747_a(new TextComponentTranslation("for.chat.memorizednote2",
						TextFormatting.GRAY + species0.getName(),
						TextFormatting.GRAY + species1.getName(),
						TextFormatting.GREEN + speciesResult.getName()));

				return true;
			}

			return false;

		}

		public static ResearchNote createMutationNote(GameProfile researcher, IMutation mutation) {
			NBTTagCompound compound = new NBTTagCompound();
			compound.func_74778_a("ROT", mutation.getRoot().getUID());
			compound.func_74778_a("AL0", mutation.getAllele0().getUID());
			compound.func_74778_a("AL1", mutation.getAllele1().getUID());
			compound.func_74778_a("RST", mutation.getTemplate()[0].getUID());
			return new ResearchNote(researcher, MUTATION, compound);
		}

		public static ItemStack createMutationNoteStack(Item item, GameProfile researcher, IMutation mutation) {
			ResearchNote note = createMutationNote(researcher, mutation);
			NBTTagCompound compound = new NBTTagCompound();
			note.writeToNBT(compound);
			ItemStack created = new ItemStack(item);
			created.func_77982_d(compound);
			return created;
		}

		public static ResearchNote createSpeciesNote(GameProfile researcher, IAlleleSpecies species) {
			NBTTagCompound compound = new NBTTagCompound();
			compound.func_74778_a("ROT", species.getRoot().getUID());
			compound.func_74778_a("AL0", species.getUID());
			return new ResearchNote(researcher, SPECIES, compound);
		}

		public static ItemStack createSpeciesNoteStack(Item item, GameProfile researcher, IAlleleSpecies species) {
			ResearchNote note = createSpeciesNote(researcher, species);
			NBTTagCompound compound = new NBTTagCompound();
			note.writeToNBT(compound);
			ItemStack created = new ItemStack(item);
			created.func_77982_d(compound);
			return created;
		}

	}

	public static class ResearchNote {
		@Nullable
		private final GameProfile researcher;
		private final EnumNoteType type;
		private final NBTTagCompound inner;

		public ResearchNote(GameProfile researcher, EnumNoteType type, NBTTagCompound inner) {
			this.researcher = researcher;
			this.type = type;
			this.inner = inner;
		}

		public ResearchNote(@Nullable NBTTagCompound compound) {
			if (compound != null) {
				if (compound.func_74764_b("res")) {
					this.researcher = PlayerUtil.readGameProfileFromNBT(compound.func_74775_l("res"));
				} else {
					this.researcher = null;
				}
				this.type = EnumNoteType.VALUES[compound.func_74771_c("TYP")];
				this.inner = compound.func_74775_l("INN");
			} else {
				this.type = EnumNoteType.NONE;
				this.researcher = null;
				this.inner = new NBTTagCompound();
			}
		}

		public NBTTagCompound writeToNBT(NBTTagCompound compound) {
			if (this.researcher != null) {
				NBTTagCompound nbt = new NBTTagCompound();
				PlayerUtil.writeGameProfile(nbt, researcher);
				compound.func_74782_a("res", nbt);
			}
			compound.func_74774_a("TYP", (byte) type.ordinal());
			compound.func_74782_a("INN", inner);
			return compound;
		}

		public void addTooltip(List<String> list) {
			List<String> tooltips = type.getTooltip(inner);
			if (tooltips.size() <= 0) {
				list.add(TextFormatting.ITALIC + TextFormatting.RED.toString() + Translator.translateToLocal("for.researchNote.error.0"));
				list.add(Translator.translateToLocal("for.researchNote.error.1"));
				return;
			}

			list.addAll(tooltips);
		}

		public boolean registerResults(World world, EntityPlayer player) {
			return type.registerResults(world, player, inner);
		}
	}

	public ItemResearchNote() {
		func_77637_a(null);
	}

	@Override
	public String func_77653_i(ItemStack itemstack) {
		ResearchNote note = new ResearchNote(itemstack.func_77978_p());
		String researcherName;
		if (note.researcher == null) {
			researcherName = "Sengir";
		} else {
			researcherName = note.researcher.getName();
		}
		return Translator.translateToLocalFormatted(func_77667_c(itemstack) + ".name", researcherName);
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void func_77624_a(ItemStack itemstack, EntityPlayer player, List<String> list, boolean flag) {
		super.func_77624_a(itemstack, player, list, flag);
		ResearchNote note = new ResearchNote(itemstack.func_77978_p());
		note.addTooltip(list);
	}

	@Override
	public ActionResult<ItemStack> func_77659_a(World worldIn, EntityPlayer playerIn, EnumHand handIn) {
		ItemStack heldItem = playerIn.func_184586_b(handIn);
		if (worldIn.field_72995_K) {
			return ActionResult.newResult(EnumActionResult.PASS, heldItem);
		}

		ResearchNote note = new ResearchNote(heldItem.func_77978_p());
		if (note.registerResults(worldIn, playerIn)) {
			playerIn.field_71071_by.func_70298_a(playerIn.field_71071_by.field_70461_c, 1);
			// Notify player that his inventory has changed.
			NetworkUtil.inventoryChangeNotify(playerIn);
		}

		return ActionResult.newResult(EnumActionResult.SUCCESS, heldItem);
	}
}
