/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.greenhouse.tiles;

import java.io.IOException;
import java.util.Random;
import java.util.Set;

import javax.annotation.Nullable;

import forestry.api.core.EnumHumidity;
import forestry.api.core.EnumTemperature;
import forestry.api.core.IErrorLogic;
import forestry.api.core.IErrorState;
import forestry.api.genetics.IIndividual;
import forestry.api.lepidopterology.ButterflyManager;
import forestry.api.lepidopterology.IButterfly;
import forestry.api.lepidopterology.IButterflyCocoon;
import forestry.api.lepidopterology.IButterflyGenome;
import forestry.api.multiblock.IGreenhouseComponent;
import forestry.core.errors.EnumErrorCode;
import forestry.core.inventory.IInventoryAdapter;
import forestry.core.inventory.InventoryAnalyzer;
import forestry.core.network.IStreamableGui;
import forestry.core.network.PacketBufferForestry;
import forestry.core.utils.ClimateUtil;
import forestry.core.utils.InventoryUtil;
import forestry.energy.EnergyHelper;
import forestry.energy.EnergyManager;
import forestry.greenhouse.gui.ContainerGreenhouseNursery;
import forestry.greenhouse.gui.GuiGreenhouseNursery;
import forestry.greenhouse.inventory.InventoryGreenhouseNursery;
import forestry.greenhouse.multiblock.IGreenhouseControllerInternal;
import forestry.lepidopterology.items.ItemButterflyGE;
import net.minecraft.client.gui.inventory.GuiContainer;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.Container;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.ITickable;
import net.minecraft.util.NonNullList;
import net.minecraft.world.biome.Biome;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

public class TileGreenhouseNursery extends TileGreenhouse implements IGreenhouseComponent.Nursery, IStreamableGui, ITickable {
	private static final int WORK_TICK_INTERVAL = 5; // one Forestry work tick happens every WORK_TICK_INTERVAL game ticks
	private static final Random rand = new Random();
	private static final int ENERGY_PER_WORK_CYCLE = 3200;
	private static final int MATURE_TIME_MULTIPLIER = 250;

	private int workCounter;
	private int ticksPerWorkCycle;
	private int energyPerWorkCycle;
	@Nullable
	private IButterfly butterfly; 
	private IButterflyCocoon cocoon;

	private int tickCount = rand.nextInt(256);
	
	// the number of work ticks that this tile has had no power
	private int noPowerTime = 0;

	private final InventoryGreenhouseNursery inventory;

	public TileGreenhouseNursery() {
		this.inventory = new InventoryGreenhouseNursery(this);
		this.ticksPerWorkCycle = 4;
	}

	public int getWorkCounter() {
		return workCounter;
	}

	public void setTicksPerWorkCycle(int ticksPerWorkCycle) {
		this.ticksPerWorkCycle = ticksPerWorkCycle;
		this.workCounter = 0;
	}

	public int getTicksPerWorkCycle() {
		return ticksPerWorkCycle;
	}

	public void setEnergyPerWorkCycle(int energyPerWorkCycle) {
		this.energyPerWorkCycle = EnergyHelper.scaleForDifficulty(energyPerWorkCycle);
	}

	public int getEnergyPerWorkCycle() {
		return energyPerWorkCycle;
	}

	@Override
	public void func_73660_a() {
		if(field_145850_b.field_72995_K){
			return;
		}
		tickCount++;
		
		if (!updateOnInterval(WORK_TICK_INTERVAL)) {
			return;
		}

		IErrorLogic errorLogic = getErrorLogic();

		if(!canWork()){
			return;
		}
		
		EnergyManager energyManager = getMultiblockLogic().getController().getEnergyManager();
		if(energyManager == null){
			return;
		}
		int ticksPerWorkCycle = getTicksPerWorkCycle();

		if (workCounter < ticksPerWorkCycle) {
			int energyPerWorkCycle = getEnergyPerWorkCycle();
			boolean consumedEnergy = EnergyHelper.consumeEnergyToDoWork(energyManager, ticksPerWorkCycle, energyPerWorkCycle);
			if (consumedEnergy) {
				errorLogic.setCondition(false, EnumErrorCode.NO_POWER);
				workCounter++;
				noPowerTime = 0;
			} else {
				noPowerTime++;
				if (noPowerTime > 4) {
					errorLogic.setCondition(true, EnumErrorCode.NO_POWER);
				}
			}
		}

		if (workCounter >= ticksPerWorkCycle) {
			if (workCycle()) {
				workCounter = 0;
			}
		}
	}
	
	protected final boolean updateOnInterval(int tickInterval) {
		return tickCount % tickInterval == 0;
	}
	
	protected boolean canWork(){
		IErrorLogic errorLogic = getErrorLogic();
		Integer inputSlotIndex = getInputSlotIndex();
		boolean hasCocoon = inputSlotIndex != null || !func_70301_a(InventoryGreenhouseNursery.SLOT_WORK).func_190926_b();
		
		errorLogic.setCondition(!hasCocoon, EnumErrorCode.NO_RESOURCE_INVENTORY);
		if(butterfly == null && hasCocoon){
			ItemStack stack = func_70301_a(InventoryGreenhouseNursery.SLOT_WORK);
			butterfly = ButterflyManager.butterflyRoot.getMember(stack);
			if(butterfly == null){
				func_70299_a(InventoryGreenhouseNursery.SLOT_WORK, ItemStack.field_190927_a);
				errorLogic.setCondition(true, EnumErrorCode.NO_RESOURCE_INVENTORY);
				return false;
			}
		}
		if(canSpawnButterfly()){
			Set<IErrorState> queenErrors = butterfly.getCanSpawn(this, null);
			for (IErrorState errorState : queenErrors) {
				errorLogic.setCondition(true, errorState);
			}
		}else{
			Set<IErrorState> queenErrors = butterfly.getCanGrow(this, null);
			for (IErrorState errorState : queenErrors) {
				errorLogic.setCondition(true, errorState);
			}
		}
		return !errorLogic.hasErrors();
	}

	protected boolean workCycle(){
		moveCocoonToWorkSlot();
		if(butterfly != null){
			if(canSpawnButterfly()){
				IGreenhouseControllerInternal controller = getMultiblockLogic().getController();
				if(controller.spawnButterfly(this)){
					func_70299_a(InventoryGreenhouseNursery.SLOT_WORK, ItemStack.field_190927_a);
					setTicksPerWorkCycle(1);
					setEnergyPerWorkCycle(0);
					butterfly = null;
					cocoon = null;
				}
				return false;
			} else {
				ItemStack stack = func_70301_a(InventoryGreenhouseNursery.SLOT_WORK);
				int age = getAge();
				IButterflyGenome genome = butterfly.getGenome();
				float matureTime = genome.getLifespan() / (genome.getFertility() * 2);
				matureTime*=MATURE_TIME_MULTIPLIER;
				int caterpillarMatureTime = Math.round(matureTime);
				if(workCounter > 4){
					age++;
					ItemButterflyGE.setAge(stack, age);
				}
				setTicksPerWorkCycle(caterpillarMatureTime);
				setEnergyPerWorkCycle(ENERGY_PER_WORK_CYCLE);
			}
		}
		return true;
	}
	
	private boolean canSpawnButterfly(){
		int age = getAge();
		return age > 2 && butterfly != null;
	}
	
	private void moveCocoonToWorkSlot() {
		if (!func_70301_a(InventoryAnalyzer.SLOT_ANALYZE).func_190926_b()) {
			return;
		}

		Integer slotIndex = getInputSlotIndex();
		if (slotIndex == null) {
			return;
		}

		ItemStack inputStack = inventory.func_70301_a(slotIndex);
		if (inputStack.func_190926_b()) {
			return;
		}

		ItemStack stack = inputStack.func_77946_l();
		stack.func_190920_e(1);
		inputStack.func_190918_g(1);
		func_70299_a(InventoryGreenhouseNursery.SLOT_WORK, stack);
		if(inputStack.func_190926_b()){
			inventory.func_70299_a(slotIndex, ItemStack.field_190927_a);
		}
	}
	
	@Nullable
	private Integer getInputSlotIndex() {
		for (int slotIndex = 0; slotIndex < InventoryGreenhouseNursery.SLOT_INPUT_COUNT; slotIndex++) {
			ItemStack inputStack = inventory.func_70301_a(InventoryGreenhouseNursery.SLOT_INPUT_1 + slotIndex);
			if (!inputStack.func_190926_b()) {
				return InventoryGreenhouseNursery.SLOT_INPUT_1 + slotIndex;
			}
		}
		return null;
	}
	
	public int getAge(){
		ItemStack cocoon = func_70301_a(InventoryGreenhouseNursery.SLOT_WORK);
		if(cocoon.func_190926_b() || !cocoon.func_77942_o()){
			return -1;
		}
		return cocoon.func_77978_p().func_74762_e(ItemButterflyGE.NBT_AGE);
	}

	public int getProgressScaled(int i) {
		int ticksPerWorkCycle = getTicksPerWorkCycle();
		if (ticksPerWorkCycle == 0) {
			return 0;
		}

		return workCounter * i / ticksPerWorkCycle;
	}

	@Override
	public void writeGuiData(PacketBufferForestry data) {
		super.writeGuiData(data);
		data.func_150787_b(workCounter);
		data.func_150787_b(getTicksPerWorkCycle());
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void readGuiData(PacketBufferForestry data) throws IOException {
		super.readGuiData(data);
		workCounter = data.func_150792_a();
		ticksPerWorkCycle = data.func_150792_a();
	}
	
	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound data) {
		super.func_189515_b(data);
		data.func_74768_a("workCounter", workCounter);
		data.func_74768_a("ticksPerWorkCycle", ticksPerWorkCycle);
		data.func_74768_a("energyPerWorkCycle", energyPerWorkCycle);
		return data;
	}
	
	@Override
	public void func_145839_a(NBTTagCompound data) {
		super.func_145839_a(data);
		workCounter = data.func_74762_e("workCounter");
		ticksPerWorkCycle = data.func_74762_e("ticksPerWorkCycle");
		energyPerWorkCycle = data.func_74762_e("energyPerWorkCycle");
	}

	@Override
	public void addCocoonLoot(IButterflyCocoon cocoon, NonNullList<ItemStack> cocoonDrops) {
		for (ItemStack drop : cocoonDrops) {
			InventoryUtil.tryAddStack(this, drop, InventoryGreenhouseNursery.SLOT_OUTPUT_1, InventoryGreenhouseNursery.SLOT_OUTPUT_COUNT, true);
		}
	}

	/* IGuiHandlerTile */
	@Override
	@SideOnly(Side.CLIENT)
	public GuiContainer getGui(EntityPlayer player, int data) {
		return new GuiGreenhouseNursery(player, this);
	}

	@Override
	public Container getContainer(EntityPlayer player, int data) {
		return new ContainerGreenhouseNursery(player.field_71071_by, this);
	}
	
	/* ITitled */
	@Override
	public String getUnlocalizedTitle() {
		return "for.gui.greenhouse.nursery.title";
	}
	
	@Override
	public IInventoryAdapter getInternalInventory() {
		return inventory;
	}

	@Override
	public IButterfly getCaterpillar() {
		return butterfly;
	}

	@Override
	public IIndividual getNanny() {
		return null;
	}

	@Override
	public void setCaterpillar(IButterfly caterpillar) {
	}

	@Override
	public boolean canNurse(IButterfly caterpillar) {
		return false;
	}

	@Override
	public Biome getBiome() {
		return field_145850_b.func_180494_b(field_174879_c);
	}

	@Override
	public EnumTemperature getTemperature() {
		return EnumTemperature.getFromBiome(getBiome(), field_145850_b, field_174879_c);
	}

	@Override
	public EnumHumidity getHumidity() {
		return EnumHumidity.getFromValue(ClimateUtil.getHumidity(field_145850_b, field_174879_c));
	}
	
}
