package forestry.farming;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;

import forestry.api.core.ForestryAPI;
import forestry.api.core.ISpriteRegister;
import forestry.api.farming.Farmables;
import forestry.api.farming.IFarmLogic;
import forestry.api.farming.IFarmRegistry;
import forestry.api.farming.IFarmable;
import forestry.api.farming.ISimpleFarmLogic;
import forestry.core.config.LocalizedConfiguration;
import forestry.core.render.TextureManagerForestry;
import forestry.core.render.TextureMapForestry;
import forestry.core.utils.ItemStackUtil;
import forestry.core.utils.Log;
import forestry.core.utils.Translator;
import forestry.farming.logic.FarmLogicSimple;
import net.minecraft.item.ItemStack;
import net.minecraftforge.common.config.Property;
import net.minecraftforge.oredict.OreDictionary;
import scala.actors.threadpool.Arrays;

public final class FarmRegistry implements IFarmRegistry {

	private static final FarmRegistry INSTANCE = new FarmRegistry();
	private final Multimap<String, IFarmable> farmables = HashMultimap.create();
	private final Map<ItemStack, Integer> fertilizers = new LinkedHashMap<>();

	static {
		ForestryAPI.farmRegistry = INSTANCE;
	}

	public static FarmRegistry getInstance() {
		return INSTANCE;
	}
	
	@Override
	public void registerFarmables(String identifier, IFarmable... farmablesArray) {
		farmables.putAll(identifier, Arrays.asList(farmablesArray));
	}

	@Override
	public Collection<IFarmable> getFarmables(String identifier) {
		return farmables.get(identifier);
	}

	@Override
	public IFarmLogic createLogic(ISimpleFarmLogic simpleFarmLogic) {
		return new FarmLogicSimple(simpleFarmLogic);
	}

	@Override
	public void registerFertilizer(ItemStack itemStack, int value) {
		fertilizers.put(itemStack, value);
	}

	@Override
	public int getFertilizeValue(ItemStack itemStack) {
		for (Entry<ItemStack, Integer> fertilizer : fertilizers.entrySet()) {
			ItemStack fertilizerStack = fertilizer.getKey();
			if (itemStack.func_77973_b() == fertilizerStack.func_77973_b() 
					&& (fertilizerStack.func_77952_i() == itemStack.func_77952_i() 
					|| fertilizerStack.func_77952_i() == OreDictionary.WILDCARD_VALUE)) {
				return fertilizer.getValue();
			}
		}
		return 0;
	}
	
	public void loadConfig(LocalizedConfiguration config) {
		List<String> defaultFertilizers = new ArrayList<>(getItemStrings());
		Collections.sort(defaultFertilizers);
		String[] defaultSortedFertilizers = defaultFertilizers.toArray(new String[defaultFertilizers.size()]);
		Property fertilizerConf = config.get("fertilizers", "items", defaultSortedFertilizers, Translator.translateToLocal("for.config.farm.fertilizers.items"));

		fertilizers.clear();
		String[] fertilizerList = fertilizerConf.getStringList();
		for (int i = 0; i < fertilizerList.length; i++) {
			try {
				String fertilizer = fertilizerList[i];
				String[] fertilizers = fertilizer.split(";");
				ItemStack fertilizerItem = ItemStackUtil.parseItemStackString(fertilizers[0], OreDictionary.WILDCARD_VALUE);
				int fertilizerValue = Integer.parseInt(fertilizers[1]);
				registerFertilizer(fertilizerItem, fertilizerValue);
			} catch (Exception e) {
				Log.error("Forestry failed to parse a fertilizer entry at the farm config, at the position " + i + ".", e);
			}
		}
	}
	
	private Set<String> getItemStrings() {
		Set<String> itemStrings = new HashSet<>(fertilizers.size());
		for (Entry<ItemStack, Integer> itemStack : fertilizers.entrySet()) {
			String itemString = ItemStackUtil.getStringForItemStack(itemStack.getKey());
			itemString += ";" + itemStack.getValue();
			itemStrings.add(itemString);
		}
		return itemStrings;
	}
	
}
