/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.core.render;

import net.minecraft.block.Block;
import net.minecraft.block.state.IBlockState;
import net.minecraft.client.Minecraft;
import net.minecraft.client.particle.ParticleDigging;
import net.minecraft.client.particle.ParticleManager;
import net.minecraft.client.renderer.BlockModelShapes;
import net.minecraft.client.renderer.BlockRendererDispatcher;
import net.minecraft.client.renderer.texture.TextureAtlasSprite;
import net.minecraft.util.EnumBlockRenderType;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumParticleTypes;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

/**
 * @author CovertJaguar <http://www.railcraft.info>
 */
public class ParticleHelper {

	@SideOnly(Side.CLIENT)
	public static boolean addBlockHitEffects(World world, BlockPos pos, EnumFacing side, ParticleManager effectRenderer, Callback callback) {
		IBlockState iblockstate = world.func_180495_p(pos);
		if (iblockstate.func_185901_i() != EnumBlockRenderType.INVISIBLE) {
			int x = pos.func_177958_n();
			int y = pos.func_177956_o();
			int z = pos.func_177952_p();
			float f = 0.1F;
			AxisAlignedBB axisalignedbb = iblockstate.func_185900_c(world, pos);
			double px = x + world.field_73012_v.nextDouble() * (axisalignedbb.field_72336_d - axisalignedbb.field_72340_a - f * 2.0F) + f + axisalignedbb.field_72340_a;
			double py = y + world.field_73012_v.nextDouble() * (axisalignedbb.field_72337_e - axisalignedbb.field_72338_b - f * 2.0F) + f + axisalignedbb.field_72338_b;
			double pz = z + world.field_73012_v.nextDouble() * (axisalignedbb.field_72334_f - axisalignedbb.field_72339_c - f * 2.0F) + f + axisalignedbb.field_72339_c;
			if (side == EnumFacing.DOWN) {
				py = y + axisalignedbb.field_72338_b - f;
			}

			if (side == EnumFacing.UP) {
				py = y + axisalignedbb.field_72337_e + f;
			}

			if (side == EnumFacing.NORTH) {
				pz = z + axisalignedbb.field_72339_c - f;
			}

			if (side == EnumFacing.SOUTH) {
				pz = z + axisalignedbb.field_72334_f + f;
			}

			if (side == EnumFacing.WEST) {
				px = x + axisalignedbb.field_72340_a - f;
			}

			if (side == EnumFacing.EAST) {
				px = x + axisalignedbb.field_72336_d + f;
			}

			ParticleDigging fx = (ParticleDigging) effectRenderer.func_178927_a(EnumParticleTypes.BLOCK_DUST.func_179348_c(), px, py, pz, 0.0D, 0.0D, 0.0D, Block.func_176210_f(iblockstate));
			if (fx != null) {
				callback.addHitEffects(fx, world, pos, iblockstate);
				effectRenderer.func_78873_a(fx.func_174846_a(new BlockPos(x, y, z)).func_70543_e(0.2F).func_70541_f(0.6F));
			}
		}
		return true;
	}

	/**
	 * Spawn particles for when the block is destroyed. Due to the nature of how
	 * this is invoked, the x/y/z locations are not always guaranteed to host
	 * your block. So be sure to do proper sanity checks before assuming that
	 * the location is this block.
	 *
	 * @return True to prevent vanilla break particles from spawning.
	 */
	@SideOnly(Side.CLIENT)
	public static boolean addDestroyEffects(World world, Block block, IBlockState state, BlockPos pos, ParticleManager effectRenderer, Callback callback) {
		if (block != state.func_177230_c()) {
			return true;
		}

		byte iterations = 4;
		for (int i = 0; i < iterations; ++i) {
			for (int j = 0; j < iterations; ++j) {
				for (int k = 0; k < iterations; ++k) {
					double px = pos.func_177958_n() + (i + 0.5D) / iterations;
					double py = pos.func_177956_o() + (j + 0.5D) / iterations;
					double pz = pos.func_177952_p() + (k + 0.5D) / iterations;

					ParticleDigging fx = (ParticleDigging) effectRenderer.func_178927_a(EnumParticleTypes.BLOCK_CRACK.func_179348_c(), px, py, pz, px - pos.func_177958_n() - 0.5D, py - pos.func_177956_o() - 0.5D, pz - pos.func_177952_p() - 0.5D, Block.func_176210_f(state));
					if (fx != null) {
						callback.addDestroyEffects(fx, world, pos, state);
						effectRenderer.func_78873_a(fx.func_174846_a(pos));
					}
				}
			}
		}

		return true;
	}

	public interface Callback {

		@SideOnly(Side.CLIENT)
		void addHitEffects(ParticleDigging fx, World world, BlockPos pos, IBlockState state);

		@SideOnly(Side.CLIENT)
		void addDestroyEffects(ParticleDigging fx, World world, BlockPos pos, IBlockState state);
	}

	public static class DefaultCallback<B extends Block> implements ParticleHelper.Callback {

		protected final B block;

		public DefaultCallback(B block) {
			this.block = block;
		}

		@Override
		@SideOnly(Side.CLIENT)
		public void addHitEffects(ParticleDigging fx, World world, BlockPos pos, IBlockState state) {
			setTexture(fx, world, pos, state);
		}

		@Override
		@SideOnly(Side.CLIENT)
		public void addDestroyEffects(ParticleDigging fx, World world, BlockPos pos, IBlockState state) {
			setTexture(fx, world, pos, state);
		}

		@SideOnly(Side.CLIENT)
		protected void setTexture(ParticleDigging fx, World world, BlockPos pos, IBlockState state) {
			Minecraft minecraft = Minecraft.func_71410_x();
			BlockRendererDispatcher blockRendererDispatcher = minecraft.func_175602_ab();
			BlockModelShapes blockModelShapes = blockRendererDispatcher.func_175023_a();
			TextureAtlasSprite texture = blockModelShapes.func_178122_a(state);
			fx.func_187117_a(texture);
		}
	}
}
