/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.core.inventory;

import com.google.common.collect.ImmutableSet;
import forestry.api.core.ForestryAPI;
import forestry.api.core.IErrorSource;
import forestry.api.core.IErrorState;
import forestry.api.genetics.AlleleManager;
import forestry.api.genetics.IBreedingTracker;
import forestry.api.genetics.IIndividual;
import forestry.api.genetics.ISpeciesRoot;
import forestry.apiculture.PluginApiculture;
import forestry.apiculture.items.ItemRegistryApiculture;
import forestry.core.errors.EnumErrorCode;
import forestry.core.utils.GeneticsUtil;
import forestry.plugins.ForestryPluginUids;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;

public class ItemInventoryAlyzer extends ItemInventory implements IErrorSource {
	public static final int SLOT_ENERGY = 0;
	public static final int SLOT_SPECIMEN = 1;
	public static final int SLOT_ANALYZE_1 = 2;
	public static final int SLOT_ANALYZE_2 = 3;
	public static final int SLOT_ANALYZE_3 = 4;
	public static final int SLOT_ANALYZE_4 = 5;
	public static final int SLOT_ANALYZE_5 = 6;

	public ItemInventoryAlyzer(EntityPlayer player, ItemStack itemstack) {
		super(player, 7, itemstack);
	}

	public static boolean isAlyzingFuel(ItemStack itemstack) {
		if (itemstack.func_190926_b()) {
			return false;
		}

		if (ForestryAPI.enabledPlugins.contains(ForestryPluginUids.APICULTURE)) {
			ItemRegistryApiculture beeItems = PluginApiculture.getItems();

			Item item = itemstack.func_77973_b();
			return beeItems.honeyDrop == item || beeItems.honeydew == item;
		}

		return false;
	}

	@Override
	public boolean canSlotAccept(int slotIndex, ItemStack itemStack) {
		if (slotIndex == SLOT_ENERGY) {
			return isAlyzingFuel(itemStack);
		}

		// only allow one slot to be used at a time
		if (hasSpecimen() && func_70301_a(slotIndex).func_190926_b()) {
			return false;
		}

		itemStack = GeneticsUtil.convertToGeneticEquivalent(itemStack);
		ISpeciesRoot speciesRoot = AlleleManager.alleleRegistry.getSpeciesRoot(itemStack);
		if (speciesRoot == null) {
			return false;
		}

		if (slotIndex == SLOT_SPECIMEN) {
			return true;
		}

		IIndividual individual = speciesRoot.getMember(itemStack);
		return individual != null && individual.isAnalyzed();
	}

	@Override
	public void func_70299_a(int index, ItemStack itemStack) {
		super.func_70299_a(index, itemStack);
		if (index == SLOT_SPECIMEN) {
			analyzeSpecimen(itemStack);
		}
	}

	private void analyzeSpecimen(ItemStack specimen) {
		if (specimen.func_190926_b()) {
			return;
		}

		ItemStack convertedSpecimen = GeneticsUtil.convertToGeneticEquivalent(specimen);
		if (!ItemStack.func_77989_b(specimen, convertedSpecimen)) {
			func_70299_a(SLOT_SPECIMEN, convertedSpecimen);
			specimen = convertedSpecimen;
		}

		ISpeciesRoot speciesRoot = AlleleManager.alleleRegistry.getSpeciesRoot(specimen);

		// No individual, abort
		if (speciesRoot == null) {
			return;
		}

		IIndividual individual = speciesRoot.getMember(specimen);

		// Analyze if necessary
		if (individual != null && !individual.isAnalyzed()) {
			final boolean requiresEnergy = ForestryAPI.enabledPlugins.contains(ForestryPluginUids.APICULTURE);
			if (requiresEnergy) {
				// Requires energy
				if (!isAlyzingFuel(func_70301_a(SLOT_ENERGY))) {
					return;
				}
			}

			if (individual.analyze()) {
				IBreedingTracker breedingTracker = speciesRoot.getBreedingTracker(player.field_70170_p, player.func_146103_bH());
				breedingTracker.registerSpecies(individual.getGenome().getPrimary());
				breedingTracker.registerSpecies(individual.getGenome().getSecondary());

				NBTTagCompound nbttagcompound = new NBTTagCompound();
				individual.writeToNBT(nbttagcompound);
				specimen.func_77982_d(nbttagcompound);

				if (requiresEnergy) {
					// Decrease energy
					func_70298_a(SLOT_ENERGY, 1);
				}
			}
		}

		func_70299_a(SLOT_ANALYZE_1, specimen);
		func_70299_a(SLOT_SPECIMEN, ItemStack.field_190927_a);
	}

	@Override
	public final ImmutableSet<IErrorState> getErrorStates() {
		ImmutableSet.Builder<IErrorState> errorStates = ImmutableSet.builder();

		if (!hasSpecimen()) {
			errorStates.add(EnumErrorCode.NO_SPECIMEN);
		} else {
			ISpeciesRoot speciesRoot = AlleleManager.alleleRegistry.getSpeciesRoot(getSpecimen());
			if (speciesRoot != null && !isAlyzingFuel(func_70301_a(SLOT_ENERGY))) {
				errorStates.add(EnumErrorCode.NO_HONEY);
			}
		}

		return errorStates.build();
	}

	public ItemStack getSpecimen() {
		for (int i = SLOT_SPECIMEN; i <= SLOT_ANALYZE_5; i++) {
			ItemStack itemStack = func_70301_a(i);
			if (!itemStack.func_190926_b()) {
				return itemStack;
			}
		}
		return ItemStack.field_190927_a;
	}

	public boolean hasSpecimen() {
		return !getSpecimen().func_190926_b();
	}
}
