/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.greenhouse.gui;

import java.io.IOException;

import com.google.common.base.Predicate;
import com.google.common.primitives.Floats;
import forestry.api.climate.IClimateControlProvider;
import forestry.api.climate.IClimateInfo;
import forestry.api.core.CamouflageManager;
import forestry.api.multiblock.IGreenhouseController;
import forestry.core.climate.ClimateInfo;
import forestry.core.config.Constants;
import forestry.core.gui.GuiForestryTitled;
import forestry.core.gui.ledgers.Ledger;
import forestry.core.gui.widgets.TankWidget;
import forestry.core.network.packets.PacketUpdateClimateControl;
import forestry.core.render.ColourProperties;
import forestry.core.render.TextureManagerForestry;
import forestry.core.utils.NetworkUtil;
import forestry.core.utils.Translator;
import forestry.greenhouse.multiblock.IGreenhouseControllerInternal;
import forestry.greenhouse.tiles.TileGreenhouse;
import net.minecraft.client.gui.GuiTextField;
import net.minecraft.entity.player.EntityPlayer;

public class GuiGreenhouse extends GuiForestryTitled<ContainerGreenhouse> {

	private static final Predicate<String> numberFilter = text -> {
		if (text == null) {
			return false;
		}
		if (!text.isEmpty() && text.length() > 1) {
			if (!text.contains(".")) {
				return false;
			}
		}
		if (text.length() > 7) {
			return false;
		}
		Float f = Floats.tryParse(text);
		return text.isEmpty() || f != null && Floats.isFinite(f) && f >= 0.0F;
	};

	private final TileGreenhouse tile;
	private GuiTextField humidityField;
	private GuiTextField temperatureField;
	private boolean fieldsEnabeled;

	public GuiGreenhouse(EntityPlayer player, TileGreenhouse tile) {
		super(Constants.TEXTURE_PATH_GUI + "/greenhouse.png", new ContainerGreenhouse(player.field_71071_by, tile), tile);
		this.tile = tile;

		//Add the water tank
		widgetManager.add(new TankWidget(widgetManager, 152, 16, 0).setOverlayOrigin(176, 0));
		//Add the multiblock camouflage tabs
		WidgetCamouflageTab previous;
		int x = 3;
		widgetManager.add(previous = new WidgetCamouflageTab(widgetManager, field_147003_i + x, field_147009_r - 25, tile.getMultiblockLogic().getController(), tile, CamouflageManager.BLOCK));
		x+=50 + (previous.getHandlerSlot()  != null ? 20 : 0);
		widgetManager.add(previous = new WidgetCamouflageTab(widgetManager, field_147003_i + x, field_147009_r - 25, tile.getMultiblockLogic().getController(), tile, CamouflageManager.GLASS));
		x+=50 + (previous.getHandlerSlot()  != null ? 20 : 0);
		widgetManager.add(new WidgetCamouflageTab(widgetManager, field_147003_i + x, field_147009_r - 25, tile.getMultiblockLogic().getController(), tile, CamouflageManager.DOOR));
		
		widgetManager.add(new WidgetClimatePillar(widgetManager, field_147003_i- 23, field_147009_r + 5));
		
		fieldsEnabeled = true;
	}

	@Override
	public void func_73866_w_() {
		super.func_73866_w_();

		temperatureField = new GuiTextField(0, field_146289_q, field_147003_i + 64, field_147009_r + 31, 50, 10);
		humidityField = new GuiTextField(1, field_146289_q, field_147003_i + 64, field_147009_r + 61, 50, 10);

		temperatureField.func_175205_a(numberFilter);
		humidityField.func_175205_a(numberFilter);
		
		temperatureField.func_146185_a(false);
		humidityField.func_146185_a(false);

		IGreenhouseController controller = tile.getMultiblockLogic().getController();
		if (controller == null || controller.getControlClimate() == ClimateInfo.MAX) {
			temperatureField.func_146184_c(false);
			temperatureField.func_146189_e(false);
			humidityField.func_146184_c(false);
			humidityField.func_146189_e(false);
			fieldsEnabeled = false;
		} else {
			IClimateInfo info = controller.getControlClimate();
			temperatureField.func_146180_a(Float.toString(info.getTemperature()));
			humidityField.func_146180_a(Float.toString(info.getHumidity()));
		}
	}

	@Override
	protected void func_73869_a(char typedChar, int keyCode) throws IOException {
		if (fieldsEnabeled) {
			if (!humidityField.func_146201_a(typedChar, keyCode) && !temperatureField.func_146201_a(typedChar, keyCode)) {
				super.func_73869_a(typedChar, keyCode);
			}
		} else {
			super.func_73869_a(typedChar, keyCode);
		}
	}

	@Override
	protected void func_73864_a(int mouseX, int mouseY, int mouseButton) throws IOException {
		super.func_73864_a(mouseX, mouseY, mouseButton);
		if (fieldsEnabeled) {
			boolean temperatureWasFocused = temperatureField.func_146206_l();
			boolean humidityWasFocused = humidityField.func_146206_l();
			temperatureField.func_146192_a(mouseX, mouseY, mouseButton);
			humidityField.func_146192_a(mouseX, mouseY, mouseButton);
			IClimateControlProvider provider = tile.getMultiblockLogic().getController();
			if (temperatureWasFocused && !temperatureField.func_146206_l() || humidityWasFocused && !humidityField.func_146206_l()) {
				float temp = parseField(temperatureField);
				float hum = parseField(humidityField);
				setClimate(provider, temp, hum);
			}
			NetworkUtil.sendToServer(new PacketUpdateClimateControl(provider));
		}
	}
	
	public void setClimate(float temp){
		setClimate(tile.getMultiblockLogic().getController(), temp);
	}
	
	public void setClimate(IClimateControlProvider provider, float temp){
		IClimateInfo info = provider.getControlClimate();
		setClimate(provider, temp, info.getHumidity());
	}
	
	public void setClimate(IClimateControlProvider provider, float temp, float hum){
		temperatureField.func_146180_a(Float.toString(temp));
		humidityField.func_146180_a(Float.toString(hum));
		provider.setControlClimate(new ClimateInfo(temp, hum));
	}

	private float parseField(GuiTextField field) {
		String text = field.func_146179_b();
		if (text.isEmpty()) {
			return 2.0F;
		}
		try {
			float f = Float.parseFloat(text);
			if (f >= 2) {
				f = 2.0F;
			}
			return f;
		} catch (NumberFormatException var5) {
			return 2.0F;
		}
	}

	@Override
	protected void func_146976_a(float f, int mouseX, int mouseY) {
		super.func_146976_a(f, mouseX, mouseY);

		
		if (fieldsEnabeled) {		
			func_73729_b(field_147003_i + 62, field_147009_r + 29, 204, 115, 52, 12);
			func_73729_b(field_147003_i + 62, field_147009_r + 59, 204, 127, 52, 12);
		
			textLayout.line = 20;
			textLayout.drawCenteredLine(Translator.translateToLocal("for.gui.temperature"), 0, ColourProperties.INSTANCE.get("gui.greenhouse.temperature.text"));
			temperatureField.func_146194_f();
			textLayout.line = 50;
			textLayout.drawCenteredLine(Translator.translateToLocal("for.gui.humidity"), 0, ColourProperties.INSTANCE.get("gui.greenhouse.humidity.text"));

			humidityField.func_146194_f();
		}
	}

	@Override
	protected void addLedgers() {
		IGreenhouseControllerInternal greenhouseController = tile.getMultiblockLogic().getController();

		addErrorLedger(greenhouseController);
		ledgerManager.add(new EnergyLedger());
		addClimateLedger(greenhouseController);
		addHintLedger("greenhouse");
		addOwnerLedger(tile);
	}

	protected class EnergyLedger extends Ledger {

		public EnergyLedger() {
			super(ledgerManager, "power");
			maxHeight = 48;
		}

		@Override
		public void draw(int x, int y) {

			// Draw background
			drawBackground(x, y);

			// Draw icon
			drawSprite(TextureManagerForestry.getInstance().getDefault("misc/energy"), x + 3, y + 4);

			if (!isFullyOpened()) {
				return;
			}

			drawHeader(Translator.translateToLocal("for.gui.energy"), x + 22, y + 8);

			drawSubheader(Translator.translateToLocal("for.gui.stored") + ':', x + 22, y + 20);
			drawText(tile.getMultiblockLogic().getController().getEnergyManager().getEnergyStored() + " RF", x + 22, y + 32);
		}

		@Override
		public String getTooltip() {
			return Translator.translateToLocal("for.gui.energy") + ": " + tile.getMultiblockLogic().getController().getEnergyManager().getEnergyStored() + " RF/t";
		}
	}
}
