/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.apiculture.blocks;

import java.util.ArrayList;
import java.util.EnumMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import forestry.api.core.IModelManager;
import forestry.api.core.IStateMapperRegister;
import forestry.api.core.Tabs;
import forestry.apiculture.MaterialBeehive;
import forestry.apiculture.multiblock.IAlvearyControllerInternal;
import forestry.apiculture.multiblock.TileAlveary;
import forestry.apiculture.multiblock.TileAlvearyFan;
import forestry.apiculture.multiblock.TileAlvearyHeater;
import forestry.apiculture.multiblock.TileAlvearyHygroregulator;
import forestry.apiculture.multiblock.TileAlvearyPlain;
import forestry.apiculture.multiblock.TileAlvearySieve;
import forestry.apiculture.multiblock.TileAlvearyStabiliser;
import forestry.apiculture.multiblock.TileAlvearySwarmer;
import forestry.apiculture.network.packets.PacketAlvearyChange;
import forestry.core.blocks.BlockStructure;
import forestry.core.tiles.IActivatable;
import forestry.core.tiles.TileUtil;
import forestry.core.utils.BlockUtil;
import forestry.core.utils.NetworkUtil;
import forestry.core.utils.Translator;
import net.minecraft.block.Block;
import net.minecraft.block.SoundType;
import net.minecraft.block.properties.PropertyEnum;
import net.minecraft.block.state.BlockStateContainer;
import net.minecraft.block.state.IBlockState;
import net.minecraft.client.renderer.block.model.ModelResourceLocation;
import net.minecraft.client.renderer.block.statemap.StateMapperBase;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.IStringSerializable;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;
import net.minecraftforge.client.model.ModelLoader;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

public abstract class BlockAlveary extends BlockStructure implements IStateMapperRegister {
	private static final PropertyEnum<State> STATE = PropertyEnum.func_177709_a("state", State.class);
	private static final PropertyEnum<AlvearyPlainType> PLAIN_TYPE = PropertyEnum.func_177709_a("type", AlvearyPlainType.class);

	private enum State implements IStringSerializable {
		ON, OFF;

		@Override
		public String func_176610_l() {
			return name().toLowerCase(Locale.ENGLISH);
		}
	}

	private enum AlvearyPlainType implements IStringSerializable {
		NORMAL, ENTRANCE, ENTRANCE_LEFT, ENTRANCE_RIGHT;

		@Override
		public String func_176610_l() {
			return name().toLowerCase(Locale.ENGLISH);
		}
	}

	public static Map<BlockAlvearyType, BlockAlveary> create() {
		Map<BlockAlvearyType, BlockAlveary> blockMap = new EnumMap<>(BlockAlvearyType.class);
		for (final BlockAlvearyType type : BlockAlvearyType.VALUES) {
			BlockAlveary block = new BlockAlveary() {
				@Override
				public BlockAlvearyType getAlvearyType() {
					return type;
				}
			};
			blockMap.put(type, block);
		}
		return blockMap;
	}

	public BlockAlveary() {
		super(new MaterialBeehive(false));

		BlockAlvearyType alvearyType = getAlvearyType();
		IBlockState defaultState = this.field_176227_L.func_177621_b();
		if (alvearyType == BlockAlvearyType.PLAIN) {
			defaultState = defaultState.func_177226_a(PLAIN_TYPE, AlvearyPlainType.NORMAL);
		} else if (alvearyType.activatable) {
			defaultState = defaultState.func_177226_a(STATE, State.OFF);
		}
		func_180632_j(defaultState);

		func_149711_c(1.0f);
		func_149647_a(Tabs.tabApiculture);
		setHarvestLevel("axe", 0);
		func_149672_a(SoundType.field_185848_a);
	}

	public abstract BlockAlvearyType getAlvearyType();

	@Override
	public boolean isNormalCube(IBlockState state, IBlockAccess world, BlockPos pos) {
		return true;
	}

	@Override
	public int func_176201_c(IBlockState state) {
		return 0;
	}

	@Override
	public TileEntity func_149915_a(World world, int meta) {
		BlockAlvearyType type = getAlvearyType();
		switch (type) {
			case SWARMER:
				return new TileAlvearySwarmer();
			case FAN:
				return new TileAlvearyFan();
			case HEATER:
				return new TileAlvearyHeater();
			case HYGRO:
				return new TileAlvearyHygroregulator();
			case STABILISER:
				return new TileAlvearyStabiliser();
			case SIEVE:
				return new TileAlvearySieve();
			case PLAIN:
			default:
				return new TileAlvearyPlain();
		}
	}

	/* ITEM MODELS */
	@SideOnly(Side.CLIENT)
	@Override
	public void registerModel(Item item, IModelManager manager) {
		manager.registerItemModel(item, 0, "apiculture/alveary." + getAlvearyType());
	}

	@Override
	protected BlockStateContainer func_180661_e() {
		BlockAlvearyType alvearyType = getAlvearyType();

		if (alvearyType == BlockAlvearyType.PLAIN) {
			return new BlockStateContainer(this, PLAIN_TYPE);
		} else if (alvearyType.activatable) {
			return new BlockStateContainer(this, STATE);
		} else {
			return new BlockStateContainer(this);
		}
	}

	@Override
	public IBlockState func_176221_a(IBlockState state, IBlockAccess world, BlockPos pos) {
		TileAlveary tile = TileUtil.getTile(world, pos, TileAlveary.class);
		if (tile == null) {
			return super.func_176221_a(state, world, pos);
		}

		if (tile instanceof IActivatable) {
			if (((IActivatable) tile).isActive()) {
				state = state.func_177226_a(STATE, State.ON);
			} else {
				state = state.func_177226_a(STATE, State.OFF);
			}
		} else if (getAlvearyType() == BlockAlvearyType.PLAIN) {
			if (!tile.getMultiblockLogic().getController().isAssembled()) {
				state = state.func_177226_a(PLAIN_TYPE, AlvearyPlainType.NORMAL);
			} else {
				IBlockState blockStateAbove = world.func_180495_p(pos.func_177984_a());
				Block blockAbove = blockStateAbove.func_177230_c();
				if (BlockUtil.isWoodSlabBlock(blockStateAbove, blockAbove, world, pos)) {
					List<EnumFacing> blocksTouching = getBlocksTouching(world, pos);
					switch (blocksTouching.size()) {
						case 3:
							state = state.func_177226_a(PLAIN_TYPE, AlvearyPlainType.ENTRANCE);
							break;
						case 2:
							if (blocksTouching.contains(EnumFacing.SOUTH) && blocksTouching.contains(EnumFacing.EAST) ||
									blocksTouching.contains(EnumFacing.NORTH) && blocksTouching.contains(EnumFacing.WEST)) {
								state = state.func_177226_a(PLAIN_TYPE, AlvearyPlainType.ENTRANCE_LEFT);
							} else {
								state = state.func_177226_a(PLAIN_TYPE, AlvearyPlainType.ENTRANCE_RIGHT);
							}
							break;
						default:
							state = state.func_177226_a(PLAIN_TYPE, AlvearyPlainType.NORMAL);
							break;
					}
				} else {
					state = state.func_177226_a(PLAIN_TYPE, AlvearyPlainType.NORMAL);
				}
			}
		}

		return super.func_176221_a(state, world, pos);
	}


	private static List<EnumFacing> getBlocksTouching(IBlockAccess world, BlockPos blockPos) {
		List<EnumFacing> touching = new ArrayList<>();
		for (EnumFacing direction : EnumFacing.field_176754_o) {
			IBlockState blockState = world.func_180495_p(blockPos.func_177972_a(direction));
			if (blockState.func_177230_c() instanceof BlockAlveary) {
				touching.add(direction);
			}
		}
		return touching;
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void registerStateMapper() {
		ModelLoader.setCustomStateMapper(this, new AlvearyStateMapper(getAlvearyType()));
	}

	@SideOnly(Side.CLIENT)
	private static class AlvearyStateMapper extends StateMapperBase {
		private final BlockAlvearyType type;

		public AlvearyStateMapper(BlockAlvearyType type) {
			this.type = type;
		}

		@Override
		protected ModelResourceLocation func_178132_a(IBlockState state) {
			String resourceDomain = Block.field_149771_c.func_177774_c(state.func_177230_c()).func_110624_b();
			String resourceLocation = "apiculture/alveary_" + type;
			String propertyString = func_178131_a(state.func_177228_b());
			return new ModelResourceLocation(resourceDomain + ':' + resourceLocation, propertyString);
		}

	}

	@Override
	public void func_189540_a(IBlockState state, World worldIn, BlockPos pos, Block blockIn, BlockPos fromPos) {
		TileEntity tileEntity = worldIn.func_175625_s(pos);
		if (tileEntity instanceof TileAlveary) {
			TileAlveary tileAlveary = (TileAlveary) tileEntity;

			// We must check that the slabs on top were not removed
			IAlvearyControllerInternal alveary = tileAlveary.getMultiblockLogic().getController();
			alveary.reassemble();
			BlockPos referenceCoord = alveary.getReferenceCoord();
			NetworkUtil.sendNetworkPacket(new PacketAlvearyChange(referenceCoord), referenceCoord, worldIn);
		}
	}
	
	@SideOnly(Side.CLIENT)
	@Override
	public void func_190948_a(ItemStack stack, EntityPlayer player, List<String> tooltip, boolean advanced) {
		tooltip.add(Translator.translateToLocal("tile.for.alveary.tooltip"));
	}
}
