/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.factory;

import javax.annotation.Nullable;
import java.util.Set;

import com.google.common.base.Preconditions;
import forestry.api.circuits.ChipsetManager;
import forestry.api.circuits.CircuitSocketType;
import forestry.api.circuits.ICircuit;
import forestry.api.circuits.ICircuitLayout;
import forestry.api.core.ForestryAPI;
import forestry.api.fuels.EngineBronzeFuel;
import forestry.api.fuels.EngineCopperFuel;
import forestry.api.fuels.FermenterFuel;
import forestry.api.fuels.FuelManager;
import forestry.api.fuels.MoistenerFuel;
import forestry.api.fuels.RainSubstrate;
import forestry.api.recipes.RecipeManagers;
import forestry.apiculture.PluginApiculture;
import forestry.apiculture.items.ItemRegistryApiculture;
import forestry.core.PluginCore;
import forestry.core.PluginFluids;
import forestry.core.blocks.BlockBogEarth;
import forestry.core.blocks.BlockRegistryCore;
import forestry.core.circuits.CircuitLayout;
import forestry.core.circuits.Circuits;
import forestry.core.circuits.EnumCircuitBoardType;
import forestry.core.circuits.ItemCircuitBoard;
import forestry.core.config.Constants;
import forestry.core.fluids.Fluids;
import forestry.core.items.EnumElectronTube;
import forestry.core.items.ItemElectronTube;
import forestry.core.items.ItemRegistryCore;
import forestry.core.items.ItemRegistryFluids;
import forestry.core.network.IPacketRegistry;
import forestry.core.recipes.RecipeUtil;
import forestry.core.utils.datastructures.FluidMap;
import forestry.core.utils.datastructures.ItemStackMap;
import forestry.factory.blocks.BlockRegistryFactory;
import forestry.factory.circuits.CircuitSpeedUpgrade;
import forestry.factory.network.PacketRegistryFactory;
import forestry.factory.recipes.CarpenterRecipeManager;
import forestry.factory.recipes.CentrifugeRecipeManager;
import forestry.factory.recipes.FabricatorRecipeManager;
import forestry.factory.recipes.FabricatorSmeltingRecipeManager;
import forestry.factory.recipes.FermenterRecipeManager;
import forestry.factory.recipes.MoistenerRecipeManager;
import forestry.factory.recipes.SqueezerRecipeManager;
import forestry.factory.recipes.StillRecipeManager;
import forestry.plugins.BlankForestryPlugin;
import forestry.plugins.ForestryPlugin;
import forestry.plugins.ForestryPluginUids;
import forestry.storage.PluginStorage;
import net.minecraft.init.Blocks;
import net.minecraft.init.Items;
import net.minecraft.item.ItemStack;
import net.minecraft.util.NonNullList;
import net.minecraftforge.fluids.Fluid;
import net.minecraftforge.fluids.FluidRegistry;
import net.minecraftforge.fluids.FluidStack;

@ForestryPlugin(pluginID = ForestryPluginUids.FACTORY, name = "Factory", author = "SirSengir", url = Constants.URL, unlocalizedDescription = "for.plugin.factory.description")
public class PluginFactory extends BlankForestryPlugin {
	@Nullable
	private static BlockRegistryFactory blocks;

	public static BlockRegistryFactory getBlocks() {
		Preconditions.checkState(blocks != null);
		return blocks;
	}

	@Override
	public void setupAPI() {
		RecipeManagers.carpenterManager = new CarpenterRecipeManager();
		RecipeManagers.centrifugeManager = new CentrifugeRecipeManager();
		RecipeManagers.fabricatorManager = new FabricatorRecipeManager();
		RecipeManagers.fabricatorSmeltingManager = new FabricatorSmeltingRecipeManager();
		RecipeManagers.fermenterManager = new FermenterRecipeManager();
		RecipeManagers.moistenerManager = new MoistenerRecipeManager();
		RecipeManagers.squeezerManager = new SqueezerRecipeManager();
		RecipeManagers.stillManager = new StillRecipeManager();

		setupFuelManager();
	}

	@Override
	public void disabledSetupAPI() {
		RecipeManagers.carpenterManager = new DummyManagers.DummyCarpenterManager();
		RecipeManagers.centrifugeManager = new DummyManagers.DummyCentrifugeManager();
		RecipeManagers.fabricatorManager = new DummyManagers.DummyFabricatorManager();
		RecipeManagers.fabricatorSmeltingManager = new DummyManagers.DummyFabricatorSmeltingManager();
		RecipeManagers.fermenterManager = new DummyManagers.DummyFermenterManager();
		RecipeManagers.moistenerManager = new DummyManagers.DummyMoistenerManager();
		RecipeManagers.squeezerManager = new DummyManagers.DummySqueezerManager();
		RecipeManagers.stillManager = new DummyManagers.DummyStillManager();

		setupFuelManager();
	}

	private static void setupFuelManager() {
		FuelManager.fermenterFuel = new ItemStackMap<>();
		FuelManager.moistenerResource = new ItemStackMap<>();
		FuelManager.rainSubstrate = new ItemStackMap<>();
		FuelManager.bronzeEngineFuel = new FluidMap<>();
		FuelManager.copperEngineFuel = new ItemStackMap<>();
		FuelManager.generatorFuel = new FluidMap<>();
	}

	@Override
	public void registerItemsAndBlocks() {
		blocks = new BlockRegistryFactory();
	}

	@Override
	public IPacketRegistry getPacketRegistry() {
		return new PacketRegistryFactory();
	}

	@Override
	public void preInit() {
		super.preInit();

		ItemRegistryCore coreItems = PluginCore.getItems();

		// Set fuels and resources for the fermenter
		ItemStack fertilizerCompound = coreItems.fertilizerCompound.getItemStack();
		FuelManager.fermenterFuel.put(fertilizerCompound, new FermenterFuel(fertilizerCompound,
				ForestryAPI.activeMode.getIntegerSetting("fermenter.value.fertilizer"), ForestryAPI.activeMode.getIntegerSetting("fermenter.cycles.fertilizer")));

		int cyclesCompost = ForestryAPI.activeMode.getIntegerSetting("fermenter.cycles.compost");
		int valueCompost = ForestryAPI.activeMode.getIntegerSetting("fermenter.value.compost");
		ItemStack fertilizerBio = coreItems.fertilizerBio.getItemStack();
		ItemStack mulch = coreItems.mulch.getItemStack();
		FuelManager.fermenterFuel.put(fertilizerBio, new FermenterFuel(fertilizerBio, valueCompost, cyclesCompost));
		FuelManager.fermenterFuel.put(mulch, new FermenterFuel(mulch, valueCompost, cyclesCompost));

		// Add moistener resources
		ItemStack wheat = new ItemStack(Items.field_151015_O);
		ItemStack mouldyWheat = coreItems.mouldyWheat.getItemStack();
		ItemStack decayingWheat = coreItems.decayingWheat.getItemStack();
		FuelManager.moistenerResource.put(wheat, new MoistenerFuel(wheat, mouldyWheat, 0, 300));
		FuelManager.moistenerResource.put(mouldyWheat, new MoistenerFuel(mouldyWheat, decayingWheat, 1, 600));
		FuelManager.moistenerResource.put(decayingWheat, new MoistenerFuel(decayingWheat, mulch, 2, 900));

		// Set fuels for our own engines
		ItemStack peat = coreItems.peat.getItemStack();
		FuelManager.copperEngineFuel.put(peat, new EngineCopperFuel(peat, Constants.ENGINE_COPPER_FUEL_VALUE_PEAT, Constants.ENGINE_COPPER_CYCLE_DURATION_PEAT));

		ItemStack bituminousPeat = coreItems.bituminousPeat.getItemStack();
		FuelManager.copperEngineFuel.put(bituminousPeat, new EngineCopperFuel(bituminousPeat, Constants.ENGINE_COPPER_FUEL_VALUE_BITUMINOUS_PEAT, Constants.ENGINE_COPPER_CYCLE_DURATION_BITUMINOUS_PEAT));

		Fluid biomass = Fluids.BIOMASS.getFluid();
		if (biomass != null) {
			FuelManager.bronzeEngineFuel.put(biomass, new EngineBronzeFuel(biomass,
					Constants.ENGINE_FUEL_VALUE_BIOMASS, (int) (Constants.ENGINE_CYCLE_DURATION_BIOMASS * ForestryAPI.activeMode.getFloatSetting("fuel.biomass.biogas")), 1));
		}

		FuelManager.bronzeEngineFuel.put(FluidRegistry.WATER, new EngineBronzeFuel(FluidRegistry.WATER,
				Constants.ENGINE_FUEL_VALUE_WATER, Constants.ENGINE_CYCLE_DURATION_WATER, 3));

		Fluid milk = Fluids.MILK.getFluid();
		if (milk != null) {
			FuelManager.bronzeEngineFuel.put(milk, new EngineBronzeFuel(milk,
					Constants.ENGINE_FUEL_VALUE_MILK, Constants.ENGINE_CYCLE_DURATION_MILK, 3));
		}

		Fluid seedOil = Fluids.SEED_OIL.getFluid();
		if (seedOil != null) {
			FuelManager.bronzeEngineFuel.put(seedOil, new EngineBronzeFuel(seedOil,
					Constants.ENGINE_FUEL_VALUE_SEED_OIL, Constants.ENGINE_CYCLE_DURATION_SEED_OIL, 1));
		}

		Fluid honey = Fluids.FOR_HONEY.getFluid();
		if (honey != null) {
			FuelManager.bronzeEngineFuel.put(honey, new EngineBronzeFuel(honey,
					Constants.ENGINE_FUEL_VALUE_HONEY, Constants.ENGINE_CYCLE_DURATION_HONEY, 1));
		}

		Fluid juice = Fluids.JUICE.getFluid();
		if (juice != null) {
			FuelManager.bronzeEngineFuel.put(juice, new EngineBronzeFuel(juice,
					Constants.ENGINE_FUEL_VALUE_JUICE, Constants.ENGINE_CYCLE_DURATION_JUICE, 1));
		}

		// Set rain substrates
		ItemStack iodineCharge = coreItems.iodineCharge.getItemStack();
		ItemStack dissipationCharge = coreItems.craftingMaterial.getDissipationCharge();
		FuelManager.rainSubstrate.put(iodineCharge, new RainSubstrate(iodineCharge, Constants.RAINMAKER_RAIN_DURATION_IODINE, 0.01f));
		FuelManager.rainSubstrate.put(dissipationCharge, new RainSubstrate(dissipationCharge, 0.075f));

		ICircuitLayout layoutMachineUpgrade = new CircuitLayout("machine.upgrade", CircuitSocketType.MACHINE);
		ChipsetManager.circuitRegistry.registerLayout(layoutMachineUpgrade);

	}

	@Override
	public void addLootPoolNames(Set<String> lootPoolNames) {
		super.addLootPoolNames(lootPoolNames);
		lootPoolNames.add("forestry_factory_items");
	}

	@Override
	public void registerTriggers() {
//		FactoryTriggers.initialize();
	}

	@Override
	public void doInit() {
		super.doInit();
		BlockRegistryFactory blocks = getBlocks();

		blocks.bottler.init();
		blocks.carpenter.init();
		blocks.centrifuge.init();
		blocks.fermenter.init();
		blocks.moistener.init();
		blocks.squeezer.init();
		blocks.still.init();
		blocks.rainmaker.init();

		blocks.fabricator.init();
		blocks.raintank.init();
		blocks.worktable.init();

		Circuits.machineSpeedUpgrade1 = new CircuitSpeedUpgrade("machine.speed.boost.1", 0.125f, 0.05f);
		Circuits.machineSpeedUpgrade2 = new CircuitSpeedUpgrade("machine.speed.boost.2", 0.250f, 0.10f);
		Circuits.machineEfficiencyUpgrade1 = new CircuitSpeedUpgrade("machine.efficiency.1", 0, -0.10f);
	}

	@Override
	public void registerRecipes() {

		// / FABRICATOR
		ItemRegistryCore coreItems = PluginCore.getItems();
		BlockRegistryCore coreBlocks = PluginCore.getBlocks();
		ItemRegistryFluids fluidItems = PluginFluids.getItems();
		BlockRegistryFactory blocks = getBlocks();

		ItemElectronTube electronTube = coreItems.tubes;

		RecipeManagers.fabricatorManager.addRecipe(ItemStack.field_190927_a, Fluids.GLASS.getFluid(500), electronTube.get(EnumElectronTube.COPPER, 4), new Object[]{
				" X ",
				"#X#",
				"XXX",
				'#', "dustRedstone",
				'X', "ingotCopper"});
		RecipeManagers.fabricatorManager.addRecipe(ItemStack.field_190927_a, Fluids.GLASS.getFluid(500), electronTube.get(EnumElectronTube.TIN, 4), new Object[]{
				" X ",
				"#X#",
				"XXX",
				'#', "dustRedstone",
				'X', "ingotTin"});
		RecipeManagers.fabricatorManager.addRecipe(ItemStack.field_190927_a, Fluids.GLASS.getFluid(500), electronTube.get(EnumElectronTube.BRONZE, 4), new Object[]{
				" X ",
				"#X#",
				"XXX",
				'#', "dustRedstone",
				'X', "ingotBronze"});
		RecipeManagers.fabricatorManager.addRecipe(ItemStack.field_190927_a, Fluids.GLASS.getFluid(500), electronTube.get(EnumElectronTube.IRON, 4), new Object[]{
				" X ",
				"#X#",
				"XXX",
				'#', "dustRedstone",
				'X', "ingotIron"});
		RecipeManagers.fabricatorManager.addRecipe(ItemStack.field_190927_a, Fluids.GLASS.getFluid(500), electronTube.get(EnumElectronTube.GOLD, 4), new Object[]{
				" X ",
				"#X#",
				"XXX",
				'#', "dustRedstone",
				'X', "ingotGold"});
		RecipeManagers.fabricatorManager.addRecipe(ItemStack.field_190927_a, Fluids.GLASS.getFluid(500), electronTube.get(EnumElectronTube.DIAMOND, 4), new Object[]{
				" X ",
				"#X#",
				"XXX",
				'#', "dustRedstone",
				'X', "gemDiamond"});
		RecipeManagers.fabricatorManager.addRecipe(ItemStack.field_190927_a, Fluids.GLASS.getFluid(500), electronTube.get(EnumElectronTube.OBSIDIAN, 4), new Object[]{
				" X ",
				"#X#",
				"XXX",
				'#', "dustRedstone",
				'X', Blocks.field_150343_Z});
		RecipeManagers.fabricatorManager.addRecipe(ItemStack.field_190927_a, Fluids.GLASS.getFluid(500), electronTube.get(EnumElectronTube.BLAZE, 4), new Object[]{
				" X ",
				"#X#",
				"XXX",
				'#', "dustRedstone",
				'X', Items.field_151065_br});
		RecipeManagers.fabricatorManager.addRecipe(ItemStack.field_190927_a, Fluids.GLASS.getFluid(500), electronTube.get(EnumElectronTube.EMERALD, 4), new Object[]{
				" X ",
				"#X#",
				"XXX",
				'#', "dustRedstone",
				'X', "gemEmerald"});
		RecipeManagers.fabricatorManager.addRecipe(ItemStack.field_190927_a, Fluids.GLASS.getFluid(500), electronTube.get(EnumElectronTube.APATITE, 4), new Object[]{
				" X ",
				"#X#",
				"XXX",
				'#', "dustRedstone",
				'X', "gemApatite"});
		RecipeManagers.fabricatorManager.addRecipe(ItemStack.field_190927_a, Fluids.GLASS.getFluid(500), electronTube.get(EnumElectronTube.LAPIS, 4), new Object[]{
				" X ",
				"#X#",
				"XXX",
				'#', "dustRedstone",
				'X', new ItemStack(Items.field_151100_aR, 1, 4)});
		RecipeManagers.fabricatorManager.addRecipe(ItemStack.field_190927_a, Fluids.GLASS.getFluid(500), electronTube.get(EnumElectronTube.ENDER, 4), new Object[]{
				" X ",
				"#X#",
				"XXX",
				'#', new ItemStack(Items.field_151061_bv, 1, 0),
				'X', new ItemStack(Blocks.field_150377_bs, 1, 0)});

		String[] dyes = {"dyeBlack", "dyeRed", "dyeGreen", "dyeBrown", "dyeBlue", "dyePurple", "dyeCyan", "dyeLightGray", "dyeGray", "dyePink", "dyeLime",
				"dyeYellow", "dyeLightBlue", "dyeMagenta", "dyeOrange", "dyeWhite"};

		if (ForestryAPI.enabledPlugins.contains(ForestryPluginUids.APICULTURE)) {
			ItemRegistryApiculture beeItems = PluginApiculture.getItems();

			FluidStack liquidGlass = Fluids.GLASS.getFluid(Fluid.BUCKET_VOLUME);
			FluidStack liquidGlassX4 = Fluids.GLASS.getFluid(Fluid.BUCKET_VOLUME * 4);
			for (int i = 0; i < 16; i++) {
				RecipeManagers.fabricatorManager.addRecipe(beeItems.waxCast.getWildcard(), liquidGlass, new ItemStack(Blocks.field_150399_cn, 4, 15 - i), new Object[]{
						"#", "X",
						'#', dyes[i],
						'X', beeItems.propolis.getWildcard()});
			}
			RecipeManagers.fabricatorManager.addRecipe(beeItems.waxCast.getWildcard(), liquidGlassX4, new ItemStack(Blocks.field_150359_w, 1, 0), new Object[]{
					"#", "X",
					'X', beeItems.propolis.getWildcard()});
		}

		// / SQUEEZER
		int appleMulchAmount = ForestryAPI.activeMode.getIntegerSetting("squeezer.mulch.apple");
		int appleJuiceAmount = ForestryAPI.activeMode.getIntegerSetting("squeezer.liquid.apple");
		RecipeManagers.squeezerManager.addRecipe(10, new ItemStack(Items.field_151034_e), Fluids.JUICE.getFluid(appleJuiceAmount),
				coreItems.mulch.getItemStack(), appleMulchAmount);
		RecipeManagers.squeezerManager.addRecipe(10, new ItemStack(Items.field_151172_bF), Fluids.JUICE.getFluid(appleJuiceAmount),
				coreItems.mulch.getItemStack(), appleMulchAmount);

		int seedOilAmount = ForestryAPI.activeMode.getIntegerSetting("squeezer.liquid.seed");
		FluidStack seedOil = Fluids.SEED_OIL.getFluid(seedOilAmount);
		RecipeManagers.squeezerManager.addRecipe(10, new ItemStack(Items.field_151014_N), seedOil);
		RecipeManagers.squeezerManager.addRecipe(10, new ItemStack(Items.field_151080_bb), seedOil);
		RecipeManagers.squeezerManager.addRecipe(10, new ItemStack(Items.field_151081_bc), seedOil);
		RecipeManagers.squeezerManager.addRecipe(10, new ItemStack(Items.field_185163_cU), seedOil);

		RecipeManagers.squeezerManager.addRecipe(10, new ItemStack(Blocks.field_150434_aF), new FluidStack(FluidRegistry.WATER, 500));

		NonNullList<ItemStack> lavaRecipeResources = NonNullList.func_191196_a();
		lavaRecipeResources.add(coreItems.phosphor.getItemStack(2));
		lavaRecipeResources.add(new ItemStack(Blocks.field_150347_e));
		RecipeManagers.squeezerManager.addRecipe(10, lavaRecipeResources, new FluidStack(FluidRegistry.LAVA, 1600));

		NonNullList<ItemStack> iceRecipeResources = NonNullList.func_191196_a();
		iceRecipeResources.add(new ItemStack(Items.field_151126_ay));
		iceRecipeResources.add(coreItems.craftingMaterial.getIceShard(4));
		RecipeManagers.squeezerManager.addRecipe(10, iceRecipeResources, Fluids.ICE.getFluid(4000));

		// STILL
		RecipeManagers.stillManager.addRecipe(Constants.STILL_DESTILLATION_DURATION, Fluids.BIOMASS.getFluid(Constants.STILL_DESTILLATION_INPUT),
				Fluids.BIO_ETHANOL.getFluid(Constants.STILL_DESTILLATION_OUTPUT));

		// MOISTENER
		RecipeManagers.moistenerManager.addRecipe(new ItemStack(Items.field_151014_N), new ItemStack(Blocks.field_150391_bh), 5000);
		RecipeManagers.moistenerManager.addRecipe(new ItemStack(Blocks.field_150347_e), new ItemStack(Blocks.field_150341_Y), 20000);
		RecipeManagers.moistenerManager.addRecipe(new ItemStack(Blocks.field_150417_aV), new ItemStack(Blocks.field_150417_aV, 1, 1), 20000);

		// FERMENTER
		for (int i = 0; i < 6; i++) {
			RecipeUtil.addFermenterRecipes(new ItemStack(Blocks.field_150345_g, 1, i), ForestryAPI.activeMode.getIntegerSetting("fermenter.yield.sapling"), Fluids.BIOMASS);
		}

		RecipeUtil.addFermenterRecipes(new ItemStack(Blocks.field_150434_aF), ForestryAPI.activeMode.getIntegerSetting("fermenter.yield.cactus"), Fluids.BIOMASS);
		RecipeUtil.addFermenterRecipes(new ItemStack(Items.field_151015_O), ForestryAPI.activeMode.getIntegerSetting("fermenter.yield.wheat"), Fluids.BIOMASS);
		RecipeUtil.addFermenterRecipes(new ItemStack(Items.field_151174_bG), 2 * ForestryAPI.activeMode.getIntegerSetting("fermenter.yield.wheat"), Fluids.BIOMASS);
		RecipeUtil.addFermenterRecipes(new ItemStack(Items.field_151120_aE), ForestryAPI.activeMode.getIntegerSetting("fermenter.yield.cane"), Fluids.BIOMASS);
		RecipeUtil.addFermenterRecipes(new ItemStack(Blocks.field_150338_P), ForestryAPI.activeMode.getIntegerSetting("fermenter.yield.mushroom"), Fluids.BIOMASS);
		RecipeUtil.addFermenterRecipes(new ItemStack(Blocks.field_150337_Q), ForestryAPI.activeMode.getIntegerSetting("fermenter.yield.mushroom"), Fluids.BIOMASS);

		// FABRICATOR

		RecipeManagers.fabricatorSmeltingManager.addSmelting(new ItemStack(Blocks.field_150359_w), Fluids.GLASS.getFluid(1000), 1000);
		RecipeManagers.fabricatorSmeltingManager.addSmelting(new ItemStack(Blocks.field_150410_aZ), Fluids.GLASS.getFluid(375), 1000);
		RecipeManagers.fabricatorSmeltingManager.addSmelting(new ItemStack(Blocks.field_150354_m), Fluids.GLASS.getFluid(1000), 3000);
		RecipeManagers.fabricatorSmeltingManager.addSmelting(new ItemStack(Blocks.field_150354_m, 1, 1), Fluids.GLASS.getFluid(1000), 3000);
		RecipeManagers.fabricatorSmeltingManager.addSmelting(new ItemStack(Blocks.field_150322_A), Fluids.GLASS.getFluid(4000), 4800);
		RecipeManagers.fabricatorSmeltingManager.addSmelting(new ItemStack(Blocks.field_150322_A, 1, 1), Fluids.GLASS.getFluid(4000), 4800);
		RecipeManagers.fabricatorSmeltingManager.addSmelting(new ItemStack(Blocks.field_150322_A, 1, 2), Fluids.GLASS.getFluid(4000), 4800);

		// / CARPENTER
		RecipeManagers.carpenterManager.addRecipe(50, Fluids.SEED_OIL.getFluid(250), ItemStack.field_190927_a, coreItems.impregnatedCasing.getItemStack(),
				"###",
				"# #",
				"###",
				'#', "logWood");
		RecipeManagers.carpenterManager.addRecipe(50, Fluids.SEED_OIL.getFluid(500), ItemStack.field_190927_a,
				new ItemStack(coreBlocks.escritoire),
				"#  ",
				"###",
				"# #",
				'#', "plankWood");

		// RESOURCES
		RecipeManagers.carpenterManager.addRecipe(10, Fluids.SEED_OIL.getFluid(100), ItemStack.field_190927_a,
				coreItems.stickImpregnated.getItemStack(2),
				"#",
				"#",
				'#', "logWood");
		RecipeManagers.carpenterManager.addRecipe(5, new FluidStack(FluidRegistry.WATER, 250), ItemStack.field_190927_a,
				coreItems.woodPulp.getItemStack(4),
				"#",
				'#', "logWood");
		RecipeManagers.carpenterManager.addRecipe(5, new FluidStack(FluidRegistry.WATER, 250), ItemStack.field_190927_a,
				new ItemStack(Items.field_151121_aF, 1),
				"#",
				"#",
				'#', "pulpWood");
		RecipeManagers.carpenterManager.addRecipe(5, new FluidStack(FluidRegistry.WATER, 1000), ItemStack.field_190927_a,
				new ItemStack(coreBlocks.humus, 9),
				"###",
				"#X#",
				"###",
				'#', Blocks.field_150346_d,
				'X', coreItems.mulch);
		RecipeManagers.carpenterManager.addRecipe(5, new FluidStack(FluidRegistry.WATER, 1000), ItemStack.field_190927_a,
				coreBlocks.bogEarth.get(BlockBogEarth.SoilType.BOG_EARTH, 8),
				"#X#",
				"XYX", "#X#",
				'#', Blocks.field_150346_d,
				'X', "sand",
				'Y', coreItems.mulch);
		RecipeManagers.carpenterManager.addRecipe(75, new FluidStack(FluidRegistry.WATER, 5000), ItemStack.field_190927_a, coreItems.hardenedCasing.getItemStack(),
				"# #",
				" Y ",
				"# #",
				'#', "gemDiamond",
				'Y', coreItems.sturdyCasing);

		// / CHIPSETS
		ItemStack basicCircuitboard = ItemCircuitBoard.createCircuitboard(EnumCircuitBoardType.BASIC, null, new ICircuit[]{});
		ItemStack enhancedCircuitboard = ItemCircuitBoard.createCircuitboard(EnumCircuitBoardType.ENHANCED, null, new ICircuit[]{});
		ItemStack refinedCircuitboard = ItemCircuitBoard.createCircuitboard(EnumCircuitBoardType.REFINED, null, new ICircuit[]{});
		ItemStack intricateCircuitboard = ItemCircuitBoard.createCircuitboard(EnumCircuitBoardType.INTRICATE, null, new ICircuit[]{});

		RecipeManagers.carpenterManager.addRecipe(20, new FluidStack(FluidRegistry.WATER, 1000), ItemStack.field_190927_a, basicCircuitboard,
				"R R", "R#R", "R R", '#', "ingotTin", 'R', "dustRedstone");

		RecipeManagers.carpenterManager.addRecipe(40, new FluidStack(FluidRegistry.WATER, 1000), ItemStack.field_190927_a, enhancedCircuitboard,
				"R#R", "R#R", "R#R", '#', "ingotBronze", 'R', "dustRedstone");

		RecipeManagers.carpenterManager.addRecipe(80, new FluidStack(FluidRegistry.WATER, 1000), ItemStack.field_190927_a, refinedCircuitboard,
				"R#R", "R#R", "R#R", '#', "ingotIron", 'R', "dustRedstone");

		RecipeManagers.carpenterManager.addRecipe(80, new FluidStack(FluidRegistry.WATER, 1000), ItemStack.field_190927_a, intricateCircuitboard,
				"R#R", "R#R", "R#R", '#', "ingotGold", 'R', "dustRedstone");
		RecipeManagers.carpenterManager.addRecipe(40, new FluidStack(FluidRegistry.WATER, 1000), ItemStack.field_190927_a, coreItems.solderingIron.getItemStack(),
				" # ", "# #", "  B", '#', "ingotIron", 'B', "ingotBronze");

		// RAIN SUBSTRATES


		if (ForestryAPI.enabledPlugins.contains(ForestryPluginUids.APICULTURE)) {
			ItemRegistryApiculture beeItems = PluginApiculture.getItems();
			RecipeManagers.carpenterManager.addRecipe(5, new FluidStack(FluidRegistry.WATER, 1000), ItemStack.field_190927_a, coreItems.iodineCharge.getItemStack(),
					"Z#Z",
					"#Y#",
					"X#X",
					'#', beeItems.pollenCluster.getWildcard(),
					'X', Items.field_151016_H,
					'Y', fluidItems.canEmpty,
					'Z', beeItems.honeyDrop);
			RecipeManagers.carpenterManager.addRecipe(5, new FluidStack(FluidRegistry.WATER, 1000), ItemStack.field_190927_a, coreItems.craftingMaterial.getDissipationCharge(),
					"Z#Z",
					"#Y#",
					"X#X",
					'#', beeItems.royalJelly,
					'X', Items.field_151016_H,
					'Y', fluidItems.canEmpty,
					'Z', beeItems.honeydew);
		}

		// Ender pearl
		RecipeManagers.carpenterManager.addRecipe(100, ItemStack.field_190927_a, new ItemStack(Items.field_151079_bi, 1), " # ", "###", " # ", '#',
				coreItems.craftingMaterial.getPulsatingMesh());

		// Woven Silk
		RecipeManagers.carpenterManager.addRecipe(10, new FluidStack(FluidRegistry.WATER, 500), ItemStack.field_190927_a, coreItems.craftingMaterial.getWovenSilk(),
				"###",
				"###",
				"###",
				'#', coreItems.craftingMaterial.getSilkWisp());

		// Boxes
		RecipeManagers.carpenterManager.addRecipe(5, new FluidStack(FluidRegistry.WATER, 1000), ItemStack.field_190927_a, coreItems.carton.getItemStack(2),
				" # ", "# #", " # ", '#', "pulpWood");

		// Assembly Kits
		RecipeManagers.carpenterManager.addRecipe(20, null, coreItems.carton.getItemStack(), coreItems.kitPickaxe.getItemStack(), new Object[]{
				"###",
				" X ",
				" X ",
				'#', "ingotBronze",
				'X', "stickWood"});

		RecipeManagers.carpenterManager.addRecipe(20, null, coreItems.carton.getItemStack(), coreItems.kitShovel.getItemStack(),
				new Object[]{" # ", " X ", " X ", '#', "ingotBronze", 'X', "stickWood"});

		// Reclamation
		ItemStack ingotBronze = coreItems.ingotBronze.func_77946_l();
		ingotBronze.func_190920_e(2);
		RecipeManagers.carpenterManager.addRecipe(ItemStack.field_190927_a, ingotBronze, "#", '#', coreItems.brokenBronzePickaxe);

		ingotBronze = ingotBronze.func_77946_l();
		ingotBronze.func_190920_e(1);
		RecipeManagers.carpenterManager.addRecipe(ItemStack.field_190927_a, ingotBronze, "#", '#', coreItems.brokenBronzeShovel);

		// Crating and uncrating
		if (ForestryAPI.enabledPlugins.contains(ForestryPluginUids.STORAGE)) {
			PluginStorage.createCrateRecipes();
		}
		ICircuitLayout layout = ChipsetManager.circuitRegistry.getLayout("forestry.machine.upgrade");

		// / Solder Manager
		ChipsetManager.solderManager.addRecipe(layout, coreItems.tubes.get(EnumElectronTube.EMERALD, 1), Circuits.machineSpeedUpgrade1);
		ChipsetManager.solderManager.addRecipe(layout, coreItems.tubes.get(EnumElectronTube.BLAZE, 1), Circuits.machineSpeedUpgrade2);
		ChipsetManager.solderManager.addRecipe(layout, coreItems.tubes.get(EnumElectronTube.GOLD, 1), Circuits.machineEfficiencyUpgrade1);

		RecipeUtil.addRecipe(blocks.bottler,
				"X#X",
				"#Y#",
				"X#X",
				'#', "blockGlass",
				'X', fluidItems.canEmpty,
				'Y', coreItems.sturdyCasing);

		RecipeUtil.addRecipe(blocks.carpenter,
				"X#X",
				"XYX",
				"X#X",
				'#', "blockGlass",
				'X', "ingotBronze",
				'Y', coreItems.sturdyCasing);

		RecipeUtil.addRecipe(blocks.centrifuge,
				"X#X",
				"XYX",
				"X#X",
				'#', "blockGlass",
				'X', "ingotCopper",
				'Y', coreItems.sturdyCasing);

		RecipeUtil.addRecipe(blocks.fermenter,
				"X#X",
				"#Y#",
				"X#X",
				'#', "blockGlass",
				'X', "gearBronze",
				'Y', coreItems.sturdyCasing);

		RecipeUtil.addRecipe(blocks.moistener,
				"X#X",
				"#Y#",
				"X#X",
				'#', "blockGlass",
				'X', "gearCopper",
				'Y', coreItems.sturdyCasing);

		RecipeUtil.addRecipe(blocks.squeezer,
				"X#X",
				"XYX",
				"X#X",
				'#', "blockGlass",
				'X', "ingotTin",
				'Y', coreItems.sturdyCasing);

		RecipeUtil.addRecipe(blocks.still,
				"X#X",
				"#Y#",
				"X#X",
				'#', "blockGlass",
				'X', "dustRedstone",
				'Y', coreItems.sturdyCasing);

		RecipeUtil.addRecipe(blocks.rainmaker,
				"X#X",
				"#Y#",
				"X#X",
				'#', "blockGlass",
				'X', "gearTin",
				'Y', coreItems.hardenedCasing);

		RecipeUtil.addRecipe(blocks.fabricator,
				"X#X",
				"#Y#",
				"XZX",
				'#', "blockGlass",
				'X', "ingotGold",
				'Y', coreItems.sturdyCasing,
				'Z', "chestWood");

		RecipeUtil.addRecipe(blocks.raintank,
				"X#X",
				"XYX",
				"X#X",
				'#', "blockGlass",
				'X', "ingotIron",
				'Y', coreItems.sturdyCasing);

		RecipeUtil.addRecipe(blocks.worktable,
				"B",
				"W",
				"C",
				'B', Items.field_151122_aG,
				'W', "craftingTableWood",
				'C', "chestWood");
	}
}
