/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.lepidopterology.genetics;

import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map.Entry;

import com.google.common.base.Preconditions;
import com.mojang.authlib.GameProfile;
import forestry.api.genetics.AlleleManager;
import forestry.api.genetics.IAllele;
import forestry.api.genetics.IAlyzerPlugin;
import forestry.api.genetics.IChromosomeType;
import forestry.api.genetics.IIndividual;
import forestry.api.genetics.IMutation;
import forestry.api.genetics.ISpeciesType;
import forestry.api.lepidopterology.ButterflyManager;
import forestry.api.lepidopterology.EnumButterflyChromosome;
import forestry.api.lepidopterology.EnumFlutterType;
import forestry.api.lepidopterology.IAlleleButterflySpecies;
import forestry.api.lepidopterology.IButterfly;
import forestry.api.lepidopterology.IButterflyGenome;
import forestry.api.lepidopterology.IButterflyMutation;
import forestry.api.lepidopterology.IButterflyNursery;
import forestry.api.lepidopterology.IButterflyRoot;
import forestry.api.lepidopterology.ILepidopteristTracker;
import forestry.core.genetics.SpeciesRoot;
import forestry.core.utils.BlockUtil;
import forestry.core.utils.EntityUtil;
import forestry.lepidopterology.PluginLepidopterology;
import forestry.lepidopterology.blocks.BlockRegistryLepidopterology;
import forestry.lepidopterology.entities.EntityButterfly;
import forestry.lepidopterology.items.ItemRegistryLepidopterology;
import forestry.lepidopterology.tiles.TileCocoon;
import net.minecraft.block.Block;
import net.minecraft.block.state.IBlockState;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;

public class ButterflyRoot extends SpeciesRoot implements IButterflyRoot {

	private static int butterflySpeciesCount = -1;
	public static final String UID = "rootButterflies";
	private static final List<IButterfly> butterflyTemplates = new ArrayList<>();


	@Override
	public String getUID() {
		return UID;
	}


	@Override
	public Class<? extends IIndividual> getMemberClass() {
		return IButterfly.class;
	}

	@Override
	public int getSpeciesCount() {
		if (butterflySpeciesCount < 0) {
			butterflySpeciesCount = 0;
			for (Entry<String, IAllele> entry : AlleleManager.alleleRegistry.getRegisteredAlleles().entrySet()) {
				if (entry.getValue() instanceof IAlleleButterflySpecies) {
					if (((IAlleleButterflySpecies) entry.getValue()).isCounted()) {
						butterflySpeciesCount++;
					}
				}
			}
		}

		return butterflySpeciesCount;
	}

	@Override
	public boolean isMember(ItemStack stack) {
		return getType(stack) != null;
	}

	@Nullable
	@Override
	public EnumFlutterType getType(ItemStack stack) {
		if (stack.func_190926_b()) {
			return null;
		}

		ItemRegistryLepidopterology butterflyItems = PluginLepidopterology.getItems();
		Preconditions.checkState(butterflyItems != null);

		Item item = stack.func_77973_b();
		if (butterflyItems.butterflyGE == item) {
			return EnumFlutterType.BUTTERFLY;
		} else if (butterflyItems.serumGE == item) {
			return EnumFlutterType.SERUM;
		} else if (butterflyItems.caterpillarGE == item) {
			return EnumFlutterType.CATERPILLAR;
		} else if (butterflyItems.cocoonGE == item) {
			return EnumFlutterType.COCOON;
		} else {
			return null;
		}
	}

	@Override
	public EnumFlutterType getIconType() {
		return EnumFlutterType.BUTTERFLY;
	}

	@Override
	public boolean isMember(ItemStack stack, ISpeciesType type) {
		return getType(stack) == type;
	}

	@Override
	public boolean isMember(IIndividual individual) {
		return individual instanceof IButterfly;
	}

	@Override
	public IButterfly getMember(ItemStack stack) {
		if (!isMember(stack) || stack.func_77978_p() == null) {
			return null;
		}

		return new Butterfly(stack.func_77978_p());
	}

	@Override
	public IButterfly getMember(NBTTagCompound compound) {
		return new Butterfly(compound);
	}

	@Override
	public ItemStack getMemberStack(IIndividual butterfly, ISpeciesType type) {
		Preconditions.checkArgument(type instanceof EnumFlutterType);
		ItemRegistryLepidopterology items = PluginLepidopterology.getItems();
		Preconditions.checkState(items != null);

		Item butterflyItem;
		switch ((EnumFlutterType) type) {
			case SERUM:
				butterflyItem = items.serumGE;
				break;
			case CATERPILLAR:
				butterflyItem = items.caterpillarGE;
				break;
			case COCOON:
				butterflyItem = items.cocoonGE;
				break;
			case BUTTERFLY:
			default:
				butterflyItem = items.butterflyGE;
				break;
		}

		NBTTagCompound nbttagcompound = new NBTTagCompound();
		butterfly.writeToNBT(nbttagcompound);
		ItemStack stack = new ItemStack(butterflyItem);
		stack.func_77982_d(nbttagcompound);
		return stack;

	}

	@Override
	public EntityButterfly spawnButterflyInWorld(World world, IButterfly butterfly, double x, double y, double z) {
		return EntityUtil.spawnEntity(world, new EntityButterfly(world, butterfly, new BlockPos(x, y, z)), x, y, z);
	}

	@Override
	public boolean plantCocoon(World world, IButterflyNursery nursery, GameProfile owner, int age) {
		if (nursery.getCaterpillar() == null) {
			return false;
		}

		BlockRegistryLepidopterology blocks = PluginLepidopterology.getBlocks();

		BlockPos pos = getNextPos(world, nursery.getCoordinates());
		IBlockState state = blocks.cocoon.func_176223_P();
		boolean placed = world.func_175656_a(pos, state);
		if (!placed) {
			return false;
		}

		Block block = world.func_180495_p(pos).func_177230_c();
		if (blocks.cocoon != block) {
			return false;
		}

		TileEntity tile = world.func_175625_s(pos);
		if (!(tile instanceof TileCocoon)) {
			world.func_175698_g(pos);
			return false;
		}

		TileCocoon cocoon = (TileCocoon) tile;
		cocoon.setCaterpillar(nursery.getCaterpillar());
		cocoon.getOwnerHandler().setOwner(owner);
		cocoon.setNursery(nursery);
		cocoon.setAge(age);

		return true;
	}

	private BlockPos getNextPos(World world, BlockPos pos) {
		IBlockState blockState;
		do {
			pos = pos.func_177977_b();
			blockState = world.func_180495_p(pos);
		} while (!BlockUtil.canReplace(blockState, world, pos));

		return pos;
	}

	@Override
	public boolean isMated(ItemStack stack) {
		IButterfly butterfly = getMember(stack);
		return butterfly != null && butterfly.getMate() != null;
	}

	/* GENOME CONVERSIONS */
	@Override
	public IButterfly templateAsIndividual(IAllele[] template) {
		return new Butterfly(templateAsGenome(template));
	}

	@Override
	public IButterfly templateAsIndividual(IAllele[] templateActive, IAllele[] templateInactive) {
		return new Butterfly(templateAsGenome(templateActive, templateInactive));
	}

	@Override
	public IButterflyGenome templateAsGenome(IAllele[] template) {
		return new ButterflyGenome(templateAsChromosomes(template));
	}

	@Override
	public IButterflyGenome templateAsGenome(IAllele[] templateActive, IAllele[] templateInactive) {
		return new ButterflyGenome(templateAsChromosomes(templateActive, templateInactive));
	}

	/* TEMPLATES */

	@Override
	public List<IButterfly> getIndividualTemplates() {
		return butterflyTemplates;
	}


	@Override
	public IAllele[] getDefaultTemplate() {
		return MothDefinition.Brimstone.getTemplate();
	}

	@Override
	public void registerTemplate(String identifier, IAllele[] template) {
		butterflyTemplates.add(ButterflyManager.butterflyRoot.templateAsIndividual(template));
		speciesTemplates.put(identifier, template);
	}

	/* MUTATIONS */
	private static final List<IButterflyMutation> butterflyMutations = new ArrayList<>();

	@Override
	public void registerMutation(IMutation mutation) {
		if (AlleleManager.alleleRegistry.isBlacklisted(mutation.getTemplate()[0].getUID())) {
			return;
		}
		if (AlleleManager.alleleRegistry.isBlacklisted(mutation.getAllele0().getUID())) {
			return;
		}
		if (AlleleManager.alleleRegistry.isBlacklisted(mutation.getAllele1().getUID())) {
			return;
		}

		butterflyMutations.add((IButterflyMutation) mutation);
	}

	@Override
	public List<IButterflyMutation> getMutations(boolean shuffle) {
		if (shuffle) {
			Collections.shuffle(butterflyMutations);
		}
		return butterflyMutations;
	}

	/* BREEDING TRACKER */
	@Override
	public ILepidopteristTracker getBreedingTracker(World world, @Nullable GameProfile player) {
		String filename = "LepidopteristTracker." + (player == null ? "common" : player.getId());
		LepidopteristTracker tracker = (LepidopteristTracker) world.func_72943_a(LepidopteristTracker.class, filename);

		// Create a tracker if there is none yet.
		if (tracker == null) {
			tracker = new LepidopteristTracker(filename);
			world.func_72823_a(filename, tracker);
		}

		tracker.setUsername(player);
		tracker.setWorld(world);

		return tracker;
	}

	@Override
	public IChromosomeType[] getKaryotype() {
		return EnumButterflyChromosome.values();
	}

	@Override
	public IChromosomeType getSpeciesChromosomeType() {
		return EnumButterflyChromosome.SPECIES;
	}

	@Override
	public IAlyzerPlugin getAlyzerPlugin() {
		return FlutterlyzerPlugin.INSTANCE;
	}
}
