/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.arboriculture.blocks;

import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Random;

import com.mojang.authlib.GameProfile;
import forestry.api.arboriculture.EnumGermlingType;
import forestry.api.arboriculture.IToolGrafter;
import forestry.api.arboriculture.ITree;
import forestry.api.arboriculture.TreeManager;
import forestry.api.core.IItemModelRegister;
import forestry.api.core.IModelManager;
import forestry.api.core.IToolScoop;
import forestry.api.core.Tabs;
import forestry.api.lepidopterology.ButterflyManager;
import forestry.api.lepidopterology.EnumFlutterType;
import forestry.api.lepidopterology.IButterfly;
import forestry.arboriculture.LeafDecayHelper;
import forestry.arboriculture.PluginArboriculture;
import forestry.arboriculture.genetics.TreeDefinition;
import forestry.arboriculture.tiles.TileLeaves;
import forestry.core.blocks.IColoredBlock;
import forestry.core.blocks.properties.UnlistedBlockAccess;
import forestry.core.blocks.properties.UnlistedBlockPos;
import forestry.core.proxy.Proxies;
import forestry.core.tiles.TileUtil;
import forestry.core.utils.BlockUtil;
import forestry.core.utils.ItemStackUtil;
import net.minecraft.block.BlockLeaves;
import net.minecraft.block.BlockPlanks;
import net.minecraft.block.IGrowable;
import net.minecraft.block.ITileEntityProvider;
import net.minecraft.block.properties.IProperty;
import net.minecraft.block.state.BlockStateContainer;
import net.minecraft.block.state.IBlockState;
import net.minecraft.client.renderer.block.model.ModelResourceLocation;
import net.minecraft.creativetab.CreativeTabs;
import net.minecraft.enchantment.EnchantmentHelper;
import net.minecraft.entity.Entity;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.init.Enchantments;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.BlockRenderLayer;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumHand;
import net.minecraft.util.NonNullList;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.RayTraceResult;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;
import net.minecraftforge.client.model.ModelLoader;
import net.minecraftforge.common.property.ExtendedBlockState;
import net.minecraftforge.common.property.IExtendedBlockState;
import net.minecraftforge.common.property.IUnlistedProperty;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

public class BlockForestryLeaves extends BlockLeaves implements ITileEntityProvider, IGrowable, IItemModelRegister, IColoredBlock {

	public BlockForestryLeaves() {
		func_149647_a(Tabs.tabArboriculture);
		func_180632_j(this.field_176227_L.func_177621_b().func_177226_a(field_176236_b, false).func_177226_a(field_176237_a, true));
	}

	@Override
	public IBlockState getExtendedState(IBlockState state, IBlockAccess world, BlockPos pos) {
		return ((IExtendedBlockState) super.getExtendedState(state, world, pos)).withProperty(UnlistedBlockPos.POS, pos)
				.withProperty(UnlistedBlockAccess.BLOCKACCESS, world);
	}

	@Override
	protected BlockStateContainer func_180661_e() {
		return new ExtendedBlockState(this, new IProperty[]{field_176237_a, field_176236_b}, new IUnlistedProperty[]{UnlistedBlockPos.POS, UnlistedBlockAccess.BLOCKACCESS});
	}

	@Override
	public int func_176201_c(IBlockState state) {
		int i = 0;
		if (!state.func_177229_b(field_176237_a)) {
			i |= 4;
		}

		if (state.func_177229_b(field_176236_b)) {
			i |= 8;
		}

		return i;
	}

	@Override
	public IBlockState func_176203_a(int meta) {
		return this.func_176223_P().func_177226_a(field_176237_a, (meta & 4) == 0).func_177226_a(field_176236_b, (meta & 8) > 0);
	}

	/* TILE ENTITY */
	@Override
	public TileEntity func_149915_a(World world, int meta) {
		return new TileLeaves();
	}

	@Override
	public void func_149666_a(Item itemIn, CreativeTabs tab, NonNullList<ItemStack> list) {

	}

	/* DROP HANDLING */
	// Hack: 	When harvesting leaves we need to get the drops in onBlockHarvested,
	// 			because Mojang destroys the block and tile before calling getDrops.
	private final ThreadLocal<List<ItemStack>> drops = new ThreadLocal<>();

	@Override
	public void func_176208_a(World world, BlockPos pos, IBlockState state, EntityPlayer player) {
		TileLeaves leafTile = TileUtil.getTile(world, pos, TileLeaves.class);
		if (leafTile == null) {
			return;
		}

		int fortune = EnchantmentHelper.func_77506_a(Enchantments.field_185308_t, player.func_184607_cu());
		float saplingModifier = 1.0f;

		ItemStack held = player.field_71071_by.func_70448_g();
		if (held.func_77973_b() instanceof IToolGrafter) {
			saplingModifier = ((IToolGrafter) held.func_77973_b()).getSaplingModifier(held, world, player, pos);
			held.func_77972_a(1, player);
			if (held.func_190926_b()) {
				net.minecraftforge.event.ForgeEventFactory.onPlayerDestroyItem(player, held, EnumHand.MAIN_HAND);
			}
		}

		GameProfile playerProfile = player.func_146103_bH();
		List<ItemStack> leafDrops = getLeafDrop(world, playerProfile, pos, saplingModifier, fortune);
		drops.set(leafDrops);
	}

	@Override
	public List<ItemStack> getDrops(IBlockAccess world, BlockPos pos, IBlockState state, int fortune) {
		List<ItemStack> ret = drops.get();
		drops.remove();

		// leaves not harvested, get drops normally
		if (ret == null) {
			ret = getLeafDrop(world, null, pos, 1.0f, fortune);
		}

		return ret;
	}

	private static List<ItemStack> getLeafDrop(IBlockAccess world, @Nullable GameProfile playerProfile, BlockPos pos, float saplingModifier, int fortune) {
		List<ItemStack> prod = new ArrayList<>();

		TileLeaves tile = TileUtil.getTile(world, pos, TileLeaves.class);
		if (tile == null) {
			return prod;
		}

		ITree tree = tile.getTree();
		if (tree == null) {
			return prod;
		}

		// Add saplings
		List<ITree> saplings = tree.getSaplings((World) world, playerProfile, pos, saplingModifier);

		for (ITree sapling : saplings) {
			if (sapling != null) {
				prod.add(TreeManager.treeRoot.getMemberStack(sapling, EnumGermlingType.SAPLING));
			}
		}

		// Add fruits
		if (tile.hasFruit()) {
			prod.addAll(tree.produceStacks((World) world, pos, tile.getRipeningTime()));
		}

		return prod;
	}

	@Override
	public ItemStack getPickBlock(IBlockState state, RayTraceResult target, World world, BlockPos pos, EntityPlayer player) {
		TileLeaves leaves = TileUtil.getTile(world, pos, TileLeaves.class);
		if (leaves == null) {
			return ItemStack.field_190927_a;
		}

		ITree tree = leaves.getTree();
		if (tree == null) {
			return ItemStack.field_190927_a;
		}

		String speciesUid = tree.getGenome().getPrimary().getUID();
		return PluginArboriculture.getBlocks().getDecorativeLeaves(speciesUid);
	}

	@Override
	public List<ItemStack> onSheared(ItemStack item, IBlockAccess world, BlockPos pos, int fortune) {
		String speciesUid = TreeDefinition.Oak.getUID();

		TileLeaves leaves = TileUtil.getTile(world, pos, TileLeaves.class);
		if (leaves != null) {
			ITree tree = leaves.getTree();
			if (tree != null) {
				speciesUid = tree.getGenome().getPrimary().getUID();
			}
		}

		ItemStack decorativeLeaves = PluginArboriculture.getBlocks().getDecorativeLeaves(speciesUid);
		if (decorativeLeaves.func_190926_b()) {
			return Collections.emptyList();
		} else {
			return Collections.singletonList(decorativeLeaves);
		}
	}

	@Nullable
	@Override
	public AxisAlignedBB func_180646_a(IBlockState blockState, IBlockAccess worldIn, BlockPos pos) {
		TileLeaves tileLeaves = TileUtil.getTile(worldIn, pos, TileLeaves.class);
		if (tileLeaves != null && TreeDefinition.Willow.getUID().equals(tileLeaves.getSpeciesUID())) {
			return null;
		}
		return super.func_180646_a(blockState, worldIn, pos);
	}

	@Override
	public void func_180634_a(World worldIn, BlockPos pos, IBlockState state, Entity entityIn) {
		super.func_180634_a(worldIn, pos, state, entityIn);
		entityIn.field_70159_w *= 0.4D;
		entityIn.field_70179_y *= 0.4D;
	}

	@Override
	public void func_180650_b(World world, BlockPos pos, IBlockState state, Random rand) {
		LeafDecayHelper.leafDecay(this, world, pos);

		TileLeaves tileLeaves = TileUtil.getTile(world, pos, TileLeaves.class);

		// check leaves tile because they might have decayed
		if (tileLeaves != null && !tileLeaves.func_145837_r() && rand.nextFloat() <= 0.1) {
			tileLeaves.onBlockTick(world, pos, state, rand);
		}
	}

	/* RENDERING */
	@Override
	public boolean func_149662_c(IBlockState state) {
		return !Proxies.render.fancyGraphicsEnabled();
	}

	@SideOnly(Side.CLIENT)
	@Override
	public boolean func_176225_a(IBlockState blockState, IBlockAccess blockAccess, BlockPos pos, EnumFacing side) {
		return (Proxies.render.fancyGraphicsEnabled() || blockAccess.func_180495_p(pos.func_177972_a(side)).func_177230_c() != this) && BlockUtil.shouldSideBeRendered(blockState, blockAccess, pos, side);
	}

	@Override
	@SideOnly(Side.CLIENT)
	public BlockRenderLayer func_180664_k() {
		return BlockRenderLayer.CUTOUT_MIPPED; // fruit overlays require CUTOUT_MIPPED, even in Fast graphics
	}

	@Override
	public BlockPlanks.EnumType func_176233_b(int meta) {
		return BlockPlanks.EnumType.OAK;
	}

	/* MODELS */
	@Override
	@SideOnly(Side.CLIENT)
	public void registerModel(Item item, IModelManager manager) {
		ModelLoader.setCustomModelResourceLocation(item, 0, new ModelResourceLocation("forestry:leaves", "inventory"));
	}

	/* PROPERTIES */
	@Override
	public int getFlammability(IBlockAccess world, BlockPos pos, EnumFacing face) {
		return 60;
	}

	@Override
	public boolean isFlammable(IBlockAccess world, BlockPos pos, EnumFacing face) {
		return true;
	}

	@Override
	public int getFireSpreadSpeed(IBlockAccess world, BlockPos pos, EnumFacing face) {
		if (face == EnumFacing.DOWN) {
			return 20;
		} else if (face != EnumFacing.UP) {
			return 10;
		} else {
			return 5;
		}
	}

	@Override
	public boolean func_180639_a(World worldIn, BlockPos pos, IBlockState state, EntityPlayer playerIn, EnumHand hand, EnumFacing facing, float hitX, float hitY, float hitZ) {
		TileEntity tile = worldIn.func_175625_s(pos);
		IButterfly caterpillar = tile instanceof TileLeaves ? ((TileLeaves) tile).getCaterpillar() : null;
		ItemStack heldItem = playerIn.func_184586_b(hand);
		if (heldItem.func_77973_b() instanceof IToolScoop && caterpillar != null) {
			ItemStack butterfly = ButterflyManager.butterflyRoot.getMemberStack(caterpillar, EnumFlutterType.CATERPILLAR);
			ItemStackUtil.dropItemStackAsEntity(butterfly, worldIn, pos);
			((TileLeaves) tile).setCaterpillar(null);
			return true;
		}

		return super.func_180639_a(worldIn, pos, state, playerIn, hand, facing, hitX, hitY, hitZ);
	}

	/* IGrowable */

	@Override
	public boolean func_176473_a(World world, BlockPos pos, IBlockState state, boolean isClient) {
		TileLeaves leafTile = TileUtil.getTile(world, pos, TileLeaves.class);
		return leafTile != null && leafTile.hasFruit() && leafTile.getRipeness() < 1.0f;
	}

	@Override
	public boolean func_180670_a(World worldIn, Random rand, BlockPos pos, IBlockState state) {
		return true;
	}

	@Override
	public void func_176474_b(World world, Random rand, BlockPos pos, IBlockState state) {
		TileLeaves leafTile = TileUtil.getTile(world, pos, TileLeaves.class);
		if (leafTile != null) {
			leafTile.addRipeness(0.5f);
		}
	}

	@Override
	public int colorMultiplier(IBlockState state, @Nullable IBlockAccess worldIn, @Nullable BlockPos pos, int tintIndex) {
		if (worldIn != null && pos != null) {
			TileLeaves leaves = TileUtil.getTile(worldIn, pos, TileLeaves.class);
			if (leaves != null) {
				if (tintIndex == 0) {
					EntityPlayer thePlayer = Proxies.common.getPlayer();
					return leaves.getFoliageColour(thePlayer);
				} else {
					return leaves.getFruitColour();
				}
			}
		}
		return PluginArboriculture.proxy.getFoliageColorBasic();
	}
}
