/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.energy.blocks;

import javax.annotation.Nullable;
import java.util.EnumMap;
import java.util.List;

import com.google.common.collect.ImmutableList;
import forestry.core.blocks.BlockBase;
import forestry.energy.EnergyHelper;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.Entity;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.RayTraceResult;
import net.minecraft.util.math.Vec3d;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;

public class BlockEngine extends BlockBase<BlockTypeEngine> {
	private static final EnumMap<EnumFacing, List<AxisAlignedBB>> boundingBoxesForDirections = new EnumMap<>(EnumFacing.class);

	static {
		boundingBoxesForDirections.put(EnumFacing.DOWN, ImmutableList.of(
				new AxisAlignedBB(0.0, 0.5, 0.0, 1.0, 1.0, 1.0), new AxisAlignedBB(0.25, 0.0, 0.25, 0.75, 0.5, 0.75)
		));
		boundingBoxesForDirections.put(EnumFacing.UP, ImmutableList.of(
				new AxisAlignedBB(0.0, 0.0, 0.0, 1.0, 0.5, 1.0), new AxisAlignedBB(0.25, 0.5, 0.25, 0.75, 1.0, 0.75)
		));
		boundingBoxesForDirections.put(EnumFacing.NORTH, ImmutableList.of(
				new AxisAlignedBB(0.0, 0.0, 0.5, 1.0, 1.0, 1.0), new AxisAlignedBB(0.25, 0.25, 0.0, 0.75, 0.75, 0.5)
		));
		boundingBoxesForDirections.put(EnumFacing.SOUTH, ImmutableList.of(
				new AxisAlignedBB(0.0, 0.0, 0.0, 1.0, 1.0, 0.5), new AxisAlignedBB(0.25, 0.25, 0.5, 0.75, 0.75, 1.0)
		));
		boundingBoxesForDirections.put(EnumFacing.WEST, ImmutableList.of(
				new AxisAlignedBB(0.5, 0.0, 0.0, 1.0, 1.0, 1.0), new AxisAlignedBB(0.0, 0.25, 0.25, 0.5, 0.75, 0.75)
		));
		boundingBoxesForDirections.put(EnumFacing.EAST, ImmutableList.of(
				new AxisAlignedBB(0.0, 0.0, 0.0, 0.5, 1.0, 1.0), new AxisAlignedBB(0.5, 0.25, 0.25, 1.0, 0.75, 0.75)
		));
	}

	public BlockEngine(BlockTypeEngine blockType) {
		super(blockType);
		setHarvestLevel("pickaxe", 0);
	}

	@Override
	public void func_185477_a(IBlockState state, World worldIn, BlockPos pos, AxisAlignedBB entityBox, List<AxisAlignedBB> collidingBoxes, @Nullable Entity entityIn, boolean p_185477_7_) {
		EnumFacing orientation = state.func_177229_b(FACING);
		List<AxisAlignedBB> boundingBoxes = boundingBoxesForDirections.get(orientation);
		if (boundingBoxes == null) {
			return;
		}

		for (AxisAlignedBB boundingBoxBase : boundingBoxes) {
			AxisAlignedBB boundingBox = boundingBoxBase.func_72317_d(pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p());
			if (entityBox.func_72326_a(boundingBox)) {
				collidingBoxes.add(boundingBox);
			}
		}
	}

	@Override
	public RayTraceResult func_180636_a(IBlockState blockState, World worldIn, BlockPos pos, Vec3d start, Vec3d end) {
		EnumFacing orientation = blockState.func_177229_b(FACING);
		List<AxisAlignedBB> boundingBoxes = boundingBoxesForDirections.get(orientation);
		if (boundingBoxes == null) {
			return super.func_180636_a(blockState, worldIn, pos, start, end);
		}

		RayTraceResult nearestIntersection = null;
		for (AxisAlignedBB boundingBoxBase : boundingBoxes) {
			AxisAlignedBB boundingBox = boundingBoxBase.func_72317_d(pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p());
			RayTraceResult intersection = boundingBox.func_72327_a(start, end);
			if (intersection != null) {
				if (nearestIntersection == null || intersection.field_72307_f.func_72438_d(start) < nearestIntersection.field_72307_f.func_72438_d(start)) {
					nearestIntersection = intersection;
				}
			}
		}

		if (nearestIntersection != null) {
			Object hitInfo = nearestIntersection.hitInfo;
			Entity entityHit = nearestIntersection.field_72308_g;
			nearestIntersection = new RayTraceResult(nearestIntersection.field_72313_a, nearestIntersection.field_72307_f, nearestIntersection.field_178784_b, pos);
			nearestIntersection.hitInfo = hitInfo;
			nearestIntersection.field_72308_g = entityHit;
		}

		return nearestIntersection;
	}

	@Override
	public boolean rotateBlock(World world, BlockPos pos, EnumFacing axis) {
		return rotate(world, pos) ||
				super.rotateBlock(world, pos, axis);
	}

	private static boolean isOrientedAtEnergyReciever(World world, BlockPos pos, EnumFacing orientation) {
		TileEntity tile = world.func_175625_s(pos.func_177972_a(orientation));
		return EnergyHelper.isEnergyReceiverOrEngine(orientation.func_176734_d(), tile);
	}

	private static boolean rotate(World world, BlockPos pos) {
		IBlockState blockState = world.func_180495_p(pos);
		EnumFacing blockFacing = blockState.func_177229_b(FACING);
		for (int i = blockFacing.ordinal() + 1; i <= blockFacing.ordinal() + 6; ++i) {
			EnumFacing orientation = EnumFacing.values()[i % 6];
			if (isOrientedAtEnergyReciever(world, pos, orientation)) {
				blockState = blockState.func_177226_a(FACING, orientation);
				world.func_175656_a(pos, blockState);
				return true;
			}
		}
		return false;
	}

	@Override
	public void rotateAfterPlacement(EntityPlayer player, World world, BlockPos pos, EnumFacing side) {
		EnumFacing orientation = side.func_176734_d();
		if (isOrientedAtEnergyReciever(world, pos, orientation)) {
			IBlockState blockState = world.func_180495_p(pos);
			blockState = blockState.func_177226_a(FACING, orientation);
			world.func_175656_a(pos, blockState);
		} else {
			super.rotateAfterPlacement(player, world, pos, side);
			rotate(world, pos);
		}
	}

	@Override
	public boolean isSideSolid(IBlockState base_state, IBlockAccess world, BlockPos pos, EnumFacing side) {
		IBlockState blockState = world.func_180495_p(pos);
		EnumFacing facing = blockState.func_177229_b(BlockBase.FACING);
		return facing.func_176734_d() == side;
	}
}
