/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.core.climate;

import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import forestry.api.climate.IClimateInfo;
import forestry.api.climate.IClimatePosition;
import forestry.api.climate.IClimateRegion;
import forestry.api.climate.IClimateSource;
import forestry.core.network.IStreamable;
import forestry.core.network.PacketBufferForestry;
import forestry.greenhouse.multiblock.IGreenhouseControllerInternal;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;

public class ClimateRegion implements IClimateRegion, IStreamable {
	protected final World world;
	protected final IGreenhouseControllerInternal controller;
	protected final Set<IClimatePosition> positions;
	protected final Set<IClimateSource> sources;
	protected float temperature;
	protected float humidity;

	public ClimateRegion(IGreenhouseControllerInternal controller, Set<IClimatePosition> positions) {
		this.world = controller.getWorldObj();
		this.controller = controller;
		this.positions = positions;
		this.sources = new HashSet<>();
		calculateAverageClimate();
	}
	
	/**
	 * Creates an empty region.
	 */
	public ClimateRegion(IGreenhouseControllerInternal controller) {
		this.world = controller.getWorldObj();
		this.controller = controller;
		this.positions = new HashSet<>();
		this.sources = new HashSet<>();
	}


	public ClimateRegion(IGreenhouseControllerInternal controller, NBTTagCompound nbtTag) {
		this.world = controller.getWorldObj();
		this.controller = controller;
		this.positions = new HashSet<>();
		this.sources = new HashSet<>();
		readFromNBT(nbtTag);
		calculateAverageClimate();
	}

	@Override
	public void calculateAverageClimate(){
		humidity = 0.0F;
		temperature = 0.0F;
		if(!positions.isEmpty()){
			int positions = 0;
			for (IClimatePosition position : this.positions) {
				if (position != null) {
					positions++;
					humidity += position.getHumidity();
					temperature += position.getTemperature();
				}
			}
	
			temperature /= positions;
			humidity /= positions;
		}
	}

	@Override
	public void updateClimate(int ticks) {
		boolean hasChange = false;
		for (IClimateSource source : sources) {
			if (source != null) {
				if (ticks % source.getTicksForChange(this) == 0) {
					hasChange |= source.changeClimate(ticks, this);
				}
			}
		}
		if (ticks % getTicksPerUpdate() == 0) {
			for (IClimatePosition position : positions) {
				BlockPos pos = position.getPos();
				if (world.func_175667_e(pos)) {
					hasChange |= updateSides(position);
					if (!controller.isAssembled()) {
						IClimateInfo climateInfo = getControl(pos);

						if (position.getTemperature() != climateInfo.getTemperature()) {
							if (position.getTemperature() > climateInfo.getTemperature()) {
								position.addTemperature(-Math.min(0.01F, position.getTemperature() - climateInfo.getTemperature()));
								hasChange = true;
							} else {
								position.addTemperature(Math.min(0.01F, climateInfo.getTemperature() - position.getTemperature()));
								hasChange = true;
							}
						}
						if (position.getHumidity() != climateInfo.getHumidity()) {
							if (position.getHumidity() > climateInfo.getHumidity()) {
								position.addHumidity(-Math.min(0.01F, position.getHumidity() - climateInfo.getHumidity()));
								hasChange = true;
							} else {
								position.addHumidity(Math.min(0.01F, climateInfo.getHumidity() - position.getHumidity()));
								hasChange = true;
							}
						}
					}
				}
			}
		}
		if (hasChange) {
			calculateAverageClimate();
		}
	}

	protected boolean updateSides(IClimatePosition positon) {
		BlockPos pos = positon.getPos();
		IClimateInfo climateInfo = getControl(pos);
		boolean hasChange = false;
		if (climateInfo.getTemperature() != temperature || climateInfo.getHumidity() != humidity) {
			for (EnumFacing facing : EnumFacing.field_82609_l) {
				IClimatePosition climatedInfoFace = getPosition(pos.func_177972_a(facing));
				if (climatedInfoFace != null) {
					if (positon.getTemperature() > climatedInfoFace.getTemperature() + 0.01F) {
						float change = Math.min(0.01F, positon.getTemperature() - climatedInfoFace.getTemperature());
						positon.addTemperature(-change);
						climatedInfoFace.addTemperature(change);
						hasChange = true;
					}
					if (positon.getHumidity() > climatedInfoFace.getHumidity() + 0.01F) {
						float change = Math.min(0.01F, positon.getHumidity() - climatedInfoFace.getHumidity());
						positon.addHumidity(-change);
						climatedInfoFace.addHumidity(change);
						hasChange = true;
					}
				}
			}
		}
		return hasChange;
	}

	protected IClimateInfo getControl(BlockPos pos) {
		if (world.func_175667_e(pos)) {
			if (!controller.isAssembled()) {
				return BiomeClimateInfo.getInfo(world.func_180494_b(pos));
			}
		}
		return controller.getControlClimate();
	}
	
	@Override
	public Collection<IClimatePosition> getPositions() {
		return positions;
	}
	
	@Override
	public IClimatePosition getPosition(BlockPos pos) {
		for(IClimatePosition position : positions){
			if(position.getPos().equals(pos)){
				return position;
			}
		}
		return null;
	}

	@Override
	public NBTTagCompound writeToNBT(NBTTagCompound nbt) {
		NBTTagList positionList = new NBTTagList();
		for (IClimatePosition positon : positions) {
			BlockPos pos = positon.getPos();
			NBTTagCompound tag = new NBTTagCompound();
			tag.func_74768_a("X", pos.func_177958_n());
			tag.func_74768_a("Y", pos.func_177956_o());
			tag.func_74768_a("Z", pos.func_177952_p());
			positionList.func_74742_a(positon.writeToNBT(tag));
		}
		nbt.func_74782_a("Positions", positionList);
		return nbt;
	}

	@Override
	public void readFromNBT(NBTTagCompound nbt) {
		NBTTagList positionList = nbt.func_150295_c("Positions", 10);
		for (int i = 0; i < positionList.func_74745_c(); i++) {
			NBTTagCompound positionTag = positionList.func_150305_b(i);
			int xPos = positionTag.func_74762_e("X");
			int yPos = positionTag.func_74762_e("Y");
			int zPos = positionTag.func_74762_e("Z");
			BlockPos pos = new BlockPos(xPos, yPos, zPos);
			IClimatePosition position = getPosition(pos);
			if (position != null) {
				position.readFromNBT(positionTag);
			} else {
				positions.add(new ClimatePosition(this, pos, positionTag));
			}
		}
	}

	@Override
	public int getTicksPerUpdate() {
		return 20;
	}

	@Override
	public World getWorld() {
		return world;
	}

	@Override
	public synchronized void setPosition(BlockPos pos, float temperature, float humidity) {
		IClimatePosition position = getPosition(pos);
		if (position != null) {
			position.setHumidity(humidity);
			position.setTemperature(temperature);
		} else {
			positions.add(new ClimatePosition(this, pos, temperature, humidity));
		}
	}

	@Override
	public synchronized void addSource(IClimateSource source) {
		if (!sources.contains(source)) {
			sources.add(source);
		}
	}

	@Override
	public synchronized void removeSource(IClimateSource source) {
		if (sources.contains(source)) {
			sources.remove(source);
		}
	}

	@Override
	public Collection<IClimateSource> getSources() {
		return sources;
	}

	@Override
	public void writeData(PacketBufferForestry data) {
		if (!positions.isEmpty()) {
			data.writeInt(positions.size());
			data.writeFloat(temperature);
			data.writeFloat(humidity);
		} else {
			data.writeInt(0);
		}
	}

	@Override
	public void readData(PacketBufferForestry data) throws IOException {
		int size = data.readInt();
		if (size != 0) {
			temperature = data.readFloat();
			humidity = data.readFloat();
		}
	}

	@Override
	public float getAverageTemperature() {
		return temperature;
	}

	@Override
	public float getAverageHumidity() {
		return humidity;
	}
	
	@Override
	public boolean equals(Object obj) {
		if(!(obj instanceof ClimateRegion)){
			return false;
		}
		ClimateRegion region = (ClimateRegion) obj;
		if(region.controller != controller){
			return false;
		}
		return true;
	}

}
