/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.arboriculture.blocks;

import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import forestry.api.arboriculture.EnumGermlingType;
import forestry.api.arboriculture.ITree;
import forestry.api.arboriculture.TreeManager;
import forestry.api.core.IItemModelRegister;
import forestry.api.core.IModelManager;
import forestry.api.core.IStateMapperRegister;
import forestry.arboriculture.genetics.TreeDefinition;
import forestry.arboriculture.render.SaplingStateMapper;
import forestry.arboriculture.tiles.TileSapling;
import forestry.core.proxy.Proxies;
import forestry.core.tiles.TileUtil;
import forestry.core.utils.ItemStackUtil;
import net.minecraft.block.Block;
import net.minecraft.block.IGrowable;
import net.minecraft.block.SoundType;
import net.minecraft.block.material.Material;
import net.minecraft.block.state.BlockStateContainer;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.BlockRenderLayer;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.RayTraceResult;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

public class BlockSapling extends BlockTreeContainer implements IGrowable, IStateMapperRegister, IItemModelRegister {
	protected static final AxisAlignedBB SAPLING_AABB = new AxisAlignedBB(0.09999999403953552D, 0.0D, 0.09999999403953552D, 0.8999999761581421D, 0.800000011920929D, 0.8999999761581421D);
	/* PROPERTYS */
	public static final PropertyTree TREE = new PropertyTree("tree");

	@Nullable
	public static TileSapling getSaplingTile(IBlockAccess world, BlockPos pos) {
		return TileUtil.getTile(world, pos, TileSapling.class);
	}

	public BlockSapling() {
		super(Material.field_151585_k);
		func_149672_a(SoundType.field_185850_c);
	}

	@Override
	public AxisAlignedBB func_185496_a(IBlockState state, IBlockAccess source, BlockPos pos) {
		return SAPLING_AABB;
	}

	@Override
	public TileEntity func_149915_a(World var1, int meta) {
		return new TileSapling();
	}

	/* COLLISION BOX */

	@Nullable
	@Override
	public AxisAlignedBB func_180646_a(IBlockState blockState, IBlockAccess worldIn, BlockPos pos) {
		return null;
	}

	/* RENDERING */

	@Override
	public boolean func_149662_c(IBlockState state) {
		return false;
	}

	@Override
	public boolean isNormalCube(IBlockState state, IBlockAccess world, BlockPos pos) {
		return false;
	}

	@Override
	public BlockRenderLayer func_180664_k() {
		return BlockRenderLayer.CUTOUT;
	}

	/* STATES */
	@Override
	public int func_176201_c(IBlockState state) {
		return 0;
	}

	@Override
	public IBlockState func_176221_a(IBlockState state, IBlockAccess world, BlockPos pos) {
		TileSapling sapling = getSaplingTile(world, pos);
		if (sapling != null && sapling.getTree() != null) {
			state = state.func_177226_a(TREE, sapling.getTree().getGenome().getPrimary());
		} else {
			state = state.func_177226_a(TREE, TreeDefinition.Oak.getGenome().getPrimary());
		}
		return state;
	}

	@Override
	protected BlockStateContainer func_180661_e() {
		return new BlockStateContainer(this, TREE);
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void registerStateMapper() {
		Proxies.render.registerStateMapper(this, new SaplingStateMapper());
	}

	/* MODELS */
	@SideOnly(Side.CLIENT)
	@Override
	public void registerModel(Item item, IModelManager manager) {
		//To delete the error message
		manager.registerItemModel(item, 0, "germlings/sapling.tree_larch");
	}

	/* PLANTING */
	public static boolean canBlockStay(IBlockAccess world, BlockPos pos) {
		TileSapling tile = getSaplingTile(world, pos);
		if (tile == null) {
			return false;
		}

		ITree tree = tile.getTree();
		return tree != null && tree.canStay(world, pos);
	}

	@Override
	public void func_189540_a(IBlockState state, World worldIn, BlockPos pos, Block blockIn, BlockPos fromPos) {
		super.func_189540_a(state, worldIn, pos, blockIn, fromPos);
		if (!worldIn.field_72995_K && !canBlockStay(worldIn, pos)) {
			dropAsSapling(worldIn, pos);
			worldIn.func_175698_g(pos);
		}
	}

	/* REMOVING */
	@Override
	public List<ItemStack> getDrops(IBlockAccess world, BlockPos pos, IBlockState state, int fortune) {
		return new ArrayList<>();
	}

	@Override
	public ItemStack getPickBlock(IBlockState state, RayTraceResult target, World world, BlockPos pos, EntityPlayer player) {
		TileSapling sapling = getSaplingTile(world, pos);
		if (sapling == null || sapling.getTree() == null) {
			return ItemStack.field_190927_a;
		}
		return TreeManager.treeRoot.getMemberStack(sapling.getTree(), EnumGermlingType.SAPLING);
	}

	@Override
	public boolean removedByPlayer(IBlockState state, World world, BlockPos pos, EntityPlayer player, boolean willHarvest) {
		if (!world.field_72995_K && canHarvestBlock(world, pos, player)) {
			if (!player.field_71075_bZ.field_75098_d) {
				dropAsSapling(world, pos);
			}
		}

		return world.func_175698_g(pos);
	}

	private static void dropAsSapling(World world, BlockPos pos) {
		if (world.field_72995_K) {
			return;
		}

		TileSapling sapling = getSaplingTile(world, pos);
		if (sapling != null && sapling.getTree() != null) {
			ItemStack saplingStack = TreeManager.treeRoot.getMemberStack(sapling.getTree(), EnumGermlingType.SAPLING);
			ItemStackUtil.dropItemStackAsEntity(saplingStack, world, pos);
		}

	}

	/* GROWNING */
	@Override
	public boolean func_180670_a(World world, Random rand, BlockPos pos, IBlockState state) {
		if (world.field_73012_v.nextFloat() >= 0.45F) {
			return false;
		}
		TileSapling saplingTile = getSaplingTile(world, pos);
		return saplingTile == null || saplingTile.canAcceptBoneMeal(rand);
	}

	@Override
	public boolean func_176473_a(World world, BlockPos pos, IBlockState state, boolean isClient) {
		return true;
	}

	@Override
	public void func_176474_b(World world, Random rand, BlockPos pos, IBlockState state) {
		TileSapling saplingTile = getSaplingTile(world, pos);
		if (saplingTile != null) {
			saplingTile.tryGrow(rand, true);
		}
	}
}
