/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.mail;

import javax.annotation.Nullable;
import java.util.List;
import java.util.Random;

import forestry.api.mail.ILetter;
import forestry.api.mail.IMailAddress;
import forestry.api.mail.IStamps;
import forestry.core.inventory.InventoryAdapter;
import forestry.core.utils.InventoryUtil;
import forestry.core.utils.Translator;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.NonNullList;
import net.minecraft.util.text.ITextComponent;
import org.apache.commons.lang3.StringUtils;

public class Letter implements ILetter {
	private static final Random rand = new Random();
	public static final short SLOT_ATTACHMENT_1 = 0;
	public static final short SLOT_ATTACHMENT_COUNT = 18;
	public static final short SLOT_POSTAGE_1 = 18;
	public static final short SLOT_POSTAGE_COUNT = 4;

	private boolean isProcessed = false;

	private IMailAddress sender;
	@Nullable
	private IMailAddress recipient;

	private String text = "";
	private final InventoryAdapter inventory = new InventoryAdapter(22, "INV");
	private final String uid;

	public Letter(IMailAddress sender, IMailAddress recipient) {
		this.sender = sender;
		this.recipient = recipient;
		this.uid = String.valueOf(rand.nextInt());
	}

	public Letter(NBTTagCompound nbttagcompound) {
		this.isProcessed = nbttagcompound.func_74767_n("PRC");
		this.sender = new MailAddress(nbttagcompound.func_74775_l("SDR"));
		this.recipient = new MailAddress(nbttagcompound.func_74775_l("RC"));

		this.text = nbttagcompound.func_74779_i("TXT");
		this.uid = nbttagcompound.func_74779_i("UID");
		this.inventory.readFromNBT(nbttagcompound);
	}

	@Override
	public NBTTagCompound writeToNBT(NBTTagCompound nbttagcompound) {

		nbttagcompound.func_74757_a("PRC", isProcessed);

		NBTTagCompound subcompound = new NBTTagCompound();
		this.sender.writeToNBT(subcompound);
		nbttagcompound.func_74782_a("SDR", subcompound);

		if (this.recipient != null) {
			subcompound = new NBTTagCompound();
			this.recipient.writeToNBT(subcompound);
			nbttagcompound.func_74782_a("RC", subcompound);
		}

		nbttagcompound.func_74778_a("TXT", this.text);
		nbttagcompound.func_74778_a("UID", this.uid);
		inventory.writeToNBT(nbttagcompound);
		return nbttagcompound;
	}

	@Override
	public NonNullList<ItemStack> getPostage() {
		return InventoryUtil.getStacks(inventory, SLOT_POSTAGE_1, SLOT_POSTAGE_COUNT);
	}

	@Override
	public NonNullList<ItemStack> getAttachments() {
		return InventoryUtil.getStacks(inventory, SLOT_ATTACHMENT_1, SLOT_ATTACHMENT_COUNT);
	}

	@Override
	public int countAttachments() {
		int count = 0;
		for (ItemStack stack : getAttachments()) {
			if (stack != null) {
				count++;
			}
		}

		return count;
	}

	@Override
	public void addAttachment(ItemStack itemstack) {
		InventoryUtil.tryAddStack(inventory, itemstack, false);
	}

	@Override
	public void addAttachments(NonNullList<ItemStack> itemstacks) {
		for (ItemStack stack : itemstacks) {
			addAttachment(stack);
		}
	}

	@Override
	public void invalidatePostage() {
		for (int i = SLOT_POSTAGE_1; i < SLOT_POSTAGE_1 + SLOT_POSTAGE_COUNT; i++) {
			inventory.func_70299_a(i, ItemStack.field_190927_a);
		}
	}

	@Override
	public void setProcessed(boolean flag) {
		this.isProcessed = flag;
	}

	@Override
	public boolean isProcessed() {
		return this.isProcessed;
	}

	@Override
	public boolean isMailable() {
		// Can't resend an already sent letter
		// Requires at least one recipient
		return !isProcessed && recipient != null;
	}

	@Override
	public boolean isPostPaid() {

		int posted = 0;

		for (ItemStack stamp : getPostage()) {
			if (stamp == null) {
				continue;
			}
			if (!(stamp.func_77973_b() instanceof IStamps)) {
				continue;
			}

			posted += ((IStamps) stamp.func_77973_b()).getPostage(stamp).getValue() * stamp.func_190916_E();
		}

		return posted >= requiredPostage();
	}

	@Override
	public int requiredPostage() {

		int required = 1;
		for (ItemStack attach : getAttachments()) {
			if (attach != null) {
				required++;
			}
		}

		return required;
	}

	@Override
	public void addStamps(ItemStack stamps) {
		InventoryUtil.tryAddStack(inventory, stamps, SLOT_POSTAGE_1, 4, false);
	}

	@Override
	public boolean hasRecipient() {
		return recipient != null && !StringUtils.isBlank(recipient.getName());
	}

	@Override
	public void setSender(IMailAddress address) {
		this.sender = address;
	}

	@Override
	public IMailAddress getSender() {
		return sender;
	}

	@Override
	public void setRecipient(@Nullable IMailAddress address) {
		this.recipient = address;
	}

	@Override
	@Nullable
	public IMailAddress getRecipient() {
		return recipient;
	}

	@Override
	public String getRecipientString() {
		if (recipient == null) {
			return "";
		}
		return recipient.getName();
	}

	@Override
	public void setText(String text) {
		this.text = text;
	}

	@Override
	public String getText() {
		return text;
	}

	@Override
	public void addTooltip(List<String> list) {
		if (StringUtils.isNotBlank(this.sender.getName())) {
			list.add(Translator.translateToLocal("for.gui.mail.from") + ": " + this.sender.getName());
		}
		if (this.recipient != null) {
			list.add(Translator.translateToLocal("for.gui.mail.to") + ": " + this.getRecipientString());
		}
	}

	// / IINVENTORY
	@Override
	public boolean func_191420_l() {
		return inventory.func_191420_l();
	}

	@Override
	public int func_70302_i_() {
		return inventory.func_70302_i_();
	}

	@Override
	public ItemStack func_70301_a(int var1) {
		return inventory.func_70301_a(var1);
	}

	@Override
	public ItemStack func_70298_a(int var1, int var2) {
		return inventory.func_70298_a(var1, var2);
	}

	@Override
	public ItemStack func_70304_b(int index) {
		return inventory.func_70304_b(index);
	}

	@Override
	public void func_70299_a(int var1, ItemStack var2) {
		inventory.func_70299_a(var1, var2);
	}

	@Override
	public String func_70005_c_() {
		return inventory.func_70005_c_();
	}

	@Override
	public int func_70297_j_() {
		return inventory.func_70297_j_();
	}

	@Override
	public void func_70296_d() {
		inventory.func_70296_d();
	}

	@Override
	public boolean func_70300_a(EntityPlayer var1) {
		return true;
	}

	@Override
	public void func_174889_b(EntityPlayer var1) {
		inventory.func_174889_b(var1);
	}

	@Override
	public void func_174886_c(EntityPlayer var1) {
		inventory.func_174886_c(var1);
	}

	@Override
	public boolean func_145818_k_() {
		return inventory.func_145818_k_();
	}

	@Override
	public boolean func_94041_b(int i, ItemStack itemstack) {
		return inventory.func_94041_b(i, itemstack);
	}

	/* FIELDS */
	@Override
	public int func_174887_a_(int id) {
		return 0;
	}

	@Override
	public void func_174885_b(int id, int value) {
	}

	@Override
	public int func_174890_g() {
		return 0;
	}

	@Override
	public void func_174888_l() {
	}

	@Override
	public ITextComponent func_145748_c_() {
		return inventory.func_145748_c_();
	}
}
