/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.core.tiles;

import javax.annotation.Nullable;
import java.io.IOException;

import forestry.api.arboriculture.TreeManager;
import forestry.api.core.ForestryAPI;
import forestry.api.genetics.AlleleManager;
import forestry.api.genetics.IIndividual;
import forestry.core.config.Constants;
import forestry.core.errors.EnumErrorCode;
import forestry.core.fluids.FilteredTank;
import forestry.core.fluids.FluidHelper;
import forestry.core.fluids.Fluids;
import forestry.core.fluids.TankManager;
import forestry.core.gui.ContainerAnalyzer;
import forestry.core.gui.GuiAnalyzer;
import forestry.core.inventory.InventoryAnalyzer;
import forestry.core.inventory.wrappers.InventoryMapper;
import forestry.core.network.PacketBufferForestry;
import forestry.core.network.packets.PacketItemStackDisplay;
import forestry.core.proxy.Proxies;
import forestry.core.utils.GeneticsUtil;
import forestry.core.utils.InventoryUtil;
import forestry.plugins.ForestryPluginUids;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.IInventory;
import net.minecraft.inventory.ISidedInventory;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.EnumFacing;
import net.minecraftforge.common.capabilities.Capability;
import net.minecraftforge.fluids.FluidStack;
import net.minecraftforge.fluids.capability.CapabilityFluidHandler;

public class TileAnalyzer extends TilePowered implements ISidedInventory, ILiquidTankTile, IItemStackDisplay {
	private static final int TIME_TO_ANALYZE = 125;
	private static final int HONEY_REQUIRED = 100;
	private static final int ENERGY_PER_WORK_CYCLE = 20320;

	private final FilteredTank resourceTank;
	private final TankManager tankManager;
	private final IInventory invInput;
	private final IInventory invOutput;

	@Nullable
	private IIndividual specimenToAnalyze;
	private ItemStack individualOnDisplayClient = ItemStack.field_190927_a;

	/* CONSTRUCTOR */
	public TileAnalyzer() {
		super(800, Constants.MACHINE_MAX_ENERGY);
		setInternalInventory(new InventoryAnalyzer(this));
		resourceTank = new FilteredTank(Constants.PROCESSOR_TANK_CAPACITY).setFilters(Fluids.FOR_HONEY.getFluid());
		tankManager = new TankManager(this, resourceTank);
		invInput = new InventoryMapper(getInternalInventory(), InventoryAnalyzer.SLOT_INPUT_1, InventoryAnalyzer.SLOT_INPUT_COUNT);
		invOutput = new InventoryMapper(getInternalInventory(), InventoryAnalyzer.SLOT_OUTPUT_1, InventoryAnalyzer.SLOT_OUTPUT_COUNT);
	}

	/* SAVING & LOADING */

	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound nbttagcompound) {
		nbttagcompound = super.func_189515_b(nbttagcompound);
		tankManager.writeToNBT(nbttagcompound);
		return nbttagcompound;
	}

	@Override
	public void func_145839_a(NBTTagCompound nbttagcompound) {
		super.func_145839_a(nbttagcompound);
		tankManager.readFromNBT(nbttagcompound);

		ItemStack stackToAnalyze = func_70301_a(InventoryAnalyzer.SLOT_ANALYZE);
		if (!stackToAnalyze.func_190926_b()) {
			specimenToAnalyze = AlleleManager.alleleRegistry.getIndividual(stackToAnalyze);
		}
	}

	@Override
	protected void updateServerSide() {
		super.updateServerSide();

		if (updateOnInterval(20)) {
			// Check if we have suitable items waiting in the can slot
			FluidHelper.drainContainers(tankManager, this, InventoryAnalyzer.SLOT_CAN);
		}
	}

	/* WORKING */
	@Override
	public boolean workCycle() {
		ItemStack stackToAnalyze = func_70301_a(InventoryAnalyzer.SLOT_ANALYZE);
		if (stackToAnalyze.func_190926_b() || specimenToAnalyze == null) {
			return false;
		}

		if (!specimenToAnalyze.isAnalyzed()) {
			FluidStack drained = resourceTank.drain(HONEY_REQUIRED, false);
			if (drained == null || drained.amount != HONEY_REQUIRED) {
				return false;
			}
			resourceTank.drain(HONEY_REQUIRED, true);

			specimenToAnalyze.analyze();

			NBTTagCompound nbttagcompound = new NBTTagCompound();
			specimenToAnalyze.writeToNBT(nbttagcompound);
			stackToAnalyze.func_77982_d(nbttagcompound);
		}

		boolean added = InventoryUtil.tryAddStack(invOutput, stackToAnalyze, true);
		if (!added) {
			return false;
		}

		func_70299_a(InventoryAnalyzer.SLOT_ANALYZE, ItemStack.field_190927_a);
		PacketItemStackDisplay packet = new PacketItemStackDisplay(this, getIndividualOnDisplay());
		Proxies.net.sendNetworkPacket(packet, field_174879_c, field_145850_b);

		return true;
	}

	@Nullable
	private Integer getInputSlotIndex() {
		for (int slotIndex = 0; slotIndex < invInput.func_70302_i_(); slotIndex++) {
			ItemStack inputStack = invInput.func_70301_a(slotIndex);
			if (AlleleManager.alleleRegistry.isIndividual(inputStack)) {
				return slotIndex;
			}
		}
		return null;
	}

	/* Network */
	@Override
	public void writeData(PacketBufferForestry data) {
		super.writeData(data);
		ItemStack displayStack = getIndividualOnDisplay();
		data.func_150788_a(displayStack);
		tankManager.writeData(data);
	}

	@Override
	public void readData(PacketBufferForestry data) throws IOException {
		super.readData(data);
		individualOnDisplayClient = data.func_150791_c();
		tankManager.readData(data);
	}

	@Override
	public void handleItemStackForDisplay(ItemStack itemStack) {
		if (!ItemStack.func_77989_b(itemStack, individualOnDisplayClient)) {
			individualOnDisplayClient = itemStack;
			field_145850_b.func_175704_b(func_174877_v(), func_174877_v());
		}
	}

	/* STATE INFORMATION */
	@Override
	public boolean hasWork() {
		moveSpecimenToAnalyzeSlot();

		ItemStack specimen = func_70301_a(InventoryAnalyzer.SLOT_ANALYZE);

		boolean hasSpecimen = !specimen.func_190926_b();
		boolean hasResource = true;
		boolean hasSpace = true;

		if (hasSpecimen) {
			hasSpace = InventoryUtil.tryAddStack(invOutput, specimen, true, false);

			if (specimenToAnalyze != null && !specimenToAnalyze.isAnalyzed()) {
				FluidStack drained = resourceTank.drain(HONEY_REQUIRED, false);
				hasResource = drained != null && drained.amount == HONEY_REQUIRED;
			}
		}

		getErrorLogic().setCondition(!hasSpecimen, EnumErrorCode.NO_SPECIMEN);
		getErrorLogic().setCondition(!hasResource, EnumErrorCode.NO_RESOURCE_LIQUID);
		getErrorLogic().setCondition(!hasSpace, EnumErrorCode.NO_SPACE_INVENTORY);

		return hasSpecimen && hasResource && hasSpace;
	}

	private void moveSpecimenToAnalyzeSlot() {
		if (!func_70301_a(InventoryAnalyzer.SLOT_ANALYZE).func_190926_b()) {
			return;
		}

		Integer slotIndex = getInputSlotIndex();
		if (slotIndex == null) {
			return;
		}

		ItemStack inputStack = invInput.func_70301_a(slotIndex);
		if (inputStack.func_190926_b()) {
			return;
		}

		if (ForestryAPI.enabledPlugins.contains(ForestryPluginUids.ARBORICULTURE) && !TreeManager.treeRoot.isMember(inputStack)) {
			inputStack = GeneticsUtil.convertToGeneticEquivalent(inputStack);
		}

		specimenToAnalyze = AlleleManager.alleleRegistry.getIndividual(inputStack);
		if (specimenToAnalyze == null) {
			return;
		}

		func_70299_a(InventoryAnalyzer.SLOT_ANALYZE, inputStack);
		invInput.func_70299_a(slotIndex, ItemStack.field_190927_a);

		if (specimenToAnalyze.isAnalyzed()) {
			setTicksPerWorkCycle(1);
			setEnergyPerWorkCycle(0);
		} else {
			setTicksPerWorkCycle(TIME_TO_ANALYZE);
			setEnergyPerWorkCycle(ENERGY_PER_WORK_CYCLE);
		}

		PacketItemStackDisplay packet = new PacketItemStackDisplay(this, getIndividualOnDisplay());
		Proxies.net.sendNetworkPacket(packet, field_174879_c, field_145850_b);
	}

	public ItemStack getIndividualOnDisplay() {
		if (field_145850_b.field_72995_K) {
			return individualOnDisplayClient;
		}
		return func_70301_a(InventoryAnalyzer.SLOT_ANALYZE);
	}

	/* ILiquidTankTile */

	@Override
	public TankManager getTankManager() {
		return tankManager;
	}

	@Override
	public boolean hasCapability(Capability<?> capability, @Nullable EnumFacing facing) {
		return capability == CapabilityFluidHandler.FLUID_HANDLER_CAPABILITY || super.hasCapability(capability, facing);
	}


	@Override
	@Nullable
	public <T> T getCapability(Capability<T> capability, @Nullable EnumFacing facing) {
		if (capability == CapabilityFluidHandler.FLUID_HANDLER_CAPABILITY) {
			return CapabilityFluidHandler.FLUID_HANDLER_CAPABILITY.cast(tankManager);
		}
		return super.getCapability(capability, facing);
	}

	@Override
	public Object getGui(EntityPlayer player, int data) {
		return new GuiAnalyzer(player.field_71071_by, this);
	}

	@Override
	public Object getContainer(EntityPlayer player, int data) {
		return new ContainerAnalyzer(player.field_71071_by, this);
	}
}
