/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.mail.gui;

import javax.annotation.Nullable;
import java.util.Iterator;

import com.mojang.authlib.GameProfile;
import forestry.api.mail.EnumAddressee;
import forestry.api.mail.ILetter;
import forestry.api.mail.IMailAddress;
import forestry.api.mail.IPostalCarrier;
import forestry.api.mail.ITradeStation;
import forestry.api.mail.ITradeStationInfo;
import forestry.api.mail.PostManager;
import forestry.core.gui.ContainerItemInventory;
import forestry.core.gui.slots.SlotFiltered;
import forestry.core.proxy.Proxies;
import forestry.core.utils.Log;
import forestry.mail.Letter;
import forestry.mail.inventory.ItemInventoryLetter;
import forestry.mail.network.packets.PacketLetterInfoResponse;
import forestry.mail.network.packets.PacketLetterTextSet;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.server.MinecraftServer;
import net.minecraft.world.World;

public class ContainerLetter extends ContainerItemInventory<ItemInventoryLetter> implements ILetterInfoReceiver {

	private EnumAddressee carrierType = EnumAddressee.PLAYER;
	@Nullable
	private ITradeStationInfo tradeInfo = null;

	public ContainerLetter(EntityPlayer player, ItemInventoryLetter inventory) {
		super(inventory, player.field_71071_by, 17, 145);

		// Init slots

		// Stamps
		for (int i = 0; i < 4; i++) {
			func_75146_a(new SlotFiltered(inventory, Letter.SLOT_POSTAGE_1 + i, 150, 14 + i * 19).setStackLimit(1));
		}

		// Attachments
		for (int i = 0; i < 2; i++) {
			for (int j = 0; j < 9; j++) {
				func_75146_a(new SlotFiltered(inventory, Letter.SLOT_ATTACHMENT_1 + j + i * 9, 17 + j * 18, 98 + i * 18));
			}
		}

		// Rip open delivered mails
		if (!player.field_70170_p.field_72995_K) {
			if (inventory.getLetter().isProcessed()) {
				inventory.onLetterOpened();
			}
		}

		// Set recipient type
		ILetter letter = inventory.getLetter();
		IMailAddress recipient = letter.getRecipient();
		if (recipient != null) {
			this.carrierType = recipient.getType();
		}
	}

	@Override
	public void func_75134_a(EntityPlayer entityplayer) {

		if (!entityplayer.field_70170_p.field_72995_K) {
			ILetter letter = inventory.getLetter();
			if (!letter.isProcessed()) {
				IMailAddress sender = PostManager.postRegistry.getMailAddress(entityplayer.func_146103_bH());
				letter.setSender(sender);
			}
		}

		inventory.onLetterClosed();

		super.func_75134_a(entityplayer);
	}

	public ILetter getLetter() {
		return inventory.getLetter();
	}

	public void setCarrierType(EnumAddressee type) {
		this.carrierType = type;
	}

	public EnumAddressee getCarrierType() {
		return this.carrierType;
	}

	public void advanceCarrierType() {
		Iterator<IPostalCarrier> it = PostManager.postRegistry.getRegisteredCarriers().values().iterator();
		while (it.hasNext()) {
			if (it.next().getType().equals(carrierType)) {
				break;
			}
		}

		IPostalCarrier postal;
		if (it.hasNext()) {
			postal = it.next();
		} else {
			postal = PostManager.postRegistry.getRegisteredCarriers().values().iterator().next();
		}

		setCarrierType(postal.getType());
	}

	public void handleRequestLetterInfo(EntityPlayer player, String recipientName, EnumAddressee type) {
		MinecraftServer server = player.func_184102_h();
		if (server == null) {
			Log.error("Could not get server");
			return;
		}
		IMailAddress recipient = getRecipient(server, recipientName, type);

		getLetter().setRecipient(recipient);

		// Update the trading info
		if (recipient == null || recipient.getType() == EnumAddressee.TRADER) {
			updateTradeInfo(player.field_70170_p, recipient);
		}

		// Update info on client
		Proxies.net.sendToPlayer(new PacketLetterInfoResponse(type, tradeInfo, recipient), player);
	}

	@Nullable
	private static IMailAddress getRecipient(MinecraftServer minecraftServer, String recipientName, EnumAddressee type) {
		switch (type) {
			case PLAYER: {
				GameProfile gameProfile = minecraftServer.func_152358_ax().func_152655_a(recipientName);
				if (gameProfile == null) {
					return null;
				}
				return PostManager.postRegistry.getMailAddress(gameProfile);
			}
			case TRADER: {
				return PostManager.postRegistry.getMailAddress(recipientName);
			}
			default:
				return null;
		}
	}

	@Nullable
	public IMailAddress getRecipient() {
		return getLetter().getRecipient();
	}

	public String getText() {
		return getLetter().getText();
	}

	public void setText(String text) {
		getLetter().setText(text);

		Proxies.net.sendToServer(new PacketLetterTextSet(text));
	}

	public void handleSetText(String text) {
		getLetter().setText(text);
	}

	/* Managing Trade info */
	private void updateTradeInfo(World world, @Nullable IMailAddress address) {
		// Updating is done by the server.
		if (world.field_72995_K) {
			return;
		}

		if (address == null) {
			setTradeInfo(null);
			return;
		}

		ITradeStation station = PostManager.postRegistry.getTradeStation(world, address);
		if (station == null) {
			setTradeInfo(null);
			return;
		}

		setTradeInfo(station.getTradeInfo());
	}

	@Override
	public void handleLetterInfoUpdate(EnumAddressee type, @Nullable IMailAddress address, @Nullable ITradeStationInfo tradeInfo) {
		carrierType = type;
		if (type == EnumAddressee.PLAYER) {
			getLetter().setRecipient(address);
		} else if (type == EnumAddressee.TRADER) {
			this.setTradeInfo(tradeInfo);
		}
	}

	@Nullable
	public ITradeStationInfo getTradeInfo() {
		return this.tradeInfo;
	}

	private void setTradeInfo(@Nullable ITradeStationInfo info) {
		this.tradeInfo = info;
		if (tradeInfo == null) {
			getLetter().setRecipient(null);
		} else {
			getLetter().setRecipient(tradeInfo.getAddress());
		}
	}
}
