/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.core.gui;

import javax.annotation.Nullable;
import java.io.IOException;
import java.util.HashMap;

import forestry.api.apiculture.IApiaristTracker;
import forestry.api.arboriculture.EnumTreeChromosome;
import forestry.api.genetics.IAlleleSpecies;
import forestry.api.genetics.IBreedingTracker;
import forestry.api.genetics.IIndividual;
import forestry.api.genetics.IMutation;
import forestry.api.genetics.ISpeciesRoot;
import forestry.core.config.Constants;
import forestry.core.genetics.mutations.EnumMutateChance;
import forestry.core.gui.buttons.GuiBetterButton;
import forestry.core.gui.buttons.StandardButtonTextureSets;
import forestry.core.network.packets.PacketGuiSelectRequest;
import forestry.core.proxy.Proxies;
import forestry.core.render.ColourProperties;
import forestry.core.utils.Translator;
import net.minecraft.client.gui.GuiButton;
import net.minecraft.client.renderer.RenderHelper;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.Container;
import net.minecraft.inventory.Slot;
import net.minecraft.item.ItemStack;

public class GuiNaturalistInventory extends GuiForestry<Container> {
	private final ISpeciesRoot speciesRoot;
	private final IBreedingTracker breedingTracker;
	private final HashMap<String, ItemStack> iconStacks = new HashMap<>();
	private final int pageCurrent, pageMax;

	public GuiNaturalistInventory(ISpeciesRoot speciesRoot, EntityPlayer player, Container container, int page, int maxPages) {
		super(Constants.TEXTURE_PATH_GUI + "/apiaristinventory.png", container);

		this.speciesRoot = speciesRoot;

		this.pageCurrent = page;
		this.pageMax = maxPages;

		field_146999_f = 196;
		field_147000_g = 202;

		for (IIndividual individual : speciesRoot.getIndividualTemplates()) {
			iconStacks.put(individual.getIdent(), speciesRoot.getMemberStack(individual, speciesRoot.getIconType()));
		}

		breedingTracker = speciesRoot.getBreedingTracker(player.field_70170_p, player.func_146103_bH());
	}

	@Override
	protected void func_146976_a(float f, int i, int j) {
		super.func_146976_a(f, i, j);
		String header = Translator.translateToLocal("for.gui.page") + " " + (pageCurrent + 1) + "/" + pageMax;
		field_146289_q.func_78276_b(header, field_147003_i + 95 + textLayout.getCenteredOffset(header, 98), field_147009_r + 10, ColourProperties.INSTANCE.get("gui.title"));

		IIndividual individual = getIndividualAtPosition(i, j);
		if (individual == null) {
			displayBreedingStatistics(10);
		}

		if (individual != null) {
			RenderHelper.func_74520_c();
			textLayout.startPage();

			displaySpeciesInformation(true, individual.getGenome().getPrimary(), iconStacks.get(individual.getIdent()), 10);
			if (!individual.isPureBred(EnumTreeChromosome.SPECIES)) {
				displaySpeciesInformation(individual.isAnalyzed(), individual.getGenome().getSecondary(), iconStacks.get(individual.getGenome().getSecondary().getUID()), 10);
			}

			textLayout.endPage();
		}
	}

	@Override
	public void func_73866_w_() {
		super.func_73866_w_();

		field_146292_n.add(new GuiBetterButton(1, field_147003_i + 99, field_147009_r + 7, StandardButtonTextureSets.LEFT_BUTTON_SMALL));
		field_146292_n.add(new GuiBetterButton(2, field_147003_i + 180, field_147009_r + 7, StandardButtonTextureSets.RIGHT_BUTTON_SMALL));
	}

	private static void flipPage(int page) {
		Proxies.net.sendToServer(new PacketGuiSelectRequest(page, 0));
	}

	@Override
	protected void func_146284_a(GuiButton guibutton) throws IOException {
		super.func_146284_a(guibutton);

		if (guibutton.field_146127_k == 1 && pageCurrent > 0) {
			flipPage(pageCurrent - 1);
		} else if (guibutton.field_146127_k == 2 && pageCurrent < pageMax - 1) {
			flipPage(pageCurrent + 1);
		}
	}

	@Nullable
	private IIndividual getIndividualAtPosition(int x, int y) {
		Slot slot = getSlotAtPosition(x, y);
		if (slot == null) {
			return null;
		}

		if (!slot.func_75216_d()) {
			return null;
		}

		if (!slot.func_75211_c().func_77942_o()) {
			return null;
		}

		if (!speciesRoot.isMember(slot.func_75211_c())) {
			return null;
		}

		return speciesRoot.getMember(slot.func_75211_c());
	}

	private void displayBreedingStatistics(int x) {

		textLayout.startPage();

		textLayout.drawLine(Translator.translateToLocal("for.gui.speciescount") + ": " + breedingTracker.getSpeciesBred() + "/" + speciesRoot.getSpeciesCount(), x);
		textLayout.newLine();
		textLayout.newLine();

		if (breedingTracker instanceof IApiaristTracker) {
			IApiaristTracker tracker = (IApiaristTracker) breedingTracker;
			textLayout.drawLine(Translator.translateToLocal("for.gui.queens") + ": " + tracker.getQueenCount(), x);
			textLayout.newLine();

			textLayout.drawLine(Translator.translateToLocal("for.gui.princesses") + ": " + tracker.getPrincessCount(), x);
			textLayout.newLine();

			textLayout.drawLine(Translator.translateToLocal("for.gui.drones") + ": " + tracker.getDroneCount(), x);
			textLayout.newLine();
		}

		textLayout.endPage();
	}

	private void displaySpeciesInformation(boolean analyzed, IAlleleSpecies species, ItemStack iconStack, int x) {

		if (!analyzed) {
			textLayout.drawLine(Translator.translateToLocal("for.gui.unknown"), x);
			return;
		}

		textLayout.drawLine(species.getName(), x);
		GuiUtil.drawItemStack(this, iconStack, field_147003_i + x + 69, field_147009_r + textLayout.getLineY() - 2);

		textLayout.newLine();

		// Viable Combinations
		int columnWidth = 16;
		int column = 10;

		for (IMutation combination : speciesRoot.getCombinations(species)) {
			if (combination.isSecret()) {
				continue;
			}

			if (breedingTracker.isDiscovered(combination)) {
				drawMutationIcon(combination, species, column);
			} else {
				drawUnknownIcon(combination, column);
			}

			column += columnWidth;
			if (column > 75) {
				column = 10;
				textLayout.newLine(18);
			}
		}

		textLayout.newLine();
		textLayout.newLine();
	}

	private void drawMutationIcon(IMutation combination, IAlleleSpecies species, int x) {
		GuiUtil.drawItemStack(this, iconStacks.get(combination.getPartner(species).getUID()), field_147003_i + x, field_147009_r + textLayout.getLineY());

		int line = 48;
		int column;
		EnumMutateChance chance = EnumMutateChance.rateChance(combination.getBaseChance());
		if (chance == EnumMutateChance.HIGHEST) {
			line += 16;
			column = 228;
		} else if (chance == EnumMutateChance.HIGHER) {
			line += 16;
			column = 212;
		} else if (chance == EnumMutateChance.HIGH) {
			line += 16;
			column = 196;
		} else if (chance == EnumMutateChance.NORMAL) {
			line += 0;
			column = 228;
		} else if (chance == EnumMutateChance.LOW) {
			line += 0;
			column = 212;
		} else {
			line += 0;
			column = 196;
		}

		Proxies.render.bindTexture(textureFile);
		func_73729_b(field_147003_i + x, field_147009_r + textLayout.getLineY(), column, line, 16, 16);

	}

	private void drawUnknownIcon(IMutation mutation, int x) {

		float chance = mutation.getBaseChance();

		int line;
		int column;
		if (chance >= 20) {
			line = 16;
			column = 228;
		} else if (chance >= 15) {
			line = 16;
			column = 212;
		} else if (chance >= 12) {
			line = 16;
			column = 196;
		} else if (chance >= 10) {
			line = 0;
			column = 228;
		} else if (chance >= 5) {
			line = 0;
			column = 212;
		} else {
			line = 0;
			column = 196;
		}

		Proxies.render.bindTexture(textureFile);
		func_73729_b(field_147003_i + x, field_147009_r + textLayout.getLineY(), column, line, 16, 16);
	}

	@Override
	protected void addLedgers() {
		addHintLedger("naturalist.chest");
	}
}
