/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.greenhouse.tiles;

import javax.annotation.Nonnull;

import net.minecraft.nbt.NBTTagCompound;

import forestry.api.core.IClimateControlled;
import forestry.api.greenhouse.IGreenhouseHousing;
import forestry.api.multiblock.IGreenhouseComponent;
import forestry.api.multiblock.IGreenhouseController;
import forestry.apiculture.network.packets.PacketActiveUpdate;
import forestry.core.proxy.Proxies;
import forestry.core.tiles.IActivatable;
import forestry.greenhouse.multiblock.IGreenhouseControllerInternal;

public class TileGreenhouseClimatiser extends TileGreenhouse implements IActivatable, IGreenhouseComponent.Climatiser {
	
	protected static final int WORK_CYCLES = 1;
	protected static final int ENERGY_PER_OPERATION = 50;
	
	protected enum ClimitiserType {
		TEMPERATURE, HUMIDITY
	}
	
	protected interface IClimitiserDefinition {
		float getChangePerTransfer();

		float getBoundaryUp();

		float getBoundaryDown();
		
		ClimitiserType getType();
	}
	
	private final IClimitiserDefinition definition;
	
	protected int workingTime = 0;
	
	private boolean active;
	
	protected TileGreenhouseClimatiser(IClimitiserDefinition definition) {
		this.definition = definition;
	}
	
	@Override
	public <G extends IGreenhouseController & IGreenhouseHousing & IClimateControlled> void changeClimate(int tick, G greenhouse) {
		IGreenhouseControllerInternal greenhouseInternal = (IGreenhouseControllerInternal) greenhouse;
		if (workingTime < 20 && greenhouseInternal.getEnergyManager().consumeEnergyToDoWork(WORK_CYCLES, ENERGY_PER_OPERATION)) {
			// one tick of work for every 10 RF
			workingTime += ENERGY_PER_OPERATION / 10;
		}

		if (workingTime > 0) {
			workingTime--;
			if (definition.getType() == ClimitiserType.TEMPERATURE) {
				greenhouse.addTemperatureChange(definition.getChangePerTransfer(), definition.getBoundaryDown(), definition.getBoundaryUp());
			} else if (definition.getType() == ClimitiserType.HUMIDITY) {
				greenhouse.addHumidityChange(definition.getChangePerTransfer(), definition.getBoundaryDown(), definition.getBoundaryUp());
			}
		}

		setActive(workingTime > 0);
	}
	
	/* LOADING & SAVING */
	@Override
	public void func_145839_a(NBTTagCompound nbttagcompound) {
		super.func_145839_a(nbttagcompound);
		workingTime = nbttagcompound.func_74762_e("Heating");
		setActive(workingTime > 0);
	}

	@Nonnull
	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound nbttagcompound) {
		nbttagcompound = super.func_189515_b(nbttagcompound);
		nbttagcompound.func_74768_a("Heating", workingTime);
		return nbttagcompound;
	}
	
	/* Network */
	@Override
	protected void encodeDescriptionPacket(NBTTagCompound packetData) {
		super.encodeDescriptionPacket(packetData);
		packetData.func_74757_a("Active", active);
	}

	@Override
	protected void decodeDescriptionPacket(NBTTagCompound packetData) {
		super.decodeDescriptionPacket(packetData);
		setActive(packetData.func_74767_n("Active"));
	}

	/* IActivatable */
	@Override
	public boolean isActive() {
		return active;
	}
	
	public IClimitiserDefinition getDefinition() {
		return definition;
	}

	@Override
	public void setActive(boolean active) {
		if (this.active == active) {
			return;
		}

		this.active = active;

		if (field_145850_b != null) {
			if (field_145850_b.field_72995_K) {
				field_145850_b.func_175704_b(func_174877_v(), func_174877_v());
			} else {
				Proxies.net.sendNetworkPacket(new PacketActiveUpdate(this), field_145850_b);
			}
		}
	}

}
