/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.arboriculture.tiles;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Random;

import net.minecraft.block.BlockCocoa;
import net.minecraft.block.state.IBlockState;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.play.server.SPacketUpdateTileEntity;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;

import forestry.api.arboriculture.IAlleleFruit;
import forestry.api.genetics.AlleleManager;
import forestry.api.genetics.IAllele;
import forestry.api.genetics.IFruitBearer;
import forestry.api.genetics.IFruitFamily;
import forestry.core.config.Constants;
import forestry.core.network.DataInputStreamForestry;
import forestry.core.network.DataOutputStreamForestry;
import forestry.core.network.IStreamable;
import forestry.core.utils.BlockUtil;
import forestry.core.utils.Log;
import forestry.core.utils.NBTUtilForestry;

public class TileFruitPod extends TileEntity implements IFruitBearer, IStreamable {

	private static final short MAX_MATURITY = 2;

	private IAlleleFruit allele;

	private short maturity;
	private float sappiness;

	public TileFruitPod() {
		Log.debug("Made a fruit pod");
	}

	public void setProperties(IAlleleFruit allele, float sappiness) {
		this.allele = allele;
		this.sappiness = sappiness;
	}

	/* SAVING & LOADING */
	@Override
	public void func_145839_a(NBTTagCompound nbttagcompound) {
		super.func_145839_a(nbttagcompound);

		IAllele stored = AlleleManager.alleleRegistry.getAllele(nbttagcompound.func_74779_i("UID"));
		if (stored instanceof IAlleleFruit) {
			allele = (IAlleleFruit) stored;
		} else {
			allele = (IAlleleFruit) AlleleManager.alleleRegistry.getAllele(Constants.MOD_ID + ".fruitCocoa");
		}

		maturity = nbttagcompound.func_74765_d("MT");
		sappiness = nbttagcompound.func_74760_g("SP");
	}

	@Override
	public NBTTagCompound func_189515_b(NBTTagCompound nbttagcompound) {
		nbttagcompound = super.func_189515_b(nbttagcompound);

		if (allele != null) {
			nbttagcompound.func_74778_a("UID", allele.getUID());
		}

		nbttagcompound.func_74777_a("MT", maturity);
		nbttagcompound.func_74776_a("SP", sappiness);
		return nbttagcompound;
	}

	/* UPDATING */
	public void onBlockTick(World world, BlockPos pos, IBlockState state, Random rand) {
		if (canMature() && rand.nextFloat() <= sappiness) {
			addRipeness(0.5f);
		}
	}

	public boolean canMature() {
		return maturity < MAX_MATURITY;
	}

	public short getMaturity() {
		return maturity;
	}

	@Nullable
	public ItemStack getPickBlock() {
		Map<ItemStack, Float> products = allele.getProvider().getProducts();

		ItemStack pickBlock = null;
		Float maxChance = 0.0f;
		for (Map.Entry<ItemStack, Float> product : products.entrySet()) {
			if (maxChance < product.getValue()) {
				maxChance = product.getValue();
				pickBlock = product.getKey().func_77946_l();
			}
		}

		pickBlock.field_77994_a = 1;
		return pickBlock;
	}

	public List<ItemStack> getDrops() {
		return allele.getProvider().getFruits(null, field_145850_b, func_174877_v(), maturity);
	}

	/* NETWORK */
	@Nullable
	@Override
	public SPacketUpdateTileEntity func_189518_D_() {
		return new SPacketUpdateTileEntity(this.func_174877_v(), 0, func_189517_E_());
	}

	@Nonnull
	@Override
	public NBTTagCompound func_189517_E_() {
		NBTTagCompound tag = super.func_189517_E_();
		return NBTUtilForestry.writeStreamableToNbt(this, tag);
	}

	@Override
	public void handleUpdateTag(@Nonnull NBTTagCompound tag) {
		super.handleUpdateTag(tag);
		NBTUtilForestry.readStreamableFromNbt(this, tag);
	}

	/* IFRUITBEARER */
	@Override
	public boolean hasFruit() {
		return true;
	}

	@Override
	public IFruitFamily getFruitFamily() {
		return allele.getProvider().getFamily();
	}

	@Override
	public Collection<ItemStack> pickFruit(ItemStack tool) {
		Collection<ItemStack> fruits = getDrops();
		maturity = 0;

		IBlockState oldState = field_145850_b.func_180495_p(func_174877_v());
		IBlockState newState = oldState.func_177226_a(BlockCocoa.field_176501_a, 0);
		BlockUtil.setBlockWithBreakSound(field_145850_b, func_174877_v(), newState, oldState);

		return fruits;
	}

	@Override
	public float getRipeness() {
		return (float) maturity / MAX_MATURITY;
	}

	@Override
	public void addRipeness(float add) {
		int previousAge = (int) Math.floor(maturity);

		maturity += MAX_MATURITY * add;
		if (maturity > MAX_MATURITY) {
			maturity = MAX_MATURITY;
		}

		int age = (int) Math.floor(maturity);
		if (age - previousAge > 0) {
			IBlockState state = field_145850_b.func_180495_p(func_174877_v()).func_177226_a(BlockCocoa.field_176501_a, age);
			field_145850_b.func_175656_a(func_174877_v(), state);
		}
	}

	@Override
	public boolean shouldRefresh(World world, BlockPos pos, IBlockState oldState, IBlockState newState) {
		return oldState.func_177230_c() != newState.func_177230_c();
	}

	@Override
	public void writeData(DataOutputStreamForestry data) throws IOException {
		if (allele != null) {
			data.writeUTF(allele.getUID());
		} else {
			data.writeUTF("");
		}
	}

	@Override
	public void readData(DataInputStreamForestry data) throws IOException {
		IAllele stored = AlleleManager.alleleRegistry.getAllele(data.readUTF());
		if (stored instanceof IAlleleFruit) {
			allele = (IAlleleFruit) stored;
		} else {
			allele = (IAlleleFruit) AlleleManager.alleleRegistry.getAllele(Constants.MOD_ID + ".fruitCocoa");
		}
		field_145850_b.func_175704_b(func_174877_v(), func_174877_v());
	}
}
