/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.greenhouse.tiles;

import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.math.BlockPos;
import forestry.api.climate.IClimateSource;
import forestry.api.climate.IClimatiserDefinition;
import forestry.api.multiblock.IGreenhouseComponent;
import forestry.api.multiblock.IMultiblockController;
import forestry.api.multiblock.IMultiblockLogic;
import forestry.apiculture.network.packets.PacketActiveUpdate;
import forestry.core.climate.ClimateSource;
import forestry.core.proxy.Proxies;
import forestry.core.tiles.IActivatable;
import forestry.greenhouse.GreenhouseClimateSource;

public class TileGreenhouseClimatiser extends TileGreenhouse implements IActivatable, IGreenhouseComponent.Climatiser {
	
	protected static final int WORK_CYCLES = 1;
	protected static final int ENERGY_PER_OPERATION = 150;
	
	private final IClimatiserDefinition definition;
	private final ClimateSource source;
	
	protected EnumFacing inwards;
	protected EnumFacing leftwards;
	protected BlockPos maxPos;
	protected BlockPos minPos;
	
	private boolean active;
	
	protected TileGreenhouseClimatiser(IClimatiserDefinition definition, int ticksForChange) {
		this(definition, new GreenhouseClimateSource(ticksForChange));
	}
	
	protected TileGreenhouseClimatiser(IClimatiserDefinition definition, ClimateSource source) {
		this.definition = definition;
		this.source = source;
		this.source.setProvider(this);
	}
	
	protected TileGreenhouseClimatiser(IClimatiserDefinition definition) {
		this(definition, 20 + ENERGY_PER_OPERATION / 25);
	}
	
	@Override
	public void onMachineBroken() {
		inwards = null;
		leftwards = null;
		
		minPos = null;
		maxPos = null;
	}
	
	@Override
	public void onMachineAssembled(IMultiblockController multiblockController, BlockPos minCoord, BlockPos maxCoord) {
		recalculateDirections(minCoord, maxCoord);
		int range = Math.round((float)definition.getRange() / 2);
		
		if(leftwards != null){
			maxPos = getCoordinates().func_177967_a(inwards, range).func_177967_a(leftwards, range).func_177967_a(EnumFacing.UP, range);
			minPos = getCoordinates().func_177972_a(inwards).func_177967_a(leftwards.func_176734_d(), range).func_177967_a(EnumFacing.DOWN, range);
			
		}else{
			maxPos = getCoordinates().func_177967_a(inwards, range).func_177967_a(EnumFacing.EAST, range).func_177967_a(EnumFacing.NORTH, range);
			minPos = getCoordinates().func_177972_a(inwards).func_177967_a(EnumFacing.WEST, range).func_177967_a(EnumFacing.SOUTH, range);
		}
	}
	
	public BlockPos getMinPos()	{
		return minPos;
	}
	
	public BlockPos getMaxPos()	{
		return maxPos;
	}
	
	/* Network */
	@Override
	protected void encodeDescriptionPacket(NBTTagCompound packetData) {
		super.encodeDescriptionPacket(packetData);
		packetData.func_74757_a("Active", active);
	}

	@Override
	protected void decodeDescriptionPacket(NBTTagCompound packetData) {
		super.decodeDescriptionPacket(packetData);
		setActive(packetData.func_74767_n("Active"));
	}

	/* IActivatable */
	@Override
	public boolean isActive() {
		return active;
	}
	
	public void recalculateDirections(BlockPos minCoord, BlockPos maxCoord) {
		inwards = null;
		leftwards = null;

		int facesMatching = 0;
		if (maxCoord.func_177958_n() == getCoordinates().func_177958_n() || minCoord.func_177958_n() == getCoordinates().func_177958_n()) {
			facesMatching++;
		}
		if (maxCoord.func_177956_o() == getCoordinates().func_177956_o() || minCoord.func_177956_o() == getCoordinates().func_177956_o()) {
			facesMatching++;
		}
		if (maxCoord.func_177952_p() == getCoordinates().func_177952_p() || minCoord.func_177952_p() == getCoordinates().func_177952_p()) {
			facesMatching++;
		}
		if (facesMatching == 1) {
			if (maxCoord.func_177958_n() == getCoordinates().func_177958_n()) {
				inwards = EnumFacing.WEST;
				leftwards = EnumFacing.SOUTH;
			} else if (minCoord.func_177958_n() == getCoordinates().func_177958_n()) {
				inwards = EnumFacing.EAST;
				leftwards = EnumFacing.NORTH;
			} else if (maxCoord.func_177952_p() == getCoordinates().func_177952_p()) {
				inwards = EnumFacing.NORTH;
				leftwards = EnumFacing.WEST;
			} else if (minCoord.func_177952_p() == getCoordinates().func_177952_p()) {
				inwards = EnumFacing.SOUTH;
				leftwards = EnumFacing.EAST;
			} else if (maxCoord.func_177956_o() == getCoordinates().func_177956_o()) {
				inwards = EnumFacing.DOWN;
			} else {
				inwards = EnumFacing.UP;
			}
		}else{
			inwards = EnumFacing.DOWN;
		}
	}
	
	@Override
	public IClimatiserDefinition getDefinition() {
		return definition;
	}
	
	@Override
	public IClimateSource getClimateSource() {
		return source;
	}
	
	public boolean canWork(){
		IMultiblockLogic logic = getMultiblockLogic();
		if(logic == null || !logic.isConnected() || getMultiblockLogic().getController().getEnergyManager() == null){
			return false;
		}
		return getMultiblockLogic().getController().getEnergyManager().consumeEnergyToDoWork(WORK_CYCLES, ENERGY_PER_OPERATION);
	}
	
	public Iterable<BlockPos> getPositionsInRange(){
		if(maxPos == null || minPos == null){
			return null;
		}
		return BlockPos.func_177980_a(maxPos, minPos);
	}

	@Override
	public void setActive(boolean active) {
		if (this.active == active) {
			return;
		}

		this.active = active;

		if (field_145850_b != null) {
			if (field_145850_b.field_72995_K) {
				field_145850_b.func_175704_b(getCoordinates(), getCoordinates());
			} else {
				Proxies.net.sendNetworkPacket(new PacketActiveUpdate(this), field_145850_b);
			}
		}
	}

}
