/*******************************************************************************
 * Copyright (c) 2011-2014 SirSengir.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v3
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 *
 * Various Contributors including, but not limited to:
 * SirSengir (original work), CovertJaguar, Player, Binnie, MysteriousAges
 ******************************************************************************/
package forestry.greenhouse.gui;

import net.minecraft.client.gui.GuiTextField;
import net.minecraft.entity.player.EntityPlayer;
import java.io.IOException;

import javax.annotation.Nullable;

import com.google.common.base.Predicate;
import com.google.common.primitives.Floats;

import forestry.api.climate.IClimateControl;
import forestry.api.climate.IClimateControlProvider;
import forestry.api.core.CamouflageManager;
import forestry.api.multiblock.IGreenhouseController;
import forestry.core.config.Constants;
import forestry.core.gui.GuiForestryTitled;
import forestry.core.gui.ledgers.ClimateLedger;
import forestry.core.gui.ledgers.Ledger;
import forestry.core.gui.widgets.TankWidget;
import forestry.core.gui.widgets.WidgetCamouflageSlot;
import forestry.core.network.packets.PacketUpdateClimateControl;
import forestry.core.proxy.Proxies;
import forestry.core.render.TextureManager;
import forestry.core.utils.Translator;
import forestry.greenhouse.multiblock.DefaultClimateControl;
import forestry.greenhouse.multiblock.IGreenhouseControllerInternal;
import forestry.greenhouse.tiles.TileGreenhouse;

public class GuiGreenhouse extends GuiForestryTitled<ContainerGreenhouse, TileGreenhouse> {

    private static final Predicate<String> numberFilter = new Predicate<String>(){
        @Override
		public boolean apply(@Nullable String text){
        	if(!text.isEmpty() && text.length() > 1){
        		if(!text.contains(".")){
        			return false;
        		}
        	}
        	if(text.length() > 7){
        		return false;
        	}
            Float f = Floats.tryParse(text);
            return text.isEmpty() || f != null && Floats.isFinite(f.floatValue()) && f.floatValue() >= 0.0F;
        }
    };
	
	private GuiTextField humidityField;
	private GuiTextField temperatureField;
	private boolean fieldsEnabeled;
	
	public GuiGreenhouse(EntityPlayer player, TileGreenhouse tile) {
		super(Constants.TEXTURE_PATH_GUI + "/greenhouse.png", new ContainerGreenhouse(player.field_71071_by, tile), tile);

		//Add the water tank
		widgetManager.add(new TankWidget(widgetManager, 152, 16, 0).setOverlayOrigin(176, 0));
		
		//Add the multiblock camouflage slots
		widgetManager.add(new WidgetCamouflageSlot(widgetManager, 8, 16, inventory.getMultiblockLogic().getController(), CamouflageManager.DEFAULT));
		widgetManager.add(new WidgetCamouflageSlot(widgetManager, 8, 37, inventory.getMultiblockLogic().getController(), CamouflageManager.GLASS));
		widgetManager.add(new WidgetCamouflageSlot(widgetManager, 8, 58, inventory.getMultiblockLogic().getController(), CamouflageManager.DOOR));
		
		//Add the tile camouflage slots
		widgetManager.add(new WidgetCamouflageSlot(widgetManager, 35, 37, inventory, tile.getCamouflageType()));
		fieldsEnabeled = true;
		
	}
	
	@Override
	public void func_73866_w_() {
		super.func_73866_w_();
		
		temperatureField = new GuiTextField(0, field_146289_q, field_147003_i + 63, field_147009_r + 30, 50, 10);
		humidityField = new GuiTextField(1, field_146289_q, field_147003_i + 63, field_147009_r + 60, 50, 10);
		
		temperatureField.func_175205_a(numberFilter);
		humidityField.func_175205_a(numberFilter);
		
		IGreenhouseController controller = inventory.getMultiblockLogic().getController();
		if(controller == null || controller.getClimateControl() == null || controller.getClimateControl() == DefaultClimateControl.instance){
			temperatureField.func_146184_c(false);
			temperatureField.func_146189_e(false);
			humidityField.func_146184_c(false);
			humidityField.func_146189_e(false);
			fieldsEnabeled = false;
		}else{
			IClimateControl control = controller.getClimateControl();
			temperatureField.func_146180_a(Float.toString(control.getControlTemperature()));
			humidityField.func_146180_a(Float.toString(control.getControlHumidity()));
		}
	}
	
    @Override
	protected void func_73869_a(char typedChar, int keyCode) throws IOException{
    	 if(fieldsEnabeled){
	        if(!humidityField.func_146201_a(typedChar, keyCode) && !temperatureField.func_146201_a(typedChar, keyCode)){
	        	super.func_73869_a(typedChar, keyCode);
	        }
    	 }else{
    		 super.func_73869_a(typedChar, keyCode);
    	 }
    }

    @Override
	protected void func_73864_a(int mouseX, int mouseY, int mouseButton) throws IOException{
        super.func_73864_a(mouseX, mouseY, mouseButton);
        if(fieldsEnabeled){
	        boolean temperatureWasFocused = temperatureField.func_146206_l();
	        boolean  humidityWasFocused = humidityField.func_146206_l();
	        temperatureField.func_146192_a(mouseX, mouseY, mouseButton);
	        humidityField.func_146192_a(mouseX, mouseY, mouseButton);
	        IClimateControlProvider provider = inventory.getMultiblockLogic().getController();
	        IClimateControl control = provider.getClimateControl();
	        if(temperatureWasFocused && !temperatureField.func_146206_l()){
	        	float temp = parseField(temperatureField);
	        	temperatureField.func_146180_a(Float.toString(temp));
	            control.setControlTemperature(temp);
	        }else if(humidityWasFocused && !humidityField.func_146206_l()){
	        	float hum = parseField(humidityField);
	        	humidityField.func_146180_a(Float.toString(hum));
	        	control.setControlHumidity(parseField(humidityField));
	        }
	        Proxies.net.sendToServer(new PacketUpdateClimateControl(provider));
        }
    }
    
    private float parseField(GuiTextField field){
    	String text = field.func_146179_b();
    	if(text.isEmpty()){
    		return 2.0F;
    	}
        try{
        	float f = Float.parseFloat(text);
            if(f >= 2){
            	f = 2.0F;
            }
            return f;
        }catch (NumberFormatException var5){
        	return 2.0F;
        }
    }
    
    @Override
    protected void func_146976_a(float f, int mouseX, int mouseY) {
    	super.func_146976_a(f, mouseX, mouseY);
    	
    	if(temperatureField.func_146176_q()){
			textLayout.line = 20;
			textLayout.drawCenteredLine(Translator.translateToLocal("for.gui.temperature"), 0, fontColor.get("gui.greenhouse.temperature.text"));
	        temperatureField.func_146194_f();
    	}
    	if(humidityField.func_146176_q()){
			textLayout.line = 50;
			textLayout.drawCenteredLine(Translator.translateToLocal("for.gui.humidity"), 0, fontColor.get("gui.greenhouse.humidity.text"));
	    	
	        humidityField.func_146194_f();
    	}
		bindTexture(textureFile);
    }

	@Override
	protected void addLedgers() {
		IGreenhouseControllerInternal greenhouseController = inventory.getMultiblockLogic().getController();
		
		ledgerManager.add(new ClimateLedger(ledgerManager, greenhouseController));
		ledgerManager.add(new EnergyLedger());
		super.addLedgers();
	}
	
	protected class EnergyLedger extends Ledger {

		public EnergyLedger() {
			super(ledgerManager, "power");
			maxHeight = 48;
		}

		@Override
		public void draw(int x, int y) {

			// Draw background
			drawBackground(x, y);

			// Draw icon
			drawSprite(TextureManager.getInstance().getDefault("misc/energy"), x + 3, y + 4);

			if (!isFullyOpened()) {
				return;
			}

			drawHeader(Translator.translateToLocal("for.gui.energy"), x + 22, y + 8);

			drawSubheader(Translator.translateToLocal("for.gui.stored") + ':', x + 22, y + 20);
			drawText(inventory.getMultiblockLogic().getController().getEnergyManager().getEnergyStored(null) + " RF", x + 22, y + 32);
		}

		@Override
		public String getTooltip() {
			return Translator.translateToLocal("for.gui.energy") + ": " + inventory.getMultiblockLogic().getController().getEnergyManager().getEnergyStored(null) + " RF/t";
		}
	}
}
